package org.lsst.ccs.subsystems.shutter.common;

import java.io.Serializable;

/**
 ***************************************************************************
 **
 **  Immutable class containing the data for a single blade position sample.
 ** 
 **  <p>Blade set positions come in absolute and relative flavors. Absolute
 **  positions are in mm along an axis common to both blade sets with a single origin. Relative
 **  positions are dimensionless and use a separate axis for each blade set;
 **  0.0 means fully retracted (open) and 1.0 means fully extended (closed).
 **  Each instance of this class contains both types of position.
 **
 **  @author Owen Saxton
 **  @author tether
 **
 ***************************************************************************
 */
public final class BladePosition implements Serializable {

    private static final long serialVersionUID = 1L;

    final long     time;          // Epoch microseconds
    final double   position;      // Absolute blade position
    final double   relPosition;   // Relative blade position

    /**
     * Constructs given all three values.
     * @param time        a system time stamp (in microseconds)
     * @param position    the absolute position
     * @param relPosition the relative position
     */
    public BladePosition(long time, double position, double relPosition) {
        this.time = time;
        this.position = position;
        this.relPosition = relPosition;
    }


    /**
     * Constructs given the position values. The time is taken as System.currentTimeMillis() * 1000.
     * @param position    the absolute position
     * @param relPosition the relative position
     */
    public BladePosition(double position, double relPosition) {
        this.time = System.currentTimeMillis() * 1000;
        this.position = position;
        this.relPosition = relPosition;
    }
    
    /**
     * Gets the time at which the position sample was taken..
     * @return the time value, a system time stamp in microseconds
     */
    public long getTime()
    {
        return time;
    }

    /**
     * Gets the sampled absolute position.
     * @return the absolute position
     */
    public double getPosition()
    {
        return position;
    }

    /**
     * Gets the relative position calculated from the absolute position.
     * @return the relative position
     */
    public double getRelPosition()
    {
        return relPosition;
    }

}
