package org.lsst.ccs.subsystems.shutter.status;

import org.lsst.ccs.subsystems.shutter.common.MovementHistory;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;

/**
 * Used to publish motor position samples and Hall data after a blade set motion. Immutable.
 * 
 * <p>Carrying out a command might require several blade set movements so there are two boolean flags
 * defined to mark the first and last movements required for the command. If a command requires
 * just a single movement then both flags will be set.</p>
 * @author azemoon
 * @author tether
 */
public class MovementHistoryStatus implements java.io.Serializable {

    private final MovementHistory history;
    private final boolean firstMovement, lastMovement;

    /**
     * Constructs an instance from flags and a MovementHistory.
     * @param firstMovement true if this is the first movement of a series
     * @param lastMovement true if this is the last movement of a series
     * @param history the movement history
     */
    public MovementHistoryStatus(boolean firstMovement, boolean lastMovement, MovementHistory history) {
        this.firstMovement = firstMovement;
        this.lastMovement = lastMovement;
        this.history = history;
    }

    /**
     * Gets the blade set that was moved.
     * @return the side value
     */
    public ShutterSide getSide() {
        return history.getSide();
    }

    /**
     * Get the embedded MovementHistory object.
     * @return the movement history
     */
    public MovementHistory getHistory() {
        return history;
    }

    /**
     * Is this the first movement performed for whatever command was issued?
     * @return the first-movement flag
     */
    public boolean isFirstMovement() {
        return firstMovement;
    }
    
    /**
     * Is this the last movement performed for whatever command was issued?
     * @return the last-movement flag
     */
    public boolean isLastMovement() {
        return lastMovement;
    }
}
