package org.lsst.ccs.subsystems.shutter.sim;

import java.util.stream.Stream;
import java.util.stream.DoubleStream;

/**
 * Samples a motion profile to produce a stream of MotorPosition objects.
 * 
 * <p>The physical motion profile is sampled at regular intervals, each sampling being used
 * to make a MotorPosition object. The starting time is added to the time-since-move-start
 * used by the profile. The sampling is done once but can be retrieved as many times as needed.
 * 
 * <p>This class is used both for simulating hardware and for predicting the motor movement
 * in the GUI.
 * 
 * <p> Instances are immutable.
 * 
 * @author tether
 *
 */

public final class MotorSimulator {
    
    /**
     * Samples the given physical motion profile.
     * @param startPosition the absolute start position (dimensionless in [0, 1])
     * @param dt the time between samples in seconds
     * @param numSamples the number of positions to take
     * @param startTime the starting time in microseconds
     * @param profile the motion profile to be sampled
     */
    public MotorSimulator(
            final double startPosition,
            final double dt,
            final int numSamples,
            final long startTime,
            final MotionProfile profile)
    {
        positions =
                DoubleStream
                .iterate(0.0, x -> x + dt)
                .limit(numSamples)

                 // We round the time to the nearest microsecond before creating
                 // the MotorPosition in order to make writing the unit test
                 // easier. Otherwise we'd have to compare motion at unrounded
                 // times with motion at the rounded times saved in the MotorPosition.
                .map(t -> Math.rint(1e6 * t) / 1e6)

                .mapToObj(t ->  new MotorPosition(
                        startPosition + profile.distance(t),
                        profile.velocity(t),
                        profile.acceleration(t),
                        startTime + Math.round(1e6 * t)))
                .toArray(MotorPosition[]::new);
        }
    
    /**
     * Returns the motor positions as a stream.
     * @return the position stream
     */
    public Stream<MotorPosition> getPositions() {return Stream.of(positions);}

    /** The samples taken from the motion profile. */
    final private MotorPosition[] positions;
}
