
package org.lsst.ccs.subsystems.shutter.sim;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.BladeSet;
import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallTransition;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;
import org.lsst.ccs.subsystems.shutter.common.ShutterConfiguration;
import org.lsst.ccs.subsystems.shutter.common.ShutterSide;

/**
 * Describes one simulated blade set.
 * 
 * <p>At present the simulation doesn't yet use any configuration data and so calculates
 * only relative blade set positions for both the motor samples and the Hall transitions. The motor
 * simulation calculates position samples at intervals of roughly 50 milliseconds.
 * 
 * <p>Thread publication policy: shared, thread-safe. Instances can be accessed safely through
 * the public interface by multiple threads.
 * 
 * @author azemoon
 * @author tether
 */
public final class BladeSetSimulator implements BladeSet {

    private final ShutterSide side;
    
    private final BladeSetConfiguration config; // Stored but not used.
    
    private final ShutterConfiguration shutterConfig; // Stored but not used.
    
    private final List<HallConfiguration> hallConfigs; // Stored but not used.
    
    // This is the only mutable state in the object. Declaring it volatile makes access both atomic
    // and thread-safe. Note that access to non-volatile doubles is not atomic.
    private volatile double position;

    /** Sets the index and initial relative position of the blade set. The initial
     * position of the -X set is 0.0, that of +X set is 1.0.
     * @param side the side of the shutter this blade set simulates
     * @param config the blase-set configuration
     * @param shutterConfig the general, side-independent shutter configuration
     * @param hallConfigs the Hall sensor configurations
     */
    public BladeSetSimulator(
        ShutterSide side,
        BladeSetConfiguration config,
        ShutterConfiguration shutterConfig,
        List<HallConfiguration> hallConfigs) {
        this.side = side;
        this.position = 0.0;
        this.config = config;
        this.shutterConfig = shutterConfig;
        this.hallConfigs = hallConfigs;
    }

    /** Gets the (as yet unused) blade set configuration..
     * @throws UnsupportedOperationException always
     */
    @Override
    public BladeSetConfiguration getBladeSetConfiguration() {
        return config;
    }

    /** Gets the current relative position of the edge of the blade set.
     * @return The position.
     */
    @Override
    public double getRelativePosition() {
        return position;
    }

    /**
     * Gets the side of the shutter that this object simulates.
     * @return The side..
     */
    @Override
    public ShutterSide getSide() {
        return side;
    }

    /**
     * Performs a simulated motion of the blade set.
     * @param targetPosition The desired final relative position of the blade set's edge.
     * @param moveTimeSeconds How long the move should take.
     * @return The movement history, which includes time-ordered lists of motor position
     * samples and Hall transitions.
     * <p>After returning the current position will be seen to be targetPosition in any thread.
     */
    @Override
    public MovementHistory moveToPosition(double targetPosition, double moveTimeSeconds) {
        // Approximately 50 millsec between motor position samples.
        final int numSamples = (int)(1 + Math.round(moveTimeSeconds / 0.050));
        
        final double dt = moveTimeSeconds / (numSamples - 1);
        
        final double startPosition = getRelativePosition();
        
        final long startTime = System.currentTimeMillis() * 1000;
        
        final MotionProfile profile =
                new CubicSCurve(targetPosition - getRelativePosition(), moveTimeSeconds);
        
        final MotorSimulator motor =
                new MotorSimulator(startPosition, dt, numSamples, startTime, profile);
        
        final MotorEncoderSimulator encoder = new MotorEncoderSimulator();
        
        final HallSensorSimulator hall =
                new HallSensorSimulator(startTime, startPosition, targetPosition, profile);
        
        final List<BladePosition> bladePos =
                motor.getPositions().map(encoder).collect(Collectors.toList());
        
        final List<HallTransition> hallTrans =
                hall.getTransitions().collect(Collectors.toList());
        
        final long endTime = bladePos.get(bladePos.size() - 1).getTime();
        
        this.position = targetPosition;

        return new MovementHistory(
            getSide(),
            1, // Status = success
            startTime,
            startPosition,
            endTime,
            targetPosition,
            Collections.unmodifiableList(bladePos),
            Collections.unmodifiableList(hallTrans)
        );
 }

    /** @throws UnsupportedOperationException */
    @Override
    public double encoderToAbsolute(double encoderValue) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /** @throws UnsupportedOperationException */
    @Override
    public double absoluteToEncoder(double absolute) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /** @throws UnsupportedOperationException */
    @Override
    public double absoluteToRelative(double abs) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /** @throws UnsupportedOperationException */
    @Override
    public double relativeToAbsolute(double rel) {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /** {@inheritDoc} */
    @Override
    public boolean isFullyRetracted() {
        return getRelativePosition() < 1e-4;
    }

    /** {@inheritDoc} */
    @Override
    public boolean isFullyExtended() {
        return Math.abs(1.0 - getRelativePosition()) < 1e-4;
    }

}

