package org.lsst.ccs.subsystems.shutter.common;

import java.io.Serializable;

/**
 ***************************************************************************
 **
 **  Immutable class containing the data for a single Hall sensor transition.
 **
 **  @author Owen Saxton
 **  @author tether
 **  @see BladePosition
 **
 ***************************************************************************
 */
public final class HallTransition implements Serializable {

    private static final long serialVersionUID = 1L;

    final long     time;         // Microseconds since epoch
    final int      sensor;       // ID of sensor
    final boolean  open;         // True if sensor opened; false if closed
    final boolean  reverse;      // True if and only if retracting the blade set
    final double   position;     // Absolute position of this transition
    final double   relPosition;  // Relative position of this transition

    /**
     * Full constructor.
     * @param time          The time of the transition, a system time in microseconds.
     * @param sensor        The ID number of the sensor.
     * @param open          Is the new state of the sensor "open"?
     * @param reverse       Was the blade set retracting?
     * @param position      The absolute position of the transition.
     * @param relPosition   The relative position of the transition.
     */
    public HallTransition(long time, int sensor, boolean open,
                              boolean reverse, double position,
                              double relPosition)
    {
        this.time        = time;
        this.sensor      = sensor;
        this.open        = open;
        this.reverse     = reverse;
        this.position    = position;
        this.relPosition = relPosition;
    }

    /**
     * Gets the time of the transition.
     * @return the transition time, a system time in microseconds
     */
    public long getTransitionTime()
    {
        return time;
    }

    /**
     * Gets the Hall sensor ID number.
     * @return the ID number
     */
    public int getSensorId()
    {
        return sensor;
    }


    /**
     * Is the new state of the sensor "open"?
     * @return the new-state flag
     */
    public boolean isOpen()
    {
        return open;
    }


    /**
     * Was the blade set retracting when the transition occurred?
     * @return the retracting flag
     */
    public boolean isReverse()
    {
        return reverse;
    }

    /**
     * Gets the absolute position of the transition.
     * @return the absolute position in mm
     */
    public double getPosition()
    {
        return position;
    }

    /**
     * Gets the relative position of the transition.
     * @return the relative position
     */
    public double getRelPosition()
    {
        return relPosition;
    }

}
