package org.lsst.ccs.subsystems.shutter.common;

import java.io.Serializable;
import java.util.List;
import java.util.Collections;

/**
 ***************************************************************************
 **
 **  Immutable class containing the data produced by a blade set move.
 **
 **  @author Owen Saxton
 *   @author tether
 **
 ***************************************************************************
 */
public final class MovementHistory implements Serializable {
    
    private static final long serialVersionUID = 1L;

    private final ShutterSide          side;           // Blade set index
    private final int                  status;         // Move completion status
    private final long                 startTime;      // Move start time
    private final double               startPosition;  // Move start position
    private final long                 endTime;        // Move end time
    private final double               endPosition;    // Move end position
    private final List<BladePosition>  positions;      // List of encoder data
    private final List<HallTransition> transitions;    // List of Hall sensor data

    /**
     * Constructs an instance from the individual field values.
     * @param side the side of the camera X axis to which the blade set belongs.
     * @param status the status code from the move.
     * See {@link org.lsst.ccs.subsystems.shutter.driver.MovementStatus}
     * @param startTime the start time of the move, a system time stamp in microseconds
     * @param startPosition the initial relative position
     * @param endTime the end time of the move in microseconds
     * @param endPosition the final relative position
     * @param positions the blade position info sampled during the move
     * @param transitions the Hall transitions that occurred during the move
     */
    public MovementHistory(ShutterSide side, int status,
                            long startTime, double startPosition,
                            long endTime, double endPosition,
                            List<BladePosition> positions,
                            List<HallTransition> transitions)
    {
        this.side = side;
        this.status = status;
        this.startTime = startTime;
        this.startPosition = startPosition;
        this.endTime = endTime;
        this.endPosition = endPosition;
        this.positions = Collections.unmodifiableList(positions);
        this.transitions = Collections.unmodifiableList(transitions);
    }
    
    /**
     *  Gets the side of the camera X axis to which the blade set belongs.
     *  @return the side
     */
    public ShutterSide getSide()
    {
        return side;
    }


    /**
     *  Gets the movement completion status.
     *  @return the status code
     */
    public int getStatus()
    {
        return status;
    }


    /**
     *  Gets the start time of the movement in Epoch microseconds.
     *  @return the start time
     */
    public long getStartTime()
    {
        return startTime;
    }


    /**
     *  Gets the end time of the movement in Epoch microseconds.
     *  @return the end time
     */
    public long getEndTime()
    {
        return endTime;
    }


    /**
     *  Gets the position at the beginning of the movement.
     *  @return the start position, a number in [0,1]
     *
     */
    public double getStartPosition()
    {
        return startPosition;
    }


    /**
     *  Gets the position at the end of the movement.
     *  @return the end position, a number in [0,1]
     *
     */
    public double getEndPosition()
    {
        return endPosition;
    }


    /**
     *  Gets a time-ordered list of BladePositions generated by the move.
     *  @return the unmodifiable list of positions
     *
     */
    public List<BladePosition> getPositions()
    {
        return positions;
    }



    /**
     *  Gets the time-ordered list of Hall Sensor transitions generated by this movement.
     *  @return the unmodifiable list of transitions
     *
     */
    public List<HallTransition> getHallTransitions()
    {
        return transitions;
    }

}
