package org.lsst.ccs.subsystems.shutter.sim;

import java.util.function.DoubleUnaryOperator;

/**
 * Defines a piecewise cubic S-curve motion profile. Acceleration is linearly
 * ramping up or down at all times so that there are no intervals of constant velocity.
 * 
 * <p>In motion controller parlance an S-curve motion profile is one that
 * ramps acceleration using a constant "jerk" (da/dt). The alternative
 * is a so-called "trapezoidal" profile in which acceleration changes
 * discontinuously (infinite jerk). S-curve motion is smoother, with
 * no sharp bends in the velocity curve.
 * 
 * <p>The S-curve is generated by scaling the following basic profile starting
 * at distance = 0 dUnit, velocity = 0 dUnit/tUnit and acceleration = 0 dUnit/tUnit*2:
 * <ul>
 * <li> Phase 0, t in [0,   1/4] tUnit: Jerk = +4 dUnit/tUnit^3, acceleration ramps from  0 to  1 dUnit/tUnit^2.
 * <li> Phase 1, t in [1/4, 3/4] tUnit: Jerk = -4 sUnit/tUnit^3, acceleration ramps from  1 to -1 sUnit/tUnit^2.
 * <li> Phase 2, t in [3/4,   1] tUnit: Jerk = +4 sUnit/tUnit^3, acceleration ramps from -1 to  0 sUnit/tUnit^2.
 * </ul>
 * 
 * <p>The basic profile d(t), v(t), a(t) represents of motion of +1/8 dUnit in 1 tUnit. A motion
 * of D distance units in T time units is represented by 8D*d(t/T), (8D/T)*v(t/T), (8D/T**2)*a(t/T).
 * D is positive for an extension and negative for a retraction.
 * 
 * <p>Instances of this class are immutable.
 * 
 * <p>Distance traveled, velocity and acceleration all refer to the leading edge of the blade set though
 * for brevity we just say "the blade set".
 * 
 * <p>This motion profile was first used with the one-bladed shutter prototype.
 * 
 * @author tether
 */
public class CubicSCurve implements MotionProfile {
    /**
     * Sets the scale factors for distance moved and total elapsed time.
     * @param totalDistance The total distance D to move. Negative for a retraction.
     * @param totalTime The elapsed time T allowed for the move.
     * @throws IllegalArgumentException if totalTime is zero or negative.
     */
    public CubicSCurve(double totalDistance, double totalTime) {
        if (totalTime <= 0.0) throw new IllegalArgumentException("totalTime can't be <= 0");
        this.totalDistance = totalDistance;
        this.totalTime = totalTime;
        distanceScale = 8.0 * totalDistance;
        timeScale = 1.0 / totalTime;
        velocityScale = distanceScale * timeScale;
        accelerationScale = velocityScale * timeScale;
    }
    
    // distance(), velocity() acceleration() and inverseDistance()
    // implement MotionProfile.
    
    @Override
    public double distance(double t) {
        final double ts = timeScale * t;
        return distanceScale * dis[tphase(ts)].applyAsDouble(ts);
    }
    
    @Override
    public double velocity(double t) {
        final double ts = timeScale * t;
        return velocityScale * vel[tphase(ts)].applyAsDouble(ts);
    }
    
    @Override
    public double acceleration(double t) {
        final double ts = timeScale * t;
        return accelerationScale * acc[tphase(ts)].applyAsDouble(ts);
    }

    @Override
    public double inverseDistance(final double dtarget) {
        final double d = dtarget / distanceScale;
        final int phase = dphase(d);
        return invdis[phase].applyAsDouble(d) / timeScale;
    }
    
    /** The total distance moved. */
    private final double totalDistance;
    
    /** The total time taken. */
    private final double totalTime;
    
    /** Multiply the basic distance values by this. */
    private final double distanceScale;
    
    /** Multiply time t by this before calling the basic profile functions. */
    private final double timeScale;
    
    /** Multiply the basic velocity values by this. */
    private final double velocityScale;
    
    /** Multiply the basic acceleration values by this. */
    private final double accelerationScale;
    
    /** The phase-specific position functions. */
    private final static DoubleUnaryOperator dis[];
    
    /** The phase-specific velocity functions. */
    private final static DoubleUnaryOperator vel[];
    
    /** The phase-specific acceleration functions. */
    private final static DoubleUnaryOperator acc[];
    
    /** The phase-specific inverse distance functions. */
    private final static DoubleUnaryOperator invdis[];
    
    /** The time at which motion phase 0 ends, basic profile. */
    private final static double PHASE0_TEND = 0.25;
    
    /** The time at which motion phase 1 ends, basic profile. */
    private final static double PHASE1_TEND = 0.75;
    
    /** The time at which motion phase 2 ends, basic profile. */
    private final static double PHASE2_TEND = 1.0;
    
    /**
     * The travel distance during phase 0, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The distance.
     */
    private static double dis0(double t) {
        return 2.0 * Math.pow(t, 3) / 3.0;
    }
    
    /**
     * The travel distance during phase 1, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The distance.
     */
    private static double dis1(double t) {
        final double t0 = PHASE0_TEND;
        final double dt = t - t0;
        return dis0(t0) + vel0(t0) * dt + 0.5 * acc0(t0) * Math.pow(dt, 2) - dis0(dt);
    }
    
    /**
     * The travel distance during phase 2, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The distance.
     */
    private static double dis2(double t) {
        final double t0 = PHASE1_TEND;
        final double dt = t - t0;
        return dis1(t0) + vel1(t0) * dt + 0.5 * acc1(t0) * Math.pow(dt, 2) + dis0(dt);
    }
    
    /**
     * The velocity during phase 0, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The velocity.
     */
    private static double vel0(double t) {
        return 2.0 * Math.pow(t, 2);
    }
    
    /**
     * The velocity during phase 1, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The velocity.
     */
    private static double vel1(double t) {
        final double t0 = PHASE0_TEND;
        final double dt = t - t0;
        return vel0(t0) + acc0(t0) * dt - vel0(dt);
    }
    
    /**
     * The velocity during phase 2, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The velocity.
     */
    private static double vel2(double t) {
        final double t0 = PHASE1_TEND;
        final double dt = t - t0;
        return vel1(t0) + acc1(t0) * dt + vel0(dt);
    }
    
    /**
     * The acceleration during phase 0, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The acceleration.
     */
    private static double acc0(double t) {
        return 4.0 * t;
    }
    
    /**
     * The acceleration during phase 1, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The acceleration.
     */
    private static double acc1(double t) {
        final double t0 = PHASE0_TEND;
        final double dt = t - t0;
        return acc0(t0) - acc0(dt);
    }
    
    /**
     * The acceleration during phase 2, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The acceleration.
     */
    private static double acc2(double t) {
        final double t0 = PHASE1_TEND;
        final double dt = t - t0;
        return acc1(t0) + acc0(dt);
    }
    
    /** 
     * The inverse distance for phase 0, basic profile.
     * @param d The distance.
     * @return The time at which distance d was reached.
     */
    private static double invdis0(double d) {
        return Math.cbrt(3.0 * d / 2.0);
    }
    
    /** 
     * The inverse distance for phase 2, basic profile.
     * @param d The distance.
     * @return The time at which distance d was reached.
     */
    private static double invdis2(double d) {
        // Take advantage of the symmetry of the velocity profile,
        // v(t) = v(1 - t)
        return PHASE2_TEND - invdis0(PHASE2_DEND - d);
    }
    
    /** 
     * The inverse distance for phase 1, basic profile.
     * @param d The distance.
     * @return The time at which distance d was reached.
     */
    private static double invdis1(double d) {
        // This one will need a more general cubic solver.
        // Use the method of "Numerical Recipes in Fortran 77" to solve
        // x^3 + a x^2 + b x + c = 0
        // where x = t - 1/4.
        final double a = -3.0 / 4.0;
        final double b = -3.0 / 16.0;
        final double c = (3.0/2.0) * d - 1.0 / 64.0;
        final double Q = (a*a - 3.0*b) / 9.0;
        final double R = (2.0*a*a*a - 9.0*a*b + 27.0*c) / 54.0;
        // As it turns out, R**2 is always less than Q**3 for this problem
        // so we use the trig method to find the roots. The third root
        // is the correct one.
        final double theta = Math.acos(R/Math.sqrt(Q*Q*Q));
        final double x3 = -2.0 * Math.sqrt(Q) * Math.cos((theta-2.0*Math.PI)/3.0) - a / 3.0;
        //System.out.printf("%+10.6f  %+10.6f %n", d, x3);
        return PHASE0_TEND + x3;
    }
    
    static {
        dis = new DoubleUnaryOperator[]{CubicSCurve::dis0, CubicSCurve::dis1, CubicSCurve::dis2};
        invdis = new DoubleUnaryOperator[]{CubicSCurve::invdis0, CubicSCurve::invdis1, CubicSCurve::invdis2};
        vel = new DoubleUnaryOperator[]{CubicSCurve::vel0, CubicSCurve::vel1, CubicSCurve::vel2};
        acc = new DoubleUnaryOperator[]{CubicSCurve::acc0, CubicSCurve::acc1, CubicSCurve::acc2};
    }
    
    /** The tolerance used when checking the validity of the arguments to tphase() and dphase(). */
    private final static double RANGE_TOL = 1e-8;
    
    /**
     * The phase of the motion, based on time, basic profile.
     * @param t A time in [0.0, 1.0].
     * @return The phase number.
     * @throws IllegalArgumentException if t is invalid.
     */
    static private int tphase(double t) {
        if (t < (0.0-RANGE_TOL) || t > (1.0+RANGE_TOL))
            throw new IllegalArgumentException("Scaled time t not in [0, 1]");
        return t <= PHASE0_TEND ? 0 : (t < PHASE1_TEND ? 1 : 2);
    }
    
    /** The total distance traveled by the end of phase 0, basic profile. */
    private final static double PHASE0_DEND = 1.0 /96.0;
    
    /** The total distance traveled by the end of phase 1, basic profile. */
    private final static double PHASE1_DEND = 11.0 / 96.0;
    
    /** The total distance traveled by the end of phase 2, basic profile. */
    private final static double PHASE2_DEND = 1.0 / 8.0;
    
    /**
     * The phase of the motion, based on distance traveled, basic profile.
     * @param d A distance in [0.0, 0.125].
     * @return The phase number.
     * @throws IllegalArgumentException if d is invalid.
     */
    static private int dphase(double d) {
        if (d < (0.0-RANGE_TOL) || d > (PHASE2_DEND+RANGE_TOL))
            throw new IllegalArgumentException("Scaled distance not in [0, 1/8]");
        return d <= PHASE0_DEND ? 0 : (d < PHASE1_DEND ? 1 : 2);
    }
}
