
package org.lsst.ccs.subsystems.shutter.sim;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.lsst.ccs.subsystems.shutter.common.BladePosition;
import org.lsst.ccs.subsystems.shutter.common.BladeSet;
import org.lsst.ccs.subsystems.shutter.common.BladeSetConfiguration;
import org.lsst.ccs.subsystems.shutter.common.HallTransition;
import org.lsst.ccs.subsystems.shutter.common.MovementHistory;

/**
 * Describes one simulated blade set.
 * 
 * <p>At present the simulation doesn't use a blade set configuration and so calculates
 * only relative blade set positions for both the motor samples and the Hall transitions. The motor
 * simulation calculates position samples at intervals of roughly 50 milliseconds.
 * 
 * <p>The index tells which side of the shutter the blade set is on and is also used to determine
 * the initial relative position of the blade set's edge: 0.0 for index 0 and 1.0 for index 1.
 * See {@link org.lsst.ccs.subsystems.shutter.common.BladePosition}.
 * 
 * <p>Thread publication policy: shared, thread-safe. Instances can be accessed safely through
 * the public interface by multiple threads.
 * 
 * @author azemoon
 * @author tether
 */
public final class BladeSetSimulator implements BladeSet {

    private final int index;
    
    // This is the only mutable state in the object. Declaring it volatile makes access both atomic
    // and thread-safe. Note that access to non-volatile doubles is not atomic.
    private volatile double position;

    /** Sets the index and initial relative position of the blade set. The initial
     * position of set 0 is 0.0, that of set 1 is 1.0.
     * @param index the blade set index
     */
    public BladeSetSimulator(int index) {
        this.index = (index == 0) ? 0 : 1;
        this.position = (this.index == 0) ? 0.0 : 1.0;}

    /** Not yet supported.
     * @throws UnsupportedOperationException always
     */
    @Override
    public BladeSetConfiguration getBladeSetConfiguration() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    /** Gets the current relative position of the edge of the blade set.
     * @return The position.
     */
    @Override
    public double getCurrentPosition() {
        return position;
    }

    /**
     * Gets the index.
     * @return The index (0 or 1).
     */
    @Override
    public int getIndex() {
        return index;
    }

    /**
     * Performs a simulated motion of the blade set.
     * @param targetPosition The desired final relative position of the blade set's edge.
     * @param moveTimeSeconds How long the move should take.
     * @return The movement history, which includes time-ordered lists of motor position
     * samples and Hall transitions.
     * <p>After returning the current position will be seen to be targetPosition in any thread.
     */
    @Override
    public MovementHistory moveToPosition(double targetPosition, double moveTimeSeconds) {
        // Approximately 50 millsec between motor position samples.
        final int numSamples = (int)(1 + Math.round(moveTimeSeconds / 0.050));
        
        final double dt = moveTimeSeconds / (numSamples - 1);
        
        final double startPosition = getCurrentPosition();
        
        final long startTime = System.currentTimeMillis() * 1000;
        
        final MotionProfile profile =
                new CubicSCurve(targetPosition - getCurrentPosition(), moveTimeSeconds);
        
        final MotorSimulator motor =
                new MotorSimulator(startPosition, dt, numSamples, startTime, profile);
        
        final MotorEncoderSimulator encoder = new MotorEncoderSimulator();
        
        final HallSensorSimulator hall =
                new HallSensorSimulator(startTime, startPosition, targetPosition, profile);
        
        final List<BladePosition> bladePos =
                motor.getPositions().map(encoder).collect(Collectors.toList());
        
        final List<HallTransition> hallTrans =
                hall.getTransitions().collect(Collectors.toList());
        
        final long endTime = bladePos.get(bladePos.size() - 1).getTime();
        
        this.position = targetPosition;

        return new MovementHistory(
            getIndex(),
            1, // Status = success
            startTime,
            startPosition,
            endTime,
            targetPosition,
            Collections.unmodifiableList(bladePos),
            Collections.unmodifiableList(hallTrans)
        );
 }
}

