package org.lsst.ccs.subsystems.shutter.common;

import java.io.PrintStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.Serializable;
import java.util.Scanner;
import java.util.ArrayList;
import java.util.NoSuchElementException;

/**
 ***************************************************************************
 **
 **  Contains low-level hardware information and a set of calibration data
 **  for one blade set. Immutable.
 **  
 **  <p>At the moment each instance contains hard-coded information about
 **  only one set of hardware, that used in the one-blade prototype shutter
 **  constructed in 2009:</p>
 **  <ul>
 **  <li>Parker Aries single-axis motor controller/driver (ACR-type motor controller)</li>
 **  <li>Diamond Systems Helios PC/104 with Acces I/O 104-DIO-024S ISA I/O card</li>
 **  <li>Fixed IP addresses on a local network:</li>
 **  <ul>
 **  <li>192.168.100.1 for the Aries</li>
 **  <li>192.168.100.23 for the Helios</li>
 **  <li>192.168.100.21 for a laptop or desktop computer</li>
 **  </ul>
 **  </ul>
 **  <p>Although the prototype has only one physical blade, two blade sets are simulated. Set 0
 **  is assigned a home/open position at low absolute position. Moving this blade set forward
 **  decreases the motor position encoder reading, so pulses/mm is negative. The forward direction
 **  for the "other" blade set, set 1, increases encoder values so pulses/mm is positive
 **  but has the same magnitude. However, the position conversion routines assume that encoder values
 **  are near zero for each blade set near its home position, which is impossible for a single
 **  axis unless the encoder zero point is reset when changing blade sets. It shouldn't matter
 **  if the motion commands use differences in position. In fact the ACR's "JOG INC" is used
 **  for all motions; as it takes only position changes as argument things appear to be OK.</p>
 **  @see org.lsst.ccs.drivers.iocard
 **  @see org.lsst.ccs.drivers.parker
 **  @see org.lsst.ccs.subsystems.shutter.BladeSetDrvr
 **  @author Owen Saxton
 **  @author tether
 **
 ***************************************************************************
 */
public final class BladeSetConfiguration implements Serializable {

    private final static double POS_SOFT_LIM = 1.05,
                                NEG_SOFT_LIM = -0.05;

    private final int     index;      // Index of the BladeSet
    private final String  node;       // Node name of motor controller
    private final double  ppMm;       // Motor encoder pulses per mm of travel
    private final double  homePosn;   // Blade home position (mm)
    private final double  openPosn;   // Blade open position (mm)
    private final double  clsdPosn;   // Blade closed position (mm)
    private final int     numSamp;    // Number of encoder samples to take
    private final double  moveTime;   // Duration of a BladeSet move (sec)
    private final int     dioAddr;    // DIO board I/O space address
    private final int     dioLevel;   // DIO board IRQ level
    private final int     dioConf;    // DIO board ports configuration value
    private final int     dioHPort;   // DIO board port for Hall sensor input
    private final int     dioOPort;   // DIO board port for output to motor
    private final int     dioOLine;   // DIO board line for output to motor
    private final int     dioIPort;   // DIO board port for input from motor
    private final int     dioILine;   // DIO board line for input from motor
    private final int     adAddr;     // AD board I/O space address
    private final int     adLevel;    // AD board IRQ level
    private final int     adTempChn;  // AD board channel for ambient temperature

    private final BladeSetCalibration calib;
                        // Hall switch calibration data

    private static final String[] CALIB_FILE = {"Calibration_0.dat",
                                                "Calibration_1.dat"};

    private static final PrintStream out = System.out;

    /**
     * Constructs a configuration for a given blade set.
     * All the parameters are hard coded except for the Hall sensor calibration
     * data. That's loaded from a text file using {@link readCalibration(int)}.
     * 
     * @param index the index, 0 or 1, designating the blade set.
     * 
     */
    public BladeSetConfiguration(int index)
    {
        this.index = index;
        if (index == 0) {
            node       = "192.168.100.1";
            ppMm       = -49.3504;
            homePosn   = 0.0;
            openPosn   = 0.0;
            clsdPosn   = 760.0;
            moveTime   = 1.0;
            numSamp    = 100;
            dioAddr    = 0x340;
            dioLevel   = 6;
            dioConf    = 0x07;
            dioHPort   = 0;
            dioIPort   = 2;
            dioILine   = 0;
            dioOPort   = 2;
            dioOLine   = 4;
            adAddr     = 0x280;
            adLevel    = 5;
            adTempChn  = 0;
        }
        else {
            node       = "192.168.100.1";
            ppMm       = 49.3504;
            homePosn   = 760.0;
            openPosn   = 760.0;
            clsdPosn   = 0.0;
            moveTime   = 1.0;
            numSamp    = 100;
            dioAddr    = 0x340;
            dioLevel   = 6;
            dioConf    = 0x07;
            dioHPort   = 0;
            dioIPort   = 2;
            dioILine   = 0;
            dioOPort   = 2;
            dioOLine   = 4;
            adAddr     = 0x280;
            adLevel    = 5;
            adTempChn  = 0;
        }
        calib = readCalibration(index);
    }

    /**
     * Gets the index, 0 or 1, of the blade set described by this object.
     * @return the index
     */
    public int getIndex()
    {
        return index;
    }

    /**
     * Gets the current Hall sensor calibration for the blade set.
     * @return the calibration
     */
    public BladeSetCalibration getCalibration()
    {
        return calib;
    }

    /**
     * Gets the IP address of the motor controller for the blade set.
     * @return the numerical IP address in dotted string form
     */
    public String getNode()
    {
        return node;
    }

    /**
     * Gets the pulses-per-millimeter for the blade set's motor position encoder. Although the prototype has
     * only a single blade, the second blade is simulated by reversing the sense of motion by reversing
     * the sign on this value.
     * @return the scale factor
     */
    public double getPpMm()
    {
        return ppMm;
    }

    /**
     * Gets the absolute home position of the blade set in mm. The home position is a special
     * set point that may correspond to a physical mark on the motor encoder scale or it may
     * just be a convenient reference point. It's possible that a homed blade set is not
     * ready to be moved.
     * @return the home position
     */
    public double getHome()
    {
        return homePosn;
    }

    /**
     * Gets the absolute position in mm of the blade set when it's considered fully retracted and
     * ready for motion. May be the same as the home position depending on the hardware.
     * @return the open position
     */
    public double getOpen()
    {
        return openPosn;
    }

    /**
     * Gets the absolute position in mm of the blade set when it's considered fully extended.
     * @return the closed position
     */
    public double getClosed()
    {
        return clsdPosn;
    }

    /**
     * Gets the default duration in seconds of each blade set motion.
     * @return the duration
     */
    public double getMoveTime()
    {
        return moveTime;
    }

    /**
     * Gets the number of motor position samples to be taken during each blade set motion.
     * Modern motor controllers can read and save the motor position encoder value at intervals
     * during motion.
     * @return 
     */
    public int getNumSamp()
    {
        return numSamp;
    }

    /**
     * Gets the base address in ISA space of the Acces I/O card.
     * @return the address
     */
    public int getDioAddr()
    {
        return dioAddr;
    }

    /**
     * Gets the IRQ level of the Acces I/O card.
     * @return the level
     */
    public int getDioLevel()
    {
        return dioLevel;
    }

    /**
     * Gets the value to place in the configuration register of the Acces I/O card.
     * @return the configuration value
     */
    public int getDioConf()
    {
        return dioConf;
    }

    /**
     * Gets the ISA space offset of the Acces I/O register serving inputs from the Hall sensors.
     * @return the port number
     */
    public int getDioHPort()
    {
        return dioHPort;
    }

    /**
     * Gets the ISA space offset of the Acces I/O register serving inputs from the motor controller.
     * @return the port number
     */
    public int getDioIPort()
    {
        return dioIPort;
    }

    /**
     * Gets the number of the Acces input line connected to the motor controller's "output 33".
     * @return the line number
     */
    public int getDioILine()
    {
        return dioILine;
    }

    /**
     * Gets the ISA space offset of the Acces I/O register serving outputs to the motor controller.
     * @return the port number
     */
    public int getDioOPort()
    {
        return dioOPort;
    }

    /**
     * Gets the number of the Acces output line connected to the motor controller's "input 4".
     * @return the line number
     */
    public int getDioOLine()
    {
        return dioOLine;
    }

    /**
     * Gets the ISA base address of the Helios ADC unit.
     * @return the address
     */
    public int getAdAddr()
    {
        return adAddr;
    }

    /**
     * Gets the IRQ level of the Helios ADC unit.
     * @return the level number
     */
    public int getAdLevel()
    {
        return adLevel;
    }

    /**
     * Gets the number of the Helios ADC channel connected to the motor temperature output.
     * @return the channel number
     */
    public int getAdTempChn()
    {
        return adTempChn;
    }

    /**
     * Gets the pulses per unit value given to the motor controller, the absolute value
     * of what getPpm() returns.
     * @return the ppu value
     */
    public double ppu()
    {
        return Math.abs(ppMm);
    }

    /** 
     * Converts a blade set travel distance, e.g., a step size, into a change in motor coordinates.
     * @param dist the travel distance in mm where a positive value means forward or away from home
     * and a negative value means backward or toward home
     * @return the change in motor distance = dist * signum(getPpMm()).
     */
    public double motorDistance(double dist)
    {
        return Math.signum(ppMm) * dist;
    }

    /**
     * Converts a motor position encoder reading into absolute blade set position in mm.
     * @param encoder the encoder value
     * @return the absolute position = encoder / getPpMm()
     */
    public double position(int encoder)
    {
        return homePosn + encoder / ppMm;
    }

    /**
     * Converts a blade set relative position into an absolute position.
     * @param relPosition the dimensionless relative position in the interval [0, 1]
     * @return the absolute position = getOpen() + relPosition * (getClosed() - getOpen())
     */
    public double position(double relPosition)
    {
        return openPosn + relPosition * (clsdPosn - openPosn);
    }

    /**
     * Converts a motor encoder position into a relative position. Allows for
     * the possibility that the home position is behind the position assigned to
     * relative position 0.
     * @param encoder the encoder position value.
     * @return the relative position = (position(encoder) - getOpen()) / (getClosed() - getOpen())
     */
    public double relPosition(int encoder)
    {
        return (encoder / ppMm + homePosn - openPosn) / (clsdPosn - openPosn);
    }

    /**
     * Calculates the relative position of the blade set given the absolute position.
     * @param position the absolute position in mm
     * @return the relative position = (position - getOpen()) / (getClosed() - getOpen())
     */
    public double relPosition(double position)
    {
        return (position - openPosn) / (clsdPosn - openPosn);
    }

    /**
     * Calculates the software upper limit on the blade set's absolute position.
     * @return the limit
     */
    public double posSoftLimit()
    {   // Depending on which way is forward for the blade set we use
        // either the high or low software limit set in the motor controller.
        double reach = Math.signum(ppMm) * (clsdPosn - openPosn);
        return ((reach > 0.0f) ? POS_SOFT_LIM : NEG_SOFT_LIM) * reach;
    }

    /** Calculates the software lower limit on the blade set's absolute position.
     * @return the limit
     */
    public double negSoftLimit()
    {
        double reach = Math.signum(ppMm) * (clsdPosn - openPosn);
        return ((reach > 0.0f) ? NEG_SOFT_LIM : POS_SOFT_LIM) * reach;
    }


   /**
    ***************************************************************************
    **
    **  Reads a text file of Hall sensor calibration data (transition positions)
    *   from the current working directory. The file is named Calibration_0.dat
    *   or Calibration_1.dat. For a description of the format see {@link readCalibration(String)}.
    *   @param index the blade set index, 0 or 1.
    *   @return the calibration object, or null if reading failed
    *
    **
    ***************************************************************************
    */
    public static BladeSetCalibration readCalibration(int index)
    {
        return readCalibration(CALIB_FILE[index]);
    }

    /**
     * Reads a file of Hall sensor calibration data (transition positions)
     * from a text file with the given path name (relative or absolute). The
     * file contains space-separated columns in the following order and format:
     * <ol>
     * <li>"+" or "-" if the blade set was extending or retracting, respectively,</li>
     * <li>The sensor ID number (int),</li>
     * <li>The index distinguishing consecutive transitions for the same sensor (int),</li>
     * <li>"C" or "O" (oh) if the new sensor state was open or closed, respectively,</li>
     * <li>The absolute position of the blade set when the transition took place (double).</li>
     * </ol>
     * @param file the file's pathname
     * @return the calibration object, or null if reading failed
     */
    public static BladeSetCalibration readCalibration(String file)
    {

        /*
        **  Open the file
        */
        FileInputStream ifl;
        try {
            ifl = new FileInputStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return null;
        }

        /*
        **  Parse the data and add it to the items
        */
        ArrayList<BladeSetCalibration.Item> items = new ArrayList<>();
        Scanner inp = new Scanner(ifl);
        int line = 0, field = 0, sensor = 0, indx = 0;
        double posn;
        boolean open = false, reverse = false;
        try {
            while (inp.hasNext()) {
                switch (field) {
                case 0:
                    reverse = inp.next("[+-]").equals("-");
                    break;
                case 1:
                    sensor = inp.nextInt();
                    break;
                case 2:
                    indx = inp.nextInt();
                    break;
                case 3:
                    open = inp.next("[OC]").equals("O");
                    break;
                case 4:
                    posn = inp.nextDouble();
                    items.add(new BladeSetCalibration.Item(sensor, indx, posn,
                                                          open, reverse));
                    field = -1;
                    line++;
                    inp.nextLine();
                }
                field++;
            }
        }
        catch (NoSuchElementException e) {
            out.format("Invalid field (%s) in line %s\n", field + 1, line + 1);
            return null;
        }

        if (field != 0) {
            out.format("File ended unexpectedly in line %s\n", line + 1);
            return null;
        }

        /*
        **  Close the file
        */
        try {
            ifl.close();
        }
        catch (IOException e) {
        }

        return new BladeSetCalibration(0, items);
    }


   /**
    ***************************************************************************
    **
    **  Saves the currently loaded Hall sensor calibration data (transition positions).
    **  The file will be the standard one Calibration_n.dat in the current
    **  working directory.
    **
    ***************************************************************************
    */
    public void writeCalibration()
    {
        writeCalibration(calib, CALIB_FILE[index]);
    }

    /**
     * Saves an arbitrary Hall sensor calibration. The file will be the standard
     * one Calibration_n.dat in the current working directory.
     * @param calib the calibration to save
     */
    public void writeCalibration(BladeSetCalibration calib)
    {
        writeCalibration(calib, CALIB_FILE[index]);
    }

    /**
     * Saves the currently loaded Hall sensor calibration in a non-standard location.
     * @param file the absolute or relative pathname of the file to be created
     */
    public void writeCalibration(String file)
    {
        writeCalibration(calib, file);
    }

    /**
     * Saves an arbitrary Hall sensor calibration in a non-standard location.
     * @param calib the calibration to save
     * @param file the absolute or relative pathname of the file to be created
     */
    public static void writeCalibration(BladeSetCalibration calib, String file)
    {
        PrintStream ofl;

        /*
        **  Open the output file
        */
        try {
            ofl = new PrintStream(file);
        }
        catch (IOException e) {
            out.println(e);
            return;
        }

        /*
        **  Write the data to the output file
        */
        calib.getData().stream().forEach(tran -> {
            ofl.format("%s %s %s %s %8.2f\n", tran.isReverse() ? "-" : "+",
                       tran.getSensor(), tran.getIndex(),
                       tran.isOpen() ? "O" : "C", tran.getPosition());
        });

        /*
        **  Close the output file
        */
        ofl.close();
    }

}
