package org.lsst.ccs.subsystems.shutter.common;

import java.io.Serializable;
import java.util.List;
import java.util.ArrayList;

/**
 ***************************************************************************
 **
 **  Class containing the data produced by a move
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class MovementHistory implements Serializable {

    int                       index;          // Blade set index
    int                       status;         // Move completion status
    long                      startTime;      // Move start time
    float                     startPosition;  // Move start position
    long                      endTime;        // Move end time
    float                     endPosition;    // Move end position
    List<BladePosition>  positions;      // List of encoder data
    List<HallTransition> transitions;    // List of Hall sensor data


    public MovementHistory()
    {
        positions = new ArrayList<BladePosition>();
        transitions = new ArrayList<HallTransition>();
    }

    public MovementHistory(int index, float startPosition)
    {
        this.index = index;
        this.status = 0;
        this.startPosition = startPosition;
        positions = new ArrayList<BladePosition>();
        transitions = new ArrayList<HallTransition>();
    }


        public MovementHistory(int index, int status,
                               long startTime, float startPosition,
                               long endTime, float endPosition,
                               ArrayList<BladePosition> positions,
                               ArrayList<HallTransition> transitions)
    {
        this.index = index;
        this.status = status;
        this.startTime = startTime;
        this.startPosition = startPosition;
        this.endTime = endTime;
        this.endPosition = endPosition;
        this.positions = positions;
        this.transitions = transitions;
    }


    public void setBladeSetIndex(int index)
    {
        this.index = index;
    }


    public void setStatus(int status)
    {
        this.status = status;
    }


    public void setStartTime(long startTime)
    {
        this.startTime = startTime;
    }


    public void setStartPosition(float startPosition)
    {
        this.startPosition = startPosition;
    }


    public void setEndTime(long endTime)
    {
        this.endTime = endTime;
    }


    public void setEndPosition(float endPosition)
    {
        this.endPosition = endPosition;
    }


    public void setMovementProfile(List<BladePosition> positions)
    {
        this.positions = positions;
    }


    public void setHallTransitions(List<HallTransition> transitions)
    {
        this.transitions = transitions;
    }

    public void addHallTransition(HallTransition transition) {
        this.transitions.add(transition);
    }
    
    /**
     *  Get BladeSet Index that produced the movement.
     *
     */
    public int getBladeSetIndex()
    {
        return index;
    }


    /**
     *  Get the movement completion status
     *
     */
    public int getStatus()
    {
        return status;
    }


    /**
     *  Get the start time of the movement in Epoch microseconds.
     *
     */
    public long getStartTime()
    {
        return startTime;
    }


    /**
     *  Get the end time of the movement in Epoch microseconds.
     *
     */
    public long getEndTime()
    {
        return endTime;
    }


    /**
     *  Get the position at the beginning of the movement.
     *  This is a number [0,1].
     *
     */
    public float getStartPosition()
    {
        return startPosition;
    }


    /**
     *  Get the position at the end of the movement.
     *  This is a number [0,1].
     *
     */
    public float getEndPosition()
    {
        return endPosition;
    }


    /**
     *  Get the profile of the movement.
     *  The profile is time ordered.
     *
     */
    public List<BladePosition> getMovementProfile()
    {
        return positions;
    }



    /**
     *  Get the list of Hall Sensor transitions corresponding to this movement.
     *  The transitions are globally time ordered.
     *
     */
    public List<HallTransition> getHallTransitions()
    {
        return transitions;
    }

}
