package org.lsst.ccs.subsystems.refrig.ui;

import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.util.Date;
import java.util.Timer;
import java.util.TimerTask;
import javax.swing.JComponent;
import javax.swing.JFrame;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.BusMessage;
import org.lsst.ccs.bus.CommandAck;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.MessagingFactory;
import org.lsst.ccs.bus.ModuleInvokerCommand;
import org.lsst.ccs.subsystems.refrig.status.RefrigChannelLimitStatus;
import org.lsst.ccs.subsystems.refrig.status.RefrigConfigurationSavedStatus;
import org.lsst.ccs.subsystems.refrig.status.RefrigTrendingStatus;
import org.lsst.ccs.subsystems.refrig.status.RefrigTrendingSummaryStatus;

/**
 *
 * @author max
 */
public class RefrigGUISubsystem extends Subsystem {

    static final int NOT_INITIALIZED = 0, INITIALIZATION_REQUEST_SENT = 5, INITIALIZING = 10, INITIALIZED = 15;
    static final String refrig_dest = "ccs-refrig";
    private MessagingFactory mfac;
    PrintWriter out = new PrintWriter(new OutputStreamWriter(System.out));
    RefrigAssembly assembly;
    private long lastHeartBeat;
    int currentStatus = NOT_INITIALIZED;

    public RefrigGUISubsystem() {
        assembly = new RefrigAssembly(this);
    }

    public void initGui() {
        setName("RefrigerationGUIModule");
        mfac = MessagingFactory.getInstance().forSubsystem(getName());
        setListenToStatus(true);
        setStatusBroadcastPeriod(0);
        start();


        Timer t = new Timer();
        t.scheduleAtFixedRate(new TimerTask() {

            @Override
            public void run() {
                long delta = (System.currentTimeMillis() - lastHeartBeat)/1000;
                if ( currentStatus != NOT_INITIALIZED && delta > 30 ) {
                    resetGui();
                }
            }
        }, new Date(), 5000);


    }

    void sendCommandToInitializeGui() {
        currentStatus = INITIALIZATION_REQUEST_SENT;
        ModuleInvokerCommand cmd = new ModuleInvokerCommand("publishTrendingSummary");
        cmd.setDestination(refrig_dest);
        mfac.sendCommand(cmd);

    }


    public void resetGui() {
        assembly.getTrendingTable().resetTrendingTableModel();
        currentStatus = NOT_INITIALIZED;
    }

    public JComponent getGuiLayout() {
        return assembly;
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {

        System.setProperty("java.naming.factory.initial", "com.sun.enterprise.naming.SerialInitContextFactory");
        System.setProperty("java.naming.factory.url.pkgs", "com.sun.enterprise.naming");
        System.setProperty("java.naming.factory.state", "com.sun.corba.ee.impl.presentation.rmi.JNDIStateFactoryImpl");
        System.setProperty("org.omg.CORBA.ORBInitialHost", "localhost");
        System.setProperty("org.omg.CORBA.ORBInitialPort", "3700");

//        Properties props = FrameworkUtils.loadProperties(FrameworkUtils.getDefaultJndiProperties(),RefrigGUISubsystem.class.getClass(),"jndi.properties");

        RefrigGUISubsystem t = new RefrigGUISubsystem();

        JFrame frame = new JFrame("Refrigeration Subsystem");
        frame.setContentPane(t.getGuiLayout());
        frame.pack();
        frame.setVisible(true);
        t.initGui();
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);


    }

    public void setPrintWriter(PrintWriter out) {
        this.out = out;
    }

    @Override
    public void onStatus(BusMessage s) {
        if (currentStatus == INITIALIZED) {
            if (s instanceof RefrigTrendingStatus) {
                assembly.getTrendingTable().updateTableValue((RefrigTrendingStatus) s);
                assembly.getControlPanel().updateControlPanel((RefrigTrendingStatus) s);
            } else if (s instanceof RefrigChannelLimitStatus) {
                assembly.getTrendingTable().updateTableValue((RefrigChannelLimitStatus) s);
            } else if (s instanceof RefrigConfigurationSavedStatus) {
                assembly.getTrendingTable().configurationSaved();
            }
        }



        if (s.getOrigin().equals(refrig_dest)) {
            lastHeartBeat = s.getTimeStamp();
            if (currentStatus == NOT_INITIALIZED) {
                sendCommandToInitializeGui();
            }
            if (currentStatus == INITIALIZATION_REQUEST_SENT) {
                if (s instanceof RefrigTrendingSummaryStatus) {
                    RefrigTrendingSummaryStatus status = (RefrigTrendingSummaryStatus) s;
                    initializeChannels(status);
                }
            }

        }
    }

    public void submitLimit(int index, double value, boolean isLowLimit) {
        Integer arg1 = index;
        Double arg2 = value;
        String command = isLowLimit ? "setLowLimit" : "setHighLimit";

        ModuleInvokerCommand cmd = new ModuleInvokerCommand(command, arg1, arg2);
        cmd.setDestination(refrig_dest);
        mfac.sendCommand(cmd);
    }

    public void saveConfiguration() {
        ModuleInvokerCommand cmd = new ModuleInvokerCommand("saveConfiguration");
        cmd.setDestination(refrig_dest);
        mfac.sendCommand(cmd);
    }

    public void setPowerStatus(boolean enablePower) {
        Integer arg = enablePower ? 1 : 0;
        ModuleInvokerCommand cmd = new ModuleInvokerCommand("setPowerEnable", arg);
        cmd.setDestination(refrig_dest);
        mfac.sendCommand(cmd);
    }

    public void setLoadTripStatus(boolean enableLoadTrip) {
        Integer arg = enableLoadTrip ? 1 : 0;
        ModuleInvokerCommand cmd = new ModuleInvokerCommand("setLoadTripEnable", arg);
        cmd.setDestination(refrig_dest);
        mfac.sendCommand(cmd);
    }

    public void onReply(CommandReply r) {
        //FIXME temporary fix to prevent multiple GUIs from interfering with each other.
        //See https://jira.slac.stanford.edu/browse/LSSTCCS-9
//        System.out.println("****** got command reply " + currentStatus);
//        if (currentStatus == INITIALIZING) {
//            if (r.getReply() instanceof RefrigTrendingSummaryStatus) {
//                RefrigTrendingSummaryStatus status = (RefrigTrendingSummaryStatus) r.getReply();
//                initializeChannels(status);
//            }
//        }
    }

    public void onAck(CommandAck a) {
        currentStatus = INITIALIZING;
        System.out.println("*************** got acknolegment " + currentStatus);
    }

    private void initializeChannels(RefrigTrendingSummaryStatus status) {
        assembly.getTrendingTable().updateTableModel(status);
        assembly.getControlPanel().updateControlPanel(status.getRefrigStatus());
        currentStatus = INITIALIZED;
    }

}
