package org.lsst.ccs.subsystems.refrig;

/**
 ***************************************************************************
 **
 **  Implements the simulated refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RefrigTestSim extends RefrigTest {

   /**
    ***************************************************************************
    **
    **  Container class for channel simulation data
    **
    ***************************************************************************
    */
    private static class ChanSim {

        float    operVal;       // Operating value
        float    operTime;      // Time (secs) to reach operating value
        float    relaxTime;     // Time (secs) to reach ambient value
        float    startVal;      // Starting value
        float    endVal;        // Ending value
        float    rate;          // Rate of change

        public ChanSim(float operVal, float operTime, float relaxTime)
        {
            this.operVal   = operVal;
            this.operTime  = operTime;
            this.relaxTime = relaxTime;
            this.startVal  = 0f;
            this.endVal    = 0f;
            this.rate      = 0f;
        }
    }

   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static float
        CDT_OVALUE  = 110f,
        CDT_OTIME   = 120f,
        CDT_RTIME   = 1200f,
        CDP_OVALUE  = 300f,
        CDP_OTIME   = 30f,
        CDP_RTIME   = 30f,
        C3T_OVALUE  = -120f,
        C3T_OTIME   = 3600f,
        C3T_RTIME   = 7200f,
        C4T_OVALUE  = -120f,
        C4T_OTIME   = 3600f,
        C4T_RTIME   = 7200f,
        CSP_OVALUE  = 10f,
        CSP_OTIME   = 30f,
        CSP_RTIME   = 30f,
        HLT_OVALUE  = -100f,
        HLT_OTIME   = 3600f,
        HLT_RTIME   = 7200f,
        AMBIENT_TEMP  = 20f,
        AMBIENT_PRESS = 30f,
        LOAD_POWER    = 90f;

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private int state = 0;
    private ChanSim[] simData = new ChanSim[N_CHANS];
    private long baseTime = System.currentTimeMillis();


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public RefrigTestSim()
    {
        super();
        simData[CDT_CHAN_ID] = new ChanSim(CDT_OVALUE, CDT_OTIME, CDT_RTIME);
        simData[CDP_CHAN_ID] = new ChanSim(CDP_OVALUE, CDP_OTIME, CDP_RTIME);
        simData[C3T_CHAN_ID] = new ChanSim(C3T_OVALUE, C3T_OTIME, C3T_RTIME);
        simData[C4T_CHAN_ID] = new ChanSim(C4T_OVALUE, C4T_OTIME, C4T_RTIME);
        simData[CSP_CHAN_ID] = new ChanSim(CSP_OVALUE, CSP_OTIME, CSP_RTIME);
        simData[HLT_CHAN_ID] = new ChanSim(HLT_OVALUE, HLT_OTIME, HLT_RTIME);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the simulated sensors
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Set initial simulator values
        */
        simData[CDT_CHAN_ID].startVal = AMBIENT_TEMP;
        simData[CDT_CHAN_ID].endVal   = AMBIENT_TEMP;
        chanData[CDT_CHAN_ID].value   = AMBIENT_TEMP;

        simData[C3T_CHAN_ID].startVal = AMBIENT_TEMP;
        simData[C3T_CHAN_ID].endVal   = AMBIENT_TEMP;
        chanData[C3T_CHAN_ID].value   = AMBIENT_TEMP;

        simData[C4T_CHAN_ID].startVal = AMBIENT_TEMP;
        simData[C4T_CHAN_ID].endVal   = AMBIENT_TEMP;
        chanData[C4T_CHAN_ID].value   = AMBIENT_TEMP;

        simData[HLT_CHAN_ID].startVal = AMBIENT_TEMP;
        simData[HLT_CHAN_ID].endVal   = AMBIENT_TEMP;
        chanData[HLT_CHAN_ID].value   = AMBIENT_TEMP;

        simData[CDP_CHAN_ID].startVal = AMBIENT_PRESS;
        simData[CDP_CHAN_ID].endVal   = AMBIENT_PRESS;
        chanData[CDP_CHAN_ID].value   = AMBIENT_PRESS;

        simData[CSP_CHAN_ID].startVal = AMBIENT_PRESS;
        simData[CSP_CHAN_ID].endVal   = AMBIENT_PRESS;
        chanData[CSP_CHAN_ID].value   = AMBIENT_PRESS;
    }


   /**
    ***************************************************************************
    **
    **  Read the simulated sensors
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        mccTS = powerTS = System.currentTimeMillis();

        for (int j = 0; j < N_CHANS; j++) {
            float value;
            Channel chan = chanData[j];
            if (chan.type == CHAN_TYPE_MCC) {
                float intvl = (float)((System.currentTimeMillis() - baseTime)
                                        / 1000.0);
                ChanSim sim = simData[j];
                value = sim.startVal + sim.rate * intvl;
                if (sim.rate < 0) {
                    if (value < sim.endVal) value = sim.endVal;
                }
                else {
                    if (value > sim.endVal) value = sim.endVal;
                }
            }
            else {
                value = ((state >> LOAD_STATE_BIT)
                           & (state >> ENABLE_STATE_BIT) & 1) != 0
                          ? LOAD_POWER : 0f;
            }
            chan.value = Math.signum(value)
                           * (float)(Math.floor(100.0 * Math.abs(value) + 0.5)
                                       / 100.0);
            configAlarm(j);
        }
    }


   /**
    ***************************************************************************
    **
    **  Configures the alarm for one channel using its limits
    **
    **  @param  id  The channel ID (0 - 7)
    **
    ***************************************************************************
    */
    @Override
    void configAlarm(int id)
    {
        Channel chan = chanData[id];
        if (chan.type != CHAN_TYPE_MCC) return;

        int value = 1;
        if (chan.checkLo && chan.value < chan.limitLo) value = 0;
        if (chan.checkHi && chan.value > chan.limitHi) value = 0;
        setStateBit(chan.stateBitNum, value);
    }


   /**
    ***************************************************************************
    **
    **  Sets the value of a state bit
    **
    ***************************************************************************
    */
    @Override
    void setStateBit(int bit, int value)
    {
        if (value != 0)
            state |= (1 << bit);
        else
            state &= ~(1 << bit);

        if (bit == ENABLE_STATE_BIT) {
            baseTime = System.currentTimeMillis();
            for (int j = 0; j < N_CHANS; j++) {
                Channel chan = chanData[j];
                if (chan.type != CHAN_TYPE_MCC) continue;
                ChanSim sim = simData[j];
                float relaxVal = chan.isTemp ? AMBIENT_TEMP : AMBIENT_PRESS;
                sim.startVal = chan.value;
                if (value != 0) {
                    sim.endVal = sim.operVal;
                    sim.rate = (sim.endVal - relaxVal) / sim.operTime;
                }
                else {
                    sim.endVal = relaxVal;
                    sim.rate = (sim.endVal - sim.operVal) / sim.relaxTime;
                }
            }
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the value of a state bit
    **
    ***************************************************************************
    */
    @Override
    int getStateBit(int bit)
    {
        return (state >> bit) & 1;
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the value of the state word
    **
    **  @return  The current value of the state word
    **
    ***************************************************************************
    */
    @Override
    int getStateWord()
    {
        return state;
    }

}
