package org.lsst.ccs.subsystems.refrig;

import javax.usb.UsbException;
import org.lsst.ccs.drivers.mcc.MccTc;
import org.lsst.ccs.drivers.mcc.MccTcAi;
import org.lsst.ccs.drivers.wattsup.WattsUp;

/**
 ***************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public class RefrigTestHdw extends RefrigTest implements WattsUp.DataListener {

   /**
    ***************************************************************************
    **
    **  Inner class to implement thread to update green state LEDs
    **
    ***************************************************************************
    */
    private class UpdateLEDs implements Runnable {

        @Override
        public void run()
        {
            while (true) {
                int value;
                try {
                    value = (tcai.dioIn() ^ STATE_LEDS_MASK) & STATE_LEDS_MASK;
                }
                catch (UsbException e) {
                    throw new RefrigException("Error reading DIO lines", e);
                }
                try {
                    tc.dioOut(value);
                }
                catch (UsbException e) {
                    throw new RefrigException("Error writing DIO lines", e);
                }
                try {
                    Thread.sleep(LED_UPDATE_PERIOD);
                }
                catch (InterruptedException e) {
                }
            }
        }
    }

                
   /**
    ***************************************************************************
    **
    **  Private constants
    **
    ***************************************************************************
    */
    private final static int
        STATE_FLIP_MASK = ((1 << N_STATE_BITS) - 1) ^ (1 << ENABLE_STATE_BIT),
        STATE_LEDS_MASK = STATE_FLIP_MASK,
        LED_UPDATE_PERIOD = 1000;
    private final static float
        PRESSURE_SCALE = 10000f;

   /**
    ***************************************************************************
    **
    **  Private fields
    **
    ***************************************************************************
    */
    private MccTcAi tcai;
    private MccTc tc;
    private WattsUp wtu;
    private float power;
    private Thread updateLEDs = new Thread(new UpdateLEDs());


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors
    **
    ***************************************************************************
    */
    @Override
    void initSensors()
    {
        /*
        **  Connect to and configure the first MCC DAQ box (USB-TC-AI)
        */
        try {
            tcai = new MccTcAi();
            tcai.open(null, true);
            tcai.blink();
            for (int j = 0; j < N_STATE_BITS; j++) {
                tcai.configAlarm(j, 0, 0, 0f, 0f);
            }
            tcai.dioConfig(0);
            tcai.dioOut(0);
            tcai.setTCType(CDT_HW_CHAN, MccTcAi.TC_TYPE_T);
            tcai.setTCType(C3T_HW_CHAN, MccTcAi.TC_TYPE_T);
            tcai.setTCType(C4T_HW_CHAN, MccTcAi.TC_TYPE_T);
            tcai.setTCType(HLT_HW_CHAN, MccTcAi.TC_TYPE_T);
            tcai.setGain(CDP_HW_CHAN, MccTcAi.RANGE_1_25V);
            tcai.setVoltageConnType(CDP_HW_CHAN, MccTcAi.VCT_DIFFERENTIAL);
            tcai.setGain(CSP_HW_CHAN, MccTcAi.RANGE_1_25V);
            tcai.setVoltageConnType(CSP_HW_CHAN, MccTcAi.VCT_DIFFERENTIAL);
        }
        catch (UsbException e) {
            throw new RefrigException("Error initializing USB-TC-AI module", e);
        }

        /*
        **  Connect to and configure the second MCC DAQ box (USB-TC)
        */
        try {
            tc = new MccTc();
            tc.open(null, true);
            tc.blink();
            tc.dioOut(0);
        }
        catch (UsbException e) {
            throw new RefrigException("Error initializing USB_TC module", e);
        }

        /*
        **  Connect to and configure the WattsUp? meter
        */
        try {
            wtu = new WattsUp();
            wtu.open();
            wtu.setLoggedFields(1 << WattsUp.FLD_WATTS);
            wtu.setExternalLogging(1);
            wtu.addDataListener(this);
        }
        catch (Exception e) {
            log.error("Cannot access WattsUp meter: " + e);
        }

        /*
        **  Start the thread which updates the DIO lines on the second DAQ box
        */
        updateLEDs.start();
    }


   /**
    ***************************************************************************
    **
    **  Reads the sensor data
    **
    ***************************************************************************
    */
    @Override
    void readSensors()
    {
        mccTS = System.currentTimeMillis();
        for (int j = 0; j < N_CHANS; j++) {
            Channel chan = chanData[j];
            if (chan.type == CHAN_TYPE_MCC) {
                try {
                    chan.value = tcai.adcIn(chan.hwChan, 0);
                    if (!chan.isTemp) chan.value *= PRESSURE_SCALE; 
                }
                catch (UsbException e) {
                    log.info(e);
                }
            }
            else {
                chan.value = power;
            }
            chan.value = Math.signum(chan.value)
                           * (float)(Math.floor(100.0 * Math.abs(chan.value)
                                                  + 0.5) / 100.0);
        }
    }


   /**
    ***************************************************************************
    **
    **  Receives data periodically from the WattsUp? meter
    **
    **  @param  data  The array of data from the meter.
    **
    ***************************************************************************
    */
    @Override
    public void process(float[] data)
    {
        power = data[WattsUp.FLD_WATTS];
        powerTS = System.currentTimeMillis();
    }


   /**
    ***************************************************************************
    **
    **  Configures the alarm for one channel using its limits
    **
    **  @param  id  The channel ID (0 - 7)
    **
    ***************************************************************************
    */
    @Override
    void configAlarm(int id)
    {
        Channel chan = chanData[id];
        if (chan.type != CHAN_TYPE_MCC) return;

        int inOptn = chan.hwChan, outOptn = 0;
        float value1 = 0, value2 = 0;
        if (chan.checkLo) {
            outOptn = 3;
            value1 = chan.limitLo;
            if (chan.checkHi) {
                inOptn |= 0x40;
                value2 = chan.limitHi;
            }
            else {
                inOptn |= 0x20;
            }
        }
        else if (chan.checkHi) {
            outOptn = 3;
            value1 = chan.limitHi;
        }
        if (!chan.isTemp) {
            value1 /= PRESSURE_SCALE;
            value2 /= PRESSURE_SCALE;
        }
        try {
            tcai.configAlarm(chan.stateBitNum, inOptn, outOptn, value1, value2);
            if (outOptn == 0) tcai.dioOutBit(chan.stateBitNum, 0);
        }
        catch (UsbException e) {
            throw new RefrigException("Error configuring alarms", e);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the value of a state bit
    **
    ***************************************************************************
    */
    @Override
    void setStateBit(int bit, int value)
    {
        try {
            tcai.dioOutBit(bit, ((STATE_FLIP_MASK >> bit) ^ value) & 1);
        }
        catch (UsbException e) {
            throw new RefrigException("Error setting DIO line", e);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the value of a state bit
    **
    ***************************************************************************
    */
    @Override
    int getStateBit(int bit)
    {
        try {
            return tcai.dioInBit(bit) ^ ((STATE_FLIP_MASK >> bit) & 1);
        }
        catch (UsbException e) {
            throw new RefrigException("Error setting DIO line", e);
        }
    }
            

   /**
    ***************************************************************************
    **
    **  Gets the state word value
    **
    **  @return  The current value of the state word
    **
    ***************************************************************************
    */
    @Override
    int getStateWord()
    {
        int value = 0;

        try {
            value = tcai.dioIn() ^ STATE_FLIP_MASK;
        }
        catch (UsbException e) {
            throw new RefrigException("Error reading DIO lines", e);
        }

        return value;
    }

}
