package org.lsst.ccs.subsystems.refrig;

import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Properties;
import org.lsst.ccs.bus.ValueNotification;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.refrig.data.RefrigChannel;
import org.lsst.ccs.subsystems.refrig.status.RefrigChannelLimitStatus;
import org.lsst.ccs.subsystems.refrig.status.RefrigConfigurationSavedStatus;
import org.lsst.ccs.subsystems.refrig.status.RefrigTrendingStatus;
import org.lsst.ccs.subsystems.refrig.status.RefrigTrendingSummaryStatus;
import org.lsst.ccs.subsystems.refrig.utils.RefrigUtils;

/**
 ***************************************************************************
 **
 **  Implements the refrigeration long lines test modular subsystem
 **
 **  This is an abstract class implementing the interactions with the CCS
 **  buses, but deferring the hardware interaction, real or simulated, to
 **  sub-classes.
 **
 **  @author Owen Saxton
 **
 ***************************************************************************
 */
public abstract class RefrigTest extends Module {

   /**
    ***************************************************************************
    **
    **  Container class for channel parameters
    **
    ***************************************************************************
    */
    static class Channel {

        String   name;          // Channel name
        String   description;   // Channel description
        int      type;          // Channel type (MCC or WattsUp)
        boolean  isTemp;        // Whether MCC channel is temperature
        int      hwChan;        // Hardware channel for MCC channel
        int      stateBitNum;   // Bit number for indicating good/bad state
        boolean  checkLo;       // Whether low limit is checked
        boolean  checkHi;       // Whether high limit is checked
        float    limitLo;       // Low limit value
        float    limitHi;       // High limit value
        String   limitLoName;   // Low limit name (derived)
        String   limitHiName;   // High limit name (derived)
        float    value;         // Current value read from channel

        public Channel(String name, String desc, int type,
                       boolean isTemp, int hwChan, int bitNum)
        {
            this.name = name;
            this.description = desc;
            this.type = type;
            this.isTemp = isTemp;
            this.hwChan = hwChan;
            this.stateBitNum = bitNum;
            this.limitLoName = name + "LoLim";
            this.limitHiName = name + "HiLim";
        }
    }

   /**
    ***************************************************************************
    **
    **  Exception class for laundering exceptions
    **
    ***************************************************************************
    */
    public static class RefrigException extends RuntimeException {

        public RefrigException(String msg, Throwable cause)
        {
            super(msg, cause);
        }

    }

   /**
    ***************************************************************************
    **
    **  Package constants
    **
    ***************************************************************************
    */
    final static int
        N_CHANS        = 7,
        CDT_CHAN_ID    = 0,  // Compressor discharge temperature
        CDP_CHAN_ID    = 1,  // Compressor discharge pressure
        C3T_CHAN_ID    = 2,  // C3 liquid cap temperature
        C4T_CHAN_ID    = 3,  // Pre C4 evaporator temperature
        HLT_CHAN_ID    = 4,  // Heat load temperature
        CSP_CHAN_ID    = 5,  // Compressor suction pressure
        HLP_CHAN_ID    = 6,  // Heat load power
    //        CST_CHAN_ID    = 0,  // Compressor suction temperature
    //        ORT_CHAN_ID    = 0,  // Oil return temperature
    //        EVT_CHAN_ID    = 0,  // Post C4 evaporator temperature
    //        EJP_CHAN_ID    = 0,  // Evaporator Joule-Thompson pressure
    //        ERP_CHAN_ID    = 0,  // Evaporator return pressure

        N_STATE_BITS     = 8,
        CDT_STATE_BIT    = 0,
        CDP_STATE_BIT    = 1,
        C3T_STATE_BIT    = 2,
        C4T_STATE_BIT    = 3,
        HLT_STATE_BIT    = 4,
        CSP_STATE_BIT    = 5,
        LOAD_STATE_BIT   = 6,
        ENABLE_STATE_BIT = 7,

        CHAN_TYPE_MCC  = 0,
        CHAN_TYPE_PWR  = 1,

        CDT_HW_CHAN    = 0,
        C3T_HW_CHAN    = 1,
        C4T_HW_CHAN    = 2,
        HLT_HW_CHAN    = 3,
        CDP_HW_CHAN    = 6,
        CSP_HW_CHAN    = 7;

    final static String
        HI_LIMIT_CHK = "HiLimCheck",
        LO_LIMIT_CHK = "LoLimCheck",
        PROP_FILE = "refrigtest.properties";

   /**
    ***************************************************************************
    **
    **  Package fields
    **
    ***************************************************************************
    */
    Properties config;
    Channel[] chanData = new Channel[N_CHANS];
    int loLimCheck, hiLimCheck;
    boolean enabled = false, loadTrip = false;
    long powerTS = System.currentTimeMillis(), mccTS = powerTS;


   /**
    ***************************************************************************
    **
    **  Constructor
    **
    ***************************************************************************
    */
    public RefrigTest()
    {
        super();
        chanData[CDT_CHAN_ID] = new Channel("CmpDisTmp",
                                            "Compressor discharge temperature",
                                            CHAN_TYPE_MCC, true, CDT_HW_CHAN,
                                            CDT_STATE_BIT);
        chanData[CDP_CHAN_ID] = new Channel("CmpDisPrs",
                                            "Compressor discharge pressure",
                                            CHAN_TYPE_MCC, false, CDP_HW_CHAN,
                                            CDP_STATE_BIT);
        chanData[C3T_CHAN_ID] = new Channel("C3LiqdTmp",
                                            "C3 liquid cap temperature",
                                            CHAN_TYPE_MCC, true, C3T_HW_CHAN,
                                            C3T_STATE_BIT);
        chanData[C4T_CHAN_ID] = new Channel("PreC4Tmp",
                                            "Pre C4 evaporator temperature",
                                            CHAN_TYPE_MCC, true, C4T_HW_CHAN,
                                            C4T_STATE_BIT);
        chanData[HLT_CHAN_ID] = new Channel("LoadTmp", "Heat load temperature",
                                            CHAN_TYPE_MCC, true, HLT_HW_CHAN,
                                            HLT_STATE_BIT);
        chanData[CSP_CHAN_ID] = new Channel("CmpSucPrs",
                                            "Compressor suction pressure",
                                            CHAN_TYPE_MCC, false, CSP_HW_CHAN,
                                            CSP_STATE_BIT);
        chanData[HLP_CHAN_ID] = new Channel("LoadPwr", "Heat load power",
                                            CHAN_TYPE_PWR, false, 0, -1);
    }


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration subsystem
    **
    ***************************************************************************
    */
    @Override
    public void initModule()
    {
        /*
        **  Initialize configuration data
        */
        initConfiguration();

        /*
        **  Initialize the hardware
        */
        initSensors();
        configAlarms();
        setLoadLine();
    }


   /**
    ***************************************************************************
    **
    **  Performs periodic sensor readout
    **
    ***************************************************************************
    */
    @Override
    public void tick()
    {
        readSensors();

        ArrayList<ValueNotification> trendingValues
          = new ArrayList<ValueNotification>();
        for (int j = 0; j < N_CHANS; j++) {
            Channel chan = chanData[j];
            long ts = (chan.type == CHAN_TYPE_MCC) ? mccTS : powerTS;
            trendingValues.add(new ValueNotification(chan.name, chan.value, ts));
        }
        trendingValues.add(new ValueNotification("state", getState()));
        publishData(trendingValues);

        RefrigTrendingStatus refrigTrendingStatus
          = new RefrigTrendingStatus(isPowerEnable(), isLoadTripEnable(),
                                     getState(), trendingValues);
        sendToStatus(refrigTrendingStatus);

        setLoadLine();
        }
        

   /**
    ***************************************************************************
    **
    **  Gets the enabled state
    **
    **  @return  The current power enabled state
    **
    ***************************************************************************
    */
    public boolean isPowerEnable()
    {
        return enabled;
    }


   /**
    ***************************************************************************
    **
    **  Sets the enabled state on or off
    **
    **  @param  value  The enabled state value to set: 0 = off, ~0 = on.
    **
    ***************************************************************************
    */
    public void setPowerEnable(int value)
    {
        boolean enabled = (value != 0);
        if (enabled != this.enabled) {
            this.enabled = enabled;
            setStateBit(ENABLE_STATE_BIT, enabled ? 1 : 0);
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the check load state
    **
    **  @return  The current check load temperature state
    **
    ***************************************************************************
    */
    public boolean isLoadTripEnable()
    {
        return loadTrip;
    }


   /**
    ***************************************************************************
    **
    **  Sets the check load state on or off
    **
    **  @param  value  The check load temperature state value to set: 0 = off;
    **                 ~0 = on.
    **
    ***************************************************************************
    */
    public void setLoadTripEnable(int value)
    {
        boolean loadTrip = (value != 0);
        if (loadTrip != this.loadTrip) {
            this.loadTrip = loadTrip;
            setLoadLine();
        }
    }


   /**
    ***************************************************************************
    **
    **  Gets the word of channels with low limit checking
    **
    **  @return  The bit mask of channels with low limit checking enabled
    **
    ***************************************************************************
    */
    public int getLowLimitCheck()
    {
        return loLimCheck;
    }


   /**
    ***************************************************************************
    **
    **  Gets the word of channels with high limit checking
    **
    **  @return  The bit mask of channels with high limit checking enabled
    **
    ***************************************************************************
    */
    public int getHighLimitCheck()
    {
        return hiLimCheck;
    }


   /**
    ***************************************************************************
    **
    **  Gets the operating state word
    **
    **  @return  The current value of the operating state
    **
    ***************************************************************************
    */
    public int getState()
    {
        return getStateWord();
    }


   /**
    ***************************************************************************
    **
    **  Gets the number of channels
    **
    **  @return  The number of channels being monitored
    **
    ***************************************************************************
    */
    public int getNumChans()
    {
        return N_CHANS;
    }


   /**
    ***************************************************************************
    **
    **  Gets the name of a channel
    **
    **  @param  id  The channel ID
    **
    **  @return  The name of the channel
    **
    ***************************************************************************
    */
    public String getName(int id)
    {
        return chanData[id].name;
    }


   /**
    ***************************************************************************
    **
    **  Gets the descriptive name of a channel
    **
    **  @param  id  The channel ID
    **
    **  @return  The descriptive name of the channel
    **
    ***************************************************************************
    */
    public String getDescription(int id)
    {
        return chanData[id].description;
    }


   /**
    ***************************************************************************
    **
    **  Gets the units name of a channel
    **
    **  @param  id  The channel ID
    **
    **  @return  The units name of the channel
    **
    ***************************************************************************
    */
    public String getUnits(int id)
    {
        Channel chan = chanData[id];
        if (chan.type == CHAN_TYPE_PWR) return "Watts";
        if (chan.isTemp) return "C";
        return "Psi";
    }


   /**
    ***************************************************************************
    **
    **  Gets the state bit number of a channel
    **
    **  @param  id  The channel ID
    **
    **  @return  The state bit number of the channel
    **
    ***************************************************************************
    */
    public int getStateBitNum(int id)
    {
        return chanData[id].stateBitNum;
    }


   /**
    ***************************************************************************
    **
    **  Gets the power enabled state bit number
    **
    **  @return  The enabled state bit number
    **
    ***************************************************************************
    */
    public int getPowerBitNum()
    {
        return ENABLE_STATE_BIT;
    }


   /**
    ***************************************************************************
    **
    **  Gets the heat load state bit number
    **
    **  @return  The heat load state bit number
    **
    ***************************************************************************
    */
    public int getLoadBitNum()
    {
        return LOAD_STATE_BIT;
    }


   /**
    ***************************************************************************
    **
    **  Gets the low limit for a channel
    **
    **  @param  id  The channel ID
    **
    **  @return  The low limit value for the channel
    **
    ***************************************************************************
    */
    public float getLowLimit(int id)
    {
        return chanData[id].limitLo;
    }


   /**
    ***************************************************************************
    **
    **  Sets the low limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The low limit value.
    **
    ***************************************************************************
    */
    public void setLowLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        chan.limitLo = (float)limit;
        config.setProperty(chan.limitLoName, ((Float)chan.limitLo).toString());
        configAlarm(id);
        getSubsystem().publishMetaData(chan.name, "alarmLow",
                                       String.valueOf(limit));
        sendToStatus( new RefrigChannelLimitStatus(id,limit,RefrigChannelLimitStatus.LOW_LIMIT));
    }


   /**
    ***************************************************************************
    **
    **  Gets the high limit for a channel
    **
    **  @param  id  The channel ID
    **
    **  @return  The high limit value for the channel
    **
    ***************************************************************************
    */
    public float getHighLimit(int id)
    {
        return chanData[id].limitHi;
    }


   /**
    ***************************************************************************
    **
    **  Sets the high limit for a channel
    **
    **  @param  id     The channel ID
    **
    **  @param  limit  The high limit value.
    **
    ***************************************************************************
    */
    public void setHighLimit(int id, double limit)
    {
        Channel chan = chanData[id];
        chan.limitHi = (float)limit;
        config.setProperty(chan.limitHiName, ((Float)chan.limitHi).toString());
        configAlarm(id);
        getSubsystem().publishMetaData(chan.name, "alarmHigh",
                                       String.valueOf(limit));
        sendToStatus( new RefrigChannelLimitStatus(id,limit,RefrigChannelLimitStatus.HIGH_LIMIT) );
    }


   /**
    ***************************************************************************
    **
    **  Gets the last value read from a channel
    **
    **  @param  id  The channel ID
    **
    **  @return  The current value for the channel
    **
    ***************************************************************************
    */
    public float getValue(int id)
    {
        return chanData[id].value;
    }


   /**
    ***************************************************************************
    **
    **  Saves the configuration data
    **
    ***************************************************************************
    */
    public void saveConfiguration()
    {
        FileWriter writer = null;
        try {
            writer = new FileWriter(PROP_FILE);
        }
        catch (IOException e) {
            log.info(e);
        }
        if (writer != null) {
            try {
                config.store(writer,
                             "Refrigeration long lines test configuration");
            }
            catch (IOException e) {
                log.info(e);
            }
            try {
                writer.close();
            }
            catch (IOException e) {
                log.info(e);
            }
        }
        sendToStatus(new RefrigConfigurationSavedStatus());
    }


   /**
    ***************************************************************************
    **
    **  Initializes the configuration data
    **
    ***************************************************************************
    */
    private void initConfiguration()
    {
        /*
        **  Generate a properties list containing default configuration values
        */
        Properties dfltConfig = new Properties();
        for (int j = 0; j < N_CHANS; j++) {
            Channel chan = chanData[j];
            dfltConfig.setProperty(chan.limitLoName, "0.0");
            dfltConfig.setProperty(chan.limitHiName, "0.0");
        }
        dfltConfig.setProperty(LO_LIMIT_CHK, "0");
        dfltConfig.setProperty(HI_LIMIT_CHK, "0");

        /*
        **  Create the configuration properties list and load it
        */
        config = new Properties(dfltConfig);
        FileReader reader = null;
        try {
            reader = new FileReader(PROP_FILE);
        }
        catch (FileNotFoundException e) {
            log.info(e);
        }
        if (reader != null) {
            try {
                config.load(reader);
            }
            catch (IOException e) {
                log.info(e);
            }
            try {
                reader.close();
            }
            catch (IOException e) {
                log.info(e);
            }
        }

        /*
        **  Extract values from the properties list
        */
        loLimCheck = Integer.decode(config.getProperty(LO_LIMIT_CHK));
        hiLimCheck = Integer.decode(config.getProperty(HI_LIMIT_CHK));
        for (int j = 0; j < N_CHANS; j++) {
            Channel chan = chanData[j];
            chan.checkLo = (loLimCheck & (1 << j)) != 0;
            chan.checkHi = (hiLimCheck & (1 << j)) != 0;
            chan.limitLo = Float.valueOf(config.getProperty(chan.limitLoName));
            chan.limitHi = Float.valueOf(config.getProperty(chan.limitHiName));
        }
    }


   /**
    ***************************************************************************
    **
    **  Configures all the alarms using the channel limits
    **
    ***************************************************************************
    */
    private void configAlarms()
    {
        for (int j = 0; j < N_CHANS; j++) {
            configAlarm(j);
        }
    }


   /**
    ***************************************************************************
    **
    **  Sets the state of the load line
    **
    ***************************************************************************
    */
    private void setLoadLine()
    {
        setStateBit(LOAD_STATE_BIT,
                    loadTrip ? getStateBit(chanData[HLT_CHAN_ID].stateBitNum)
                             : 1);
    }


    /**
     ***************************************************************************
     * The following methods have been added to make the GUI work.
     * This is a work-around the problem of sending commands and receive 
     * a reply on a different Thread
     ***************************************************************************
     */
    
    public RefrigTrendingSummaryStatus publishTrendingSummary() {
        RefrigTrendingSummaryStatus status = new RefrigTrendingSummaryStatus();
        for ( int i = 0; i < getNumChans(); i++) {
            RefrigChannel chan = new RefrigChannel(getName(i), getDescription(i), getUnits(i));
            int channelStateBit = getStateBitNum(i);
            chan.setStateBit(channelStateBit);
            chan.setHighLimit(getHighLimit(i));
            chan.setLowLimit(getLowLimit(i));
            chan.setValue(getValue(i));
            chan.setIsWithinLimits( RefrigUtils.isChannelWithinLimits(channelStateBit,getState()) );
            status.addRefrigChannel(chan);
        }
        // We need to add this to activate the buttons right away
        RefrigTrendingStatus refrigStatus = new RefrigTrendingStatus(isPowerEnable(),isLoadTripEnable(),getState(),null);
        status.setRefrigStatus(refrigStatus);
        // This needs to be fixed. It should not be sent to status bus.
        // Currently it is difficult to get command reply back in GUI environment
        sendToStatus(status);
        return status;
    }    


   /**
    ***************************************************************************
    **
    **  Initializes the refrigeration sensors
    **
    ***************************************************************************
    */
    abstract void initSensors();


   /**
    ***************************************************************************
    **
    **  Reads the sensor data
    **
    ***************************************************************************
    */
    abstract void readSensors();


   /**
    ***************************************************************************
    **
    **  Configures the alarm for one channel using its limits
    **
    **  @param  id  The channel ID (0 - 7)
    **
    ***************************************************************************
    */
    abstract void configAlarm(int id);


   /**
    ***************************************************************************
    **
    **  Sets the value of a state bit
    **
    **  @param  bit    The number (0 - 7) of the state bit to set.
    **
    **  @param  value  The value to set: 0 -> 0; !0 -> 1.
    **
    ***************************************************************************
    */
    abstract void setStateBit(int bit, int value);
            

   /**
    ***************************************************************************
    **
    **  Gets the value of a state bit
    **
    **  @param  bit   The number (0 - 7) of the state bit
    **
    **  @return  The value, 0 or 1, of the bit.
    **
    ***************************************************************************
    */
    abstract int getStateBit(int bit);
            

   /**
    ***************************************************************************
    **
    **  Gets the value of the entire state word
    **
    **  @return  The current state word value
    **
    ***************************************************************************
    */
    abstract int getStateWord();
    
}
