package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.DerivedChannel;

public class ConfiguredAverageChannel extends DerivedChannel {

    @LookupField(strategy = LookupField.Strategy.SIBLINGS)
    Map<String, Channel> channelMap = new HashMap<>();

    @LookupName
    private String name;

    private static final Logger LOG = Logger.getLogger(ConfiguredAverageChannel.class.getName());

    private List<Channel> channels = new ArrayList<>();
    private double sumWeights;

    @ConfigurationParameter(name = "chanNames", category = "General", 
                            description = "Names of Channels to be averaged",
                            maxLength = -1, units = "unitless")
    protected volatile List<String> chanNames = new ArrayList<>();

    @ConfigurationParameter(name = "chanWeights", category = "General", 
                            description = "Weights of Channels to be averaged",
                            maxLength = -1, units = "unitless")
    protected volatile List<Double> chanWeights = new ArrayList<>();

   /**
    *  Checks validity of a proposed set of ConfigurationParameters
    *
    *  The Channel name and weight arrays must have same length,.
    *  Channels must be found among siblings on tree.
    *  Weights must be positive-definite.
    *
    *  @param Map of ConfigurationParameters from name to value
    */
    @Override
    public void validateBulkChange(Map<String,Object> params) {
        super.validateBulkChange(params);
        List<String> channelList = (List<String>) params.get("chanNames");
        List<Double> weightList = (List<Double>) params.get("chanWeights");
        int len = channelList.size();
        if (weightList.size() != len) {
            throw new RuntimeException(name + ": chanWeights and chanNames do not have same size");
        }

        for (int i = 0; i < len; i++) {
            if (channelMap.get(channelList.get(i)) == null) {
                throw new RuntimeException(name + "/chanNames element " + 
					   channelList.get(i) + " not among available Channels");
	    }
            if (!(weightList.get(i) >= 0.)) {
		throw new RuntimeException(name + "/chanWeights element " + i +
                    " is not positive-definite");
	    }
        }
    }

   /**
    *  Set configuration parameters to new values.
    *  Checks input Map for those ConfigurationParameters which correspond
    *  to hardware settings; the others are left for framework to deal with.
    *
    *  @param  Map of parameters for which a change is requested,
    *          keyed by names
    */
    @Override
    public void setParameterBulk(Map<String,Object> params)  {
        super.setParameterBulk(params);
        Set<String> keys = params.keySet();
        if (keys.contains("chanNames")) {
	    chanNames.clear();
            chanNames.addAll((List<String>) params.get("chanNames"));
            channels.clear();
	    for (int i = 0; i < chanNames.size(); i++) {
	        channels.add(channelMap.get(chanNames.get(i)));
            }
        }
        if (keys.contains("chanWeights")) {
            chanWeights.clear();
            chanWeights.addAll((List<Double>) params.get("chanWeights"));
            sumWeights = 0.;
            for (int i = 0; i < chanWeights.size(); i++) {
                sumWeights += chanWeights.get(i);
            }
        }
    }

   /**
    *  Evaluate weighted average of specified channel values.
    *
    *  @return  double average (NaN if any input value is NaN)
    */
    @Override
    public double evaluateDerivedValue() {
        double sum = 0.;
        for (int i = 0; i < channels.size(); i++) {
            sum += (chanWeights.get(i) * channels.get(i).getValue());
        }
        return (sum/sumWeights);
    }

}