package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.data.RefrigAction;

/**
 * This class manages the refrig actions in response to Alert payloads.
 *
 * It uses the AgentPresenceManager to detect which Agents on the buses might be
 * sending Alert payloads to trigger actions.
 *
 * @author The LSST CCS Team
 */
public class RefrigActionManager implements HasLifecycle, AlertListener {

    private static final Logger LOG = Logger.getLogger(RefrigActionManager.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, ColdCompressor> coldComps = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, CryoCompressor> cryoComps = new HashMap<>();

    private final Map<String, Set<String>>[] active = new Map[RefrigAction.Action.values().length];  // Active actions with agents

    /**
     *  Constructor.
     */
    public RefrigActionManager()
    {
        for (int j = 0; j < active.length; j++) {
            active[j] = new HashMap<>();
        }
    }

    /**
     *  Build phase.
     */
    @Override
    public void build()
    {
        active[RefrigAction.Action.STOP_ALL_COLD_COMPS.ordinal()].put("", new ConcurrentSkipListSet<>());
        for (String name : coldComps.keySet()) {
            active[RefrigAction.Action.STOP_COLD_COMP.ordinal()].put(name, new ConcurrentSkipListSet<>());
        }
        active[RefrigAction.Action.STOP_ALL_CRYO_COMPS.ordinal()].put("", new ConcurrentSkipListSet<>());
        for (String name : cryoComps.keySet()) {
            active[RefrigAction.Action.STOP_CRYO_COMP.ordinal()].put(name, new ConcurrentSkipListSet<>());
        }
    }

    /**
     *  Init phase.
     */
    @Override
    public void init()
    {
        // Set a property to define agent as MPM action manager.
        propertiesService.setAgentProperty(RefrigAction.getManagerName(), "");
        // Start alert listening for Agents that have the property for RefrigActions        
        alertService.startStatusAlertListening((info) -> {
            return info.hasAgentProperty(RefrigAction.getClientName());
        });
    }

    /**
     *  Start phase.
     */
    @Override
    public void start()
    {
        //Alert listener responsible to listen for Alert emergency payloads
        //published by the origins maintained by the OriginManager
        alertService.addListener(this);
    }

    /**
     *  Shutdown phase.
     */
    @Override
    public void shutdown()
    {
        //Remove alert listeners
        alertService.removeListener(this);
    }

    /**
     *  Raised alert handler.
     * 
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        String origin = event.getSource();
        //LOG.log(Level.INFO, "onAlert called: origin = {0}, type = {1}", new Object[]{origin, event.getType()});
        switch (event.getType()) {
        case AGENT_CONNECTION:
            LOG.log(Level.INFO, "Connected to agent {0}, a source of alerts with payload {1}",
                    new Object[]{origin, RefrigAction.getName()});
            for (RaisedAlertHistory alertHistory : event.getSummary().getAllRaisedAlertHistories()) {
                processAlert(origin, alertHistory.getLatestAlert(), alertHistory.getLatestAlertState());                    
            }
            break;
        case AGENT_DISCONNECTION:
            LOG.log(Level.INFO, "Disconnected from agent {0}, a source of alerts with payload {1}",
                    new Object[]{origin, RefrigAction.getName()});
            for (RefrigAction.Action action : RefrigAction.Action.values()) {
                for (Map.Entry<String, Set<String>> entry : active[action.ordinal()].entrySet()) {
                    if (entry.getValue().remove(origin) && entry.getValue().isEmpty()) {
                        performAction(action, entry.getKey(), false);
                    }
                }
            }
            break;
        case ALERT_RAISED:
            processAlert(origin, event.getAlert(), event.getLevel());
            break;
        }

    }

    /**
     *  Processes an alert.
     * 
     *  @param  origin  Name of originating agent
     *  @param  alert  The alert
     *  @param  state  The alert state (ALARM, WARNING, NOMINAL)
     */
    private synchronized void processAlert(String origin, Alert alert, AlertState state)
    {
        RefrigAction action = RefrigAction.getData(alert);
        if (action != null) {
            LOG.log(Level.INFO, "Received {0} level alert from {1}: action = ({2}, {3})",
                    new Object[]{state, origin, action.action, action.compName});
            Set orgSet = active[action.action.ordinal()].get(action.compName);
            if (orgSet == null) return;
            boolean on = state == AlertState.ALARM;
            if (on) {
                orgSet.add(origin);
            }
            else {
                orgSet.remove(origin);
                if (!orgSet.isEmpty()) return;
            }
            performAction(action.action, action.compName, on);
        }
    }

    /**
     *  Performs the requested action.
     * 
     *  @param  action  The action
     *  @param  compName  The compressor name
     *  @param  on  Whether to turn on
     */
    private void performAction(RefrigAction.Action action, String compName, boolean on)
    {
        //LOG.log(Level.INFO, "Action requested: action = ({0}, {1}), on = {2}", new Object[]{action, compName, on});
        switch (action) {
        case STOP_ALL_COLD_COMPS:
            for (String name : coldComps.keySet()) {
                if (on || active[RefrigAction.Action.STOP_COLD_COMP.ordinal()].get(name).isEmpty()) {
                    coldComps.get(name).setPlateTempLow(on);
                }
            }
            break;
        case STOP_COLD_COMP:
            if (on || active[RefrigAction.Action.STOP_ALL_COLD_COMPS.ordinal()].get("").isEmpty()) {
                coldComps.get(compName).setPlateTempLow(on);
            }
            break;
        case STOP_ALL_CRYO_COMPS:
            for (String name : cryoComps.keySet()) {
                if (on || active[RefrigAction.Action.STOP_CRYO_COMP.ordinal()].get(name).isEmpty()) {
                    cryoComps.get(name).setPlateTempLow(on);
                }
            }
            break;
        case STOP_CRYO_COMP:
            if (on || active[RefrigAction.Action.STOP_ALL_CRYO_COMPS.ordinal()].get("").isEmpty()) {
                cryoComps.get(compName).setPlateTempLow(on);
            }
            break;
        }
    }

}
