package org.lsst.ccs.subsystem.refrig;

import java.io.Serializable;
import java.time.Duration;
import java.util.function.Predicate;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorUpdateTask;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.refrig.constants.ChillerAlerts;
import org.lsst.ccs.subsystem.refrig.constants.ChillerState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.data.ChillerControlState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/* Pre-defined command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING2;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING3;

/**
 *  Implements the chiller subsystem.
 *
 */
public class ChillerSubsystem extends Subsystem implements HasLifecycle, StatusMessageListener, AgentPresenceListener {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private InTESTChillerDevice devChiller;
    @LookupField(strategy=LookupField.Strategy.TREE)
    protected Monitor mon;; 
    @LookupName
    private String name;

    @ConfigurationParameter(isFinal = true, description = "Require listening to thermal subsystem or equivalent for Normal mode", units = "unitless")
    protected volatile boolean requireListening;

    @ConfigurationParameter(isFinal = true, description = "Which refrig subsystem to listen to (RefrigAgentProperties)", units = "unitless")
    protected volatile String listenTo;

    @ConfigurationParameter(isFinal = true, description = "path of coldplate temperature Channel to listen for", units = "unitless")
    protected volatile String coldplateChannelPath;

    // Factor to lengthen temperature-set timeout beyond estimated value
    @ConfigurationParameter(description="Factor on temperature-setting timeout",
                            range = "1.0..1.4", units = "unitless")
    protected volatile double temperatureTimeoutFactor;

    // @ConfigurationParameter(desvription = "timeout for setting flow",
    //                         units = "s")
    // protected volatile double flowTimeout;

    private static final Logger LOG = Logger.getLogger(ChillerSubsystem.class.getName());

    // Additive correction from specified ramp value to actual ramp
    private static final double rampCorrection = -0.04;  // deg-C/minute

    private final Object listeningLock = new Object();
    private volatile boolean isListeningToThermal = false;
    private volatile String thermalName = null;
    private volatile boolean setTempInProgress = false;
    private volatile boolean setFlowInProgress = false;
    private volatile String refrigAgentProperty;
    private boolean listenAlarm = false;
    private Channel chanFlow;            // flow-rate channel
    private MonitorUpdateTask taskFlow;  // Task updating flow-rate Channel
    private Channel chanFlowSet, chanTempSet, chanTankSet;  // needed for GUI
    private volatile CCSTimeStamp timeLastData;   // Most-recent received data
    private volatile AlertState lastDataAlert = AlertState.NOMINAL;
    private double lastRampValue;

    /* Threads launched from this class  */

    private Thread checkTemp;
    private Thread checkFlow;
    private static final Duration intervalCheckTemp = Duration.ofSeconds(30);
    private static final Duration intervalCheckFlow = Duration.ofSeconds(5);
    private static final Duration setFlowTimeout = Duration.ofSeconds(45);
    private static final double flowTolerance = 0.1;     // gpm

    /* Periodic Tasks  */

    private AgentPeriodicTask checkDataArrival;
    private static final Duration periodCheckData = Duration.ofMillis(500);
    private static long timeDataWarning = 3000;  // ms
    private static long timeDataAlarm = 20000;   // ms
    private AgentPeriodicTask updateCtrlState;
    private static final int UPDATE_CONTROL_INTVL = 1000;   // ms
    private AgentPeriodicTask sendDUTTemperature;
    private static final Duration periodDUT = Duration.ofMillis(1000);

    private MonitorTaskControl monitorControl;
    private final ChillerControlState controlState = new ChillerControlState();
    private boolean gotCommand;

    /**
     *  Constructor.
     */
    public ChillerSubsystem() {
        super("chiller", AgentInfo.AgentType.WORKER);
    }

    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, RefrigConstants.MONITOR_CONTROL);

        /* Set up AgentPeriodicTask to update chiller control state for gui */
        updateCtrlState = new AgentPeriodicTask("updateCtrlState", () -> updateControlState()).withPeriod(Duration.ofMillis(UPDATE_CONTROL_INTVL));
        periodicTaskService.scheduleAgentPeriodicTask(updateCtrlState);

        /* Set up periodic task to check time since last listned-for data */
        Runnable checkData = new Runnable() {
            @Override
            public void run() {
                AlertState dataAlert;
                long dt = System.currentTimeMillis() - timeLastData.getUTCInstant().toEpochMilli();
                if (dt <= timeDataWarning) {
                    dataAlert = AlertState.NOMINAL;
                } else if (dt <= timeDataAlarm) {
                    dataAlert = AlertState.WARNING;
                } else {
                    dataAlert = AlertState.ALARM;
                }
                if (requireListening && dataAlert != lastDataAlert) {
                    alertService.raiseAlert(ChillerAlerts.MISSING_THERMAL.newAlert(), dataAlert, "time since last data = " + Long.toString(dt) + " ms");
                    lastDataAlert = dataAlert;
                }
            }
	};
        checkDataArrival = new AgentPeriodicTask("checkDataArrival",
            checkData).withPeriod(periodCheckData);
        periodicTaskService.scheduleAgentPeriodicTask(checkDataArrival);

        /* Set up periodic task to send DUT temperature data to Chiller  */
        Runnable sendDUT = new Runnable() {
            @Override
            public void run() {
                if (devChiller.getTempControlMode()) {
                    devChiller.sendDUTData();
                }
            }
	};
        sendDUTTemperature = new AgentPeriodicTask("sendDutTemperature",
            sendDUT).withPeriod(periodDUT);
        periodicTaskService.scheduleAgentPeriodicTask(sendDUTTemperature);

    }

   /**
    *  init() method
    */
    @Override
    public void init() {

       /**
        *  Anonymous class to handle alert-clearing
        */
        ClearAlertHandler chillerClearAlertHandler = new ClearAlertHandler() {
           /**
            * Test whether alert can be cleared.
            *
            * @param Alert - the Alert being testsd
            * @param AlertState - state of Alert being tested
            *
            * @return ClearAlertCode
            */
            public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                 String alertId = alert.getAlertId();
                 if (alertId.equals(ChillerAlerts.TEMP_TIMEOUT.getId()) ||
                     alertId.equals(ChillerAlerts.FLOW_TIMEOUT.getId()) ) {
                     return ClearAlertCode.CLEAR_ALERT;
		 }
                 return ClearAlertCode.UNKNOWN_ALERT;
            }
	};

        // Set a property to define that this Agent is a chiller subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.CHILLER_TYPE, ChillerSubsystem.class.getCanonicalName());

        // Register alerts
        alertService.registerAlert(ChillerAlerts.TEMP_TIMEOUT.newAlert(),
                                   chillerClearAlertHandler);
        alertService.registerAlert(ChillerAlerts.FLOW_TIMEOUT.newAlert(),
                                   chillerClearAlertHandler);
        alertService.registerAlert(ChillerAlerts.LOST_LISTENED.newAlert());
        alertService.registerAlert(ChillerAlerts.MISSING_THERMAL.newAlert());

        //Register self as an AgentPresenceListener to choose when to start
        //listening to the thermal subsystem.
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);

        // Get task which updates the fluid FlowRate channel
        List<MonitorUpdateTask> listTasks =
	    mon.getMonitorUpdateTasksForDevice(devChiller);
	for (MonitorUpdateTask task : listTasks) {
            List<Channel> chanList = task.getAllChannels();
            for (Channel ch : chanList) {
                if (chanFlow==null && ch.getSubTypeStr().equals("FLOW_RATE")) {
                    chanFlow = ch;
                    taskFlow = task;
                } else if (chanFlowSet==null && ch.getSubTypeStr().equals("FLOW_SETPT")) {
                    chanFlowSet = ch;
                } else if (chanTempSet==null && ch.getSubTypeStr().equals("SET_POINT")) {
                    chanTempSet = ch;
                } else if (chanTankSet==null && ch.getSubTypeStr().equals("TANK_P_SET")) {
                    chanTankSet = ch;
                }
            }
        }
        if (taskFlow == null) {
            ErrorUtils.reportConfigError(LOG, name, "taskFlow", "no update task found with Channel subtype FLOW_RATE");
        }
        // Initial value for use in periodic task checkDataArrival
        timeLastData = CCSTimeStamp.currentTime();
    }
    
    /**
     * postStart (must follow Device.start())
     */
    @Override
    public void postStart() {
        // if (!devChiller.isOnline()) {
        //     throw new RuntimeException("InTESTChillerDevice not online");
        // }

        /* Start dummy threads, so Thread objects will not be null */
        checkTemp = new Thread("dummyT"){};
        checkFlow = new Thread("dummyF"){};
    }

    /*  shutdown method  */
    @Override
    public void shutdown() {
        // Unregister self from listening to AgentPresenceManager notifications.
        getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
	// If in DUT mode, that must be shut down.
        try {
            devChiller.shutdownDUT();
	} catch (DriverException ex) {
            throw new RuntimeException("DriverException in shutdownDUT() "+ex);
        }
    }

   /**
    *  Select refrig subsystem to listen to for average coldplate temperature
    *
    *  @param String value (case-insensitive)
    */
    @ConfigurationParameterChanger(propertyName = "listenTo")
    public void setListenTo(String value) {
        if (value.toLowerCase().equals("thermal")) {
            refrigAgentProperty = RefrigAgentProperties.THERMAL_TYPE;
            listenTo = value;
        } else if (value.toLowerCase().equals("pcp")) {
            refrigAgentProperty = RefrigAgentProperties.PCP_TYPE;
            listenTo = value;
	} else {
            ErrorUtils.reportConfigError(LOG, name, "listenTo", "Must specify either Thermal or Pcp subsystem");
        }
    }

   /**
    *  Veto transition to normal mode if Chiller state is not at setpoint,
    *  or if listening to thermal subsystem is required but not active.
    *
    *  @return  String:  null if not vetoed, explanation if vetoed
    */
    @Override
    public String vetoTransitionToNormalMode() {
        String reply = "";
        if (stateService.getState(ChillerState.class) != 
            ChillerState.SETPOINT) {
            reply += "Chiller not controlling at setpoint  ";
        }
        if (requireListening && !isListeningToThermal) {
            reply += "Chiller is not listening to Thermal";
        }
        return reply.equals("") ? null : reply;
    }

   /**
    *  Set chiller temperature and go there using default ramp
    *
    *  @param  double temperature in degrees
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING1,
             name = "setTemperature", autoAck = false,
             description = "go to temperature setting using default ramp")
    public void setTemperature(@Argument(description="tempeature in degrees")
                               double temperature) throws DriverException {
        helper()
        .precondition(devChiller.getLastErrorAlert() != AlertState.ALARM,
                      "operation blocked due to Chiller Error")
        .precondition(!setTempInProgress, "Previous set command in progress,"
                      + " quitControllingTemperature must be issued first")
        .action(() -> {
	    gotCommand = true;
	    double ramp =  Double.parseDouble(devChiller.readParameter(37));
            devChiller.setTemperature(temperature);
            setTempInProgress = true;
            lastRampValue = ramp;
            waitForTemp(temperature, ramp);
	});
    }

   /**
    *  Set chiller temperature and go there using provided ramp rate
    *
    *  @param  double temperature in degrees
    *  @param  double ramp in degrees/minute
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING1,
             name = "setTemperatureWithRamp", autoAck = false,
             description = "go to temperature setting using provided ramp")
    public void setTemperatureWithRamp(
        @Argument(description="tempeature in degrees") double temperature,
        @Argument(description="ramp in degrees/min") double ramp)
        throws DriverException {
        helper()
        .precondition(devChiller.getLastErrorAlert() != AlertState.ALARM,
                      "operation blocked due to Chiller Error")
        .precondition(!setTempInProgress, "Previous set command in progress,"
                      + " quitControllingTemperature must be issued first")
        .action(() -> {
	    gotCommand = true;
            devChiller.setTemperatureWithRamp(temperature, ramp);
            setTempInProgress = true;
            lastRampValue = ramp;
            waitForTemp(temperature, ramp);
	});
    }

   /**
    *  Wait for temperature to reach set-point or until timeout.
    *  Timeout value is estimated time to completion multiplied by a factor.
    *
    *  @param  double  Setpoint temperature (deg-C)
    *  @param  double  Ramp (deg-C/minute)
    *  @throws DriverException
    */
    private void waitForTemp(double temp, double ramp) throws DriverException {
        Double currentTemp = mon.getChannelValue("Chiller/FluidTemperature");
        // Estimated time in seconds to reach set-point
        long timeEst = Math.round(60.*Math.abs(currentTemp - temp)/(ramp + rampCorrection));
        LOG.info("Initiating temperature change to " + Double.toString(temp) +
                 " deg, estimated duration " + Double.toString(timeEst) +
                 " seconds.");
        checkTemp = new Thread("checkTemp") {
            @Override
            public void run() {
                long timeStart = System.currentTimeMillis();
                long timeout = Math.round(temperatureTimeoutFactor * 1000. * timeEst); //ms
                while (true) {
                    try {
                        Thread.sleep(intervalCheckTemp.toMillis());
                    } catch (InterruptedException ex) {
                        throw new RuntimeException("Unexpected interrupt while waiting for requested temperature",ex);
                    }
                    if (stateService.getState(ChillerState.class) ==
                        ChillerState.SETPOINT) {
                        setTempInProgress = false;
                        LOG.info("Chiller temperature is at requested setpoint");
                        break;
                    }
                    if (System.currentTimeMillis() - timeStart > timeout) {
                        alertService.raiseAlert(ChillerAlerts.TEMP_TIMEOUT.newAlert(), AlertState.ALARM,"Set temperature timed out at " + Long.toString(timeout/1000) + " s");
                        setTempInProgress = false;
                        break;
                    }
                }
            }
	};
        checkTemp.setDaemon(true);
        checkTemp.start();
    }

   /**
    *  Quit controlling temperature
    *
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING1,
             name="quitControllingTemperature",
             description="stop controlling temperature")
    public void quitControllingTemperature() throws DriverException {
        devChiller.quitControllingTemperature();
        setTempInProgress = false;  
    }

   /**
    *  Set flow rate
    *  Command is accepted even if pumps are off, still setting flow setpoint.
    *
    *  @param  double flow  Requested flow rate in gallons per minute
    *  @return String action taken
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING1,
             name = "setFlow",
             description = "set flow rate of chilled fluid")
    public String setFlow(@Argument(description="flow rate in gpm")
                           double flow) throws DriverException {
        gotCommand = true;
        devChiller.setFlow(flow);
        ChillerState state = (ChillerState) stateService.getState(ChillerState.class);
        if (state != ChillerState.SETPOINT &&
            state != ChillerState.CONTROLLING) {
            return "Chiller pumps are off, but flow setpoint has been set";
	} 

        /* Wait for flow-setting to complete, or for timeout.  */

        setFlowInProgress = true;
        checkFlow = new Thread("checkFlow") {
            @Override
            public void run() {
                long timeStart = System.currentTimeMillis();
                long timeout = setFlowTimeout.toMillis();
                // Speed up monitor data publication while setting flow.
                taskFlow.forceDataPublicationForDuration(setFlowTimeout);
		while (true) {
                    try {
                        Thread.sleep(intervalCheckFlow.toMillis());
                    } catch (InterruptedException ex) {
                        throw new RuntimeException("Unexpected interrupt while waiting for requested flow",ex);
                    }
                    if (Math.abs(chanFlow.getValue() - flow) < flowTolerance) {
                        LOG.info("Chiller flpw is at requested setpoint");
                        taskFlow.resetForcedDataPublication();
                        break;
                    }
                    if (System.currentTimeMillis() - timeStart > timeout) {
                        alertService.raiseAlert(ChillerAlerts.FLOW_TIMEOUT.newAlert(), AlertState.ALARM,"Set flow timed out at " + Long.toString(timeout/1000) + " s");
			break;
                    }
                }
                setFlowInProgress = false;
            }
	};
        checkFlow.setDaemon(true);
        checkFlow.start();
        return "Chilled fluid flow setting initiated";
    }

    /* Methods to listen to messages from specified subsystem and act on them */

    /**
    *  Command to set chiller tank pressure
    *
    *  @param  value Tank pressure setting (psig)
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING1, 
             name="setTankPressure",description="Set tank pressure in psig")
    public void setTankPressure(@Argument(description = "Tank pressure set value in psig") double value) throws DriverException {
        gotCommand = true;
        devChiller.setTankPressure(value);
    }

  /**
    *  If message is StatusSubsystemData from monitoring, search for
    *  Channel containing coldplate temperature data, and save data.
    *
    *  @parmm  StatusMessage
    */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData ssd = (StatusSubsystemData) msg;
        KeyValueData kvd = ssd.getSubsystemData();
        if ( kvd instanceof KeyValueDataList && 
             ssd.getDataKey().equals(MonitorUpdateTask.PUBLICATION_KEY) ) {
            KeyValueDataList kvdl = (KeyValueDataList) kvd;
            // Find requested Channel
            for (KeyValueData data : kvdl.getListOfKeyValueData()) {
                if (data.getKey().equals(coldplateChannelPath)) {
                    timeLastData = data.getCCSTimeStamp();
                    devChiller.updateColdplateTemp((double)data.getValue(), 
                                                   timeLastData);
                    break;
                }
            }
        }
    }

    @Override
    public void connected(AgentInfo... agents) {
        //If the thermal subsystem connects start listening to its
        //StatusSubsystemData messages.
        //Also check that we have only one thermal subsystem
        for ( AgentInfo agent : agents ) {
            if ( agent.hasAgentProperty(refrigAgentProperty) ) {
                synchronized(listeningLock) {
                    if ( !isListeningToThermal ) {
                        isListeningToThermal = true;
                        thermalName = agent.getName();
                        LOG.log(Level.INFO,"Starting to listen to messages from " + listenTo.toUpperCase() + " subsystem {0}.",thermalName);
                        Predicate<BusMessage<? extends Serializable, ?>> filter = 
                                BusMessageFilterFactory.messageOrigin(thermalName).and(BusMessageFilterFactory.messageClass(StatusSubsystemData.class));                        
                        getMessagingAccess().addStatusMessageListener(this,filter);
                        if (listenAlarm) {
                            alertService.raiseAlert(ChillerAlerts.LOST_LISTENED.newAlert(), AlertState.NOMINAL, "Subsystem " + thermalName + " reconnected");
                            listenAlarm = false;
                        }
                    } else {
                        //We should probably raise an exception here.
                        LOG.log(Level.SEVERE,"More than one thermal subsystem on the buses!!! Currently listening to {0} and just connected {1}!!", new Object[]{thermalName,agent.getName()});
                    }
                }                
            }
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            synchronized(listeningLock) {
                if ( isListeningToThermal && agent.getName().equals(thermalName) ) {
                    String msg = "Subsystem " + thermalName + "disconnected";
		    if (requireListening) {
                        alertService.raiseAlert(ChillerAlerts.LOST_LISTENED.newAlert(), AlertState.ALARM, msg);
                        listenAlarm = true;
                    }
                    LOG.log(Level.INFO,msg+", no longer listening to its messages");
                    getMessagingAccess().removeStatusMessageListener(this);
                    isListeningToThermal = false;
                    thermalName = null;
                } 
            }
        }        
    }    
    
    /**
     *  Gets the state of the chiller system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The thermal control state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the chiller state", level=0)
    public ChillerControlState getControlState()
    {
        controlState.setFastPeriod(monitorControl.getFastPeriod());
        return controlState;
    }    

    /**
     *  Updates the chiller control state periodically.
     */
    private void updateControlState()
    {
        boolean changed = monitorControl.hasPeriodChanged() || gotCommand;
        gotCommand = false;
        ChillerState state = devChiller.getChillerState();
        if (state != controlState.getChillerState()) {
            controlState.setChillerState(state);
            changed = true;
        }
        double flow = chanFlowSet.getValue();
        double oldFlow = controlState.getFlowSet();
        if ((!Double.isNaN(flow) || !Double.isNaN(oldFlow)) && flow !=oldFlow) {
            controlState.setFlowSet(flow);
            changed = true;
        }
        double temp = chanTempSet.getValue();
        double oldTemp = controlState.getSetPoint();
        if ((!Double.isNaN(temp) || !Double.isNaN(oldTemp)) && temp !=oldTemp) {
            controlState.setSetPoint(temp);
            changed = true;
        }
        double tank = chanTankSet.getValue();
        double oldTank = controlState.getTankSet();
        if ((!Double.isNaN(tank) || !Double.isNaN(oldTank)) && tank !=oldTank) {
            controlState.setTankSet(tank);
            changed = true;
        }
        double oldLastRamp = controlState.getLastRamp();
        if ((!Double.isNaN(lastRampValue) || !Double.isNaN(oldLastRamp)) && lastRampValue !=oldLastRamp) {
            controlState.setLastRamp(lastRampValue);
            changed = true;
        }
        double ramp = devChiller.getDefaultRamp();
        double oldRamp = controlState.getDefaultRamp();
        if ((!Double.isNaN(ramp) || !Double.isNaN(oldRamp)) && ramp !=oldRamp) {
            controlState.setDefaultRamp(ramp);
            changed = true;
        }
        String mode = devChiller.getTempControlMode() ? "DUT" : "RTD1";
        String oldMode = controlState.getTemperatureMode();
        if (!mode.equals(oldMode)) {
            controlState.setTemperatureMode(mode);
            changed = true;
        }
        if (changed) {
            publishControlState();
        }
    }

    /**
     *  Publishes the state of the chiller subsystem.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishControlState()
    {
        controlState.setFastPeriod(monitorControl.getFastPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(ChillerControlState.KEY, controlState));
    }    

}
