package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.refrig.constants.ChillerConditions;
import org.lsst.ccs.subsystem.refrig.constants.ChillerLatches;
import org.lsst.ccs.subsystem.refrig.constants.ChillerSwitches;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.data.ChillerPlcState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Simulates the compressor Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class SimChillerPlutoDevice extends ChillerPlutoDevice {

    /**
     *  Constants
     */
    private static final Map<String, Integer> conditionMap = new LinkedHashMap<>();
    static {
        conditionMap.put("Allowed", ChillerConditions.COND_ALLOWED);
        conditionMap.put("BdRetnNoError", ChillerConditions.COND_BD_RETN_NO_ERR);
        conditionMap.put("BdRetnNoWarn", ChillerConditions.COND_BD_RETN_NO_WRN);
        conditionMap.put("BdSuppNoError", ChillerConditions.COND_BD_SUPP_NO_ERR);
        conditionMap.put("BdSuppNoWarn", ChillerConditions.COND_BD_SUPP_NO_WRN);
        conditionMap.put("ChillerEmo", ChillerConditions.COND_CHILLER_EMO);
        conditionMap.put("ClpRefPerm", ChillerConditions.COND_CLP_REF_PERM);
        conditionMap.put("EmoReadback", ChillerConditions.COND_EMO_READBACK);
        conditionMap.put("KeySwitch", ChillerConditions.COND_KEY_SWITCH);
        conditionMap.put("MasterReset", ChillerConditions.COND_MASTER_RESET);
        conditionMap.put("Permit", ChillerConditions.COND_PERMIT);
        conditionMap.put("SmokeOkay", ChillerConditions.COND_SMOKE_OK);
    }

    /**
     *  Data fields.
     */
    private static final Logger LOG = Logger.getLogger(SimChillerPlutoDevice.class.getName());
    private final boolean[] switches = new boolean[ChillerSwitches.NUM_SWITCHES];
    private final LatchState[] latches = new LatchState[ChillerLatches.NUM_LATCHES];
    private final ConditionState[] conditions = new ConditionState[ChillerConditions.NUM_CONDITIONS];
    private boolean plcActive = true;


    /**
     *  Constructor.
     */
    public SimChillerPlutoDevice()
    {
        Arrays.fill(latches, LatchState.CLEAR);
        Arrays.fill(conditions, ConditionState.YES);
        conditions[ChillerConditions.COND_MASTER_RESET] = ConditionState.NO;
    }


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
        LOG.log(Level.INFO, "Connected to {0} (simulated Pluto gateway)", getPath());
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        switches[sw] = on;
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return switches[sw];
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    @Override
    public Boolean isLatchActive(int cond)
    {
        return latches[cond] == LatchState.ACTIVE;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    @Override
    public Boolean isLatchLatched(int cond)
    {
        return latches[cond] == LatchState.LATCHED;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    @Override
    public void clearLatch(int cond)
    {
        if (latches[cond] == LatchState.LATCHED) {
            latches[cond] = LatchState.CLEAR;
        }
    }


    /**
     *  Clears all latched conditions.
     */
    @Override
    public void clearAllLatches()
    {
        for (int cond = 0; cond < latches.length; cond++) {
            clearLatch(cond);
        }
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    @Override
    public Boolean isConditionActive(int cond)
    {
        return conditions[cond] == ConditionState.YES;
    }


    /**
     *  Checks whether PLC is active
     *
     *  @return  Whether active
     */
    @Override
    public Boolean isPlcActive()
    {
        return plcActive;
    }


    /**
     *  Gets the PLC error code.
     *
     *  @return  The error code
     */
    @Override
    public int getErrorCode()
    {
        return 42;
    }


    /**
     *  Updates the PLC state.
     *
     *  This is to  be called periodically to keep the GUI updated
     * 
     *  @param  plcState  The PLC state to be updated
     *  @return  Whether the state changed
     */
    @Override
    public boolean updateState(ChillerPlcState plcState)
    {
        conditions[ChillerConditions.COND_CLP_REF_PERM] = switches[ChillerSwitches.SW_ENABLE_CHILLER] ? ConditionState.YES : ConditionState.NO;
        conditions[ChillerConditions.COND_CHILLER_EMO]
          = (latches[ChillerLatches.LATCH_PERMIT] == LatchState.CLEAR && latches[ChillerLatches.LATCH_EXT_EMO] == LatchState.CLEAR
               && latches[ChillerLatches.LATCH_SMOKE_DETC] == LatchState.CLEAR && latches[ChillerLatches.LATCH_BD_RETURN_P] == LatchState.CLEAR
               && latches[ChillerLatches.LATCH_BD_SUPPLY_P] == LatchState.CLEAR && conditions[ChillerConditions.COND_CLP_REF_PERM] == ConditionState.YES
               && conditions[ChillerConditions.COND_KEY_SWITCH] == ConditionState.YES) ? ConditionState.YES : ConditionState.NO;
        return super.updateState(plcState);
    }


    @Command(type=Command.CommandType.QUERY, description="Get the list of latched condition names")
    public List<String> getLatchNames()
    {
        return ChillerLatches.getNames();
    }

    @Command(type=Command.CommandType.ACTION, description="Make a latched condition active")
    public void setLatchActive(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[ChillerLatches.getId(cond)] = LatchState.ACTIVE;
    }

    @Command(type=Command.CommandType.ACTION, description="Make a latched condition latched")
    public void setLatchLatched(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[ChillerLatches.getId(cond)] = LatchState.LATCHED;
    }

    @Command(type=Command.CommandType.ACTION, description="Put a latched condition into warning state")
    public void setLatchWarning(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[ChillerLatches.getId(cond)] = LatchState.WARNING;
    }

    @Command(type=Command.CommandType.QUERY, description="Get the list of condition names")
    public List<String> getConditionNames()
    {
        return new ArrayList<>(conditionMap.keySet());
    }

    @Command(type=Command.CommandType.ACTION, description="Set the state of a condition")
    public void setCondition(@Argument(description="Running condition name") String cond,
                             @Argument(description="Condition on state") boolean set) throws RefrigException {
        Integer condNum = conditionMap.get(cond);
        if (condNum == null) {
            throw new RefrigException("Unknown condition name: " + cond);
        }
        conditions[condNum] = set ? ConditionState.YES : ConditionState.NO;
    }

    @Command(type=Command.CommandType.ACTION, description="Set whether PLC is active")
    public void setPLCActive(@Argument(description="PLC active state") boolean set) {
        plcActive = set;
    }

}
