package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.subsystem.common.devices.pluto.PlutoDevice;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompLatches;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;

/**
 *  Handles a compressor control Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class CompPlutoDevice extends PlutoDevice implements Compressor.SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_ENABLE    = 0,
        SW_LIGHTS    = 1,
        NUM_COLD_SWITCHES = 2,
        SW_HEATER    = 2,
        NUM_CRYO_SWITCHES = 3;

    private static final int
        NUM_AREAS    = 15,
        SWDI_ON_BIT = 0,
        SWDI_READ_AREA = 1,
        SWDI_READ_BIT = 2,
        LTDI_AREA = 0,
        LTDI_READ_BIT = 1,
        LTDI_WARN_BIT = 2,
        LTDI_PEND_BIT = 3,
        CNDI_READ_AREA = 0,
        CNDI_READ_BIT = 1,
        RESET_BIT = 0;

    /**
     *  Private configuration data.
     */
    private static final int[][] COLD_SWITCHES = new int[NUM_COLD_SWITCHES][];
    static {
        COLD_SWITCHES[SW_ENABLE] = new int[]{1, 10, 23};   // ON/OFF bit, read area, read bit 
        COLD_SWITCHES[SW_LIGHTS] = new int[]{2, 14, 18};
    }
    private static final int[][] CRYO_SWITCHES = new int[NUM_CRYO_SWITCHES][];
    static {
        CRYO_SWITCHES[SW_ENABLE] = new int[]{1, 10, 23};   // ON/OFF bit, read area, read bit 
        CRYO_SWITCHES[SW_LIGHTS] = new int[]{2, 14, 18};
        CRYO_SWITCHES[SW_HEATER] = new int[]{3, 14, 26};
    }
    private static final int[][] COLD_LATCHES = new int[CompLatches.NUM_LATCHES][];
    static {
        COLD_LATCHES[CompLatches.LATCH_DISCHARGE_TEMP]  = new int[]{11, 17, 20, 19};  // Area, on bit, warn bit, ind bit
        COLD_LATCHES[CompLatches.LATCH_SUCTION_TEMP]    = new int[]{11, 22, 32, 24};
        COLD_LATCHES[CompLatches.LATCH_LIQUID_TEMP]     = new int[]{11, 27, 32, 29};
        COLD_LATCHES[CompLatches.LATCH_DISCHARGE_PRESS] = new int[]{12, 17, 20, 19};
        COLD_LATCHES[CompLatches.LATCH_POWER]           = new int[]{12, 27, 30, 29};
        COLD_LATCHES[CompLatches.LATCH_EXT_PERMIT]      = new int[]{13, 17, 32, 19};
        COLD_LATCHES[CompLatches.LATCH_SMOKE_DETC]      = new int[]{13, 22, 32, 24};
        COLD_LATCHES[CompLatches.LATCH_SENSORS_VALID]   = new int[]{13, 27, 32, 29};
        COLD_LATCHES[CompLatches.LATCH_OIL_LEVEL]       = new int[]{14, 27, 32, 29};
    }
    private static final int[][] CRYO_LATCHES = new int[CompLatches.NUM_LATCHES][];
    static {
        CRYO_LATCHES[CompLatches.LATCH_DISCHARGE_TEMP]  = new int[]{11, 17, 20, 19};  // Area, on bit, warn bit, ind bit
        CRYO_LATCHES[CompLatches.LATCH_SUCTION_TEMP]    = new int[]{11, 22, 32, 24};
        CRYO_LATCHES[CompLatches.LATCH_OIL_LEVEL]       = new int[]{11, 27, 32, 29};
        CRYO_LATCHES[CompLatches.LATCH_DISCHARGE_PRESS] = new int[]{12, 17, 20, 19};
        CRYO_LATCHES[CompLatches.LATCH_POWER]           = new int[]{12, 27, 30, 29};
        CRYO_LATCHES[CompLatches.LATCH_EXT_PERMIT]      = new int[]{13, 17, 32, 19};
        CRYO_LATCHES[CompLatches.LATCH_SMOKE_DETC]      = new int[]{13, 22, 32, 24};
        CRYO_LATCHES[CompLatches.LATCH_SENSORS_VALID]   = new int[]{12, 22, 32, 24};
        CRYO_LATCHES[CompLatches.LATCH_AFTER_COOLER]    = new int[]{13, 27, 32, 29};
    }
    private static final int[][] COLD_CONDITIONS = new int[CompConditions.NUM_CONDITIONS][];
    static {
        COLD_CONDITIONS[CompConditions.COND_CMP_ENABLED]      = new int[]{10, 24};  // Area, bit
        COLD_CONDITIONS[CompConditions.COND_CMP_WAITING]      = new int[]{10, 28};
        COLD_CONDITIONS[CompConditions.COND_CMP_POWERED]      = new int[]{10, 29};
        COLD_CONDITIONS[CompConditions.COND_DISC_PRESS_VALID] = new int[]{10, 19};
        COLD_CONDITIONS[CompConditions.COND_DISC_TEMP_VALID]  = new int[]{10, 16};
        COLD_CONDITIONS[CompConditions.COND_LIQD_TEMP_VALID]  = new int[]{10, 18};
        COLD_CONDITIONS[CompConditions.COND_SUCT_PRESS_VALID] = new int[]{10, 20};
        COLD_CONDITIONS[CompConditions.COND_SUCT_TEMP_VALID]  = new int[]{10, 17};
        COLD_CONDITIONS[CompConditions.COND_CURRENT_VALID]    = new int[]{10, 21};
        COLD_CONDITIONS[CompConditions.COND_VOLTAGE_VALID]    = new int[]{10, 22};
        COLD_CONDITIONS[CompConditions.COND_LATCHES_CLEAR]    = new int[]{10, 25};
        COLD_CONDITIONS[CompConditions.COND_KEYSWITCH_ON]     = new int[]{14, 19};
        COLD_CONDITIONS[CompConditions.COND_POWER_LED]        = new int[]{14, 23};
        COLD_CONDITIONS[CompConditions.COND_CURR_SENSOR_ERR]  = new int[]{13, 30};
    }
    private static final int[][] CRYO_CONDITIONS = new int[CompConditions.NUM_CONDITIONS][];
    static {
        CRYO_CONDITIONS[CompConditions.COND_CMP_ENABLED]      = new int[]{10, 24};  // Area, bit
        CRYO_CONDITIONS[CompConditions.COND_CMP_WAITING]      = new int[]{10, 28};
        CRYO_CONDITIONS[CompConditions.COND_CMP_POWERED]      = new int[]{10, 29};
        CRYO_CONDITIONS[CompConditions.COND_DISC_PRESS_VALID] = new int[]{10, 19};
        CRYO_CONDITIONS[CompConditions.COND_DISC_TEMP_VALID]  = new int[]{10, 16};
        CRYO_CONDITIONS[CompConditions.COND_OIL_LEVEL_VALID]  = new int[]{10, 18};
        CRYO_CONDITIONS[CompConditions.COND_SUCT_PRESS_VALID] = new int[]{10, 20};
        CRYO_CONDITIONS[CompConditions.COND_SUCT_TEMP_VALID]  = new int[]{10, 17};
        CRYO_CONDITIONS[CompConditions.COND_CURRENT_VALID]    = new int[]{10, 21};
        CRYO_CONDITIONS[CompConditions.COND_VOLTAGE_VALID]    = new int[]{10, 22};
        CRYO_CONDITIONS[CompConditions.COND_LATCHES_CLEAR]    = new int[]{10, 25};
        CRYO_CONDITIONS[CompConditions.COND_KEYSWITCH_ON]     = new int[]{14, 19};
        CRYO_CONDITIONS[CompConditions.COND_POWER_LED]        = new int[]{14, 23};
        CRYO_CONDITIONS[CompConditions.COND_CURR_SENSOR_ERR]  = new int[]{12, 25};
        CRYO_CONDITIONS[CompConditions.COND_CMP_ON_8HRS]      = new int[]{12, 16};
        CRYO_CONDITIONS[CompConditions.COND_HEATER_ENABLED]   = new int[]{14, 27};
        CRYO_CONDITIONS[CompConditions.COND_HEATER_POWERED]   = new int[]{14, 29};
    }

    /**
     *  Data fields.
     */
    private int[][] switches, latches, conditions;


    /**
     *   Constructor.
     */
    public CompPlutoDevice()
    {
        super(NUM_AREAS);
    }


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    public void setType(int type)
    {
        switches = type == CompTypes.TYPE_COLD ? COLD_SWITCHES : CRYO_SWITCHES;
        latches = type == CompTypes.TYPE_COLD ? COLD_LATCHES : CRYO_LATCHES;
        conditions = type == CompTypes.TYPE_COLD ? COLD_CONDITIONS : CRYO_CONDITIONS;
    }


    /**
     *  Sets a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        int bitNum = switches[sw][SWDI_ON_BIT];
        writeBit(bitNum / 16, bitNum & 0x0f, on ? 1 : 0);
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        int[] swData = switches[sw];
        Integer value = readAddBit(swData[SWDI_READ_AREA], swData[SWDI_READ_BIT]);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets the state of a latched condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    public LatchState getLatchState(int cond)
    {
        int[] latchData = latches[cond];
        Integer value = readAddInt(latchData[LTDI_AREA]);
        return value == null ? LatchState.OFFLINE :
               (value & (1 << latchData[LTDI_PEND_BIT])) != 0 ? LatchState.LATCHED :
               (value & (1 << latchData[LTDI_READ_BIT])) == 0 ? LatchState.ACTIVE :
               (value & (1 << latchData[LTDI_WARN_BIT])) != 0 ? LatchState.WARNING : LatchState.CLEAR;
    }


    /**
     *  Gets the state of a condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    public ConditionState getConditionState(int cond)
    {
        int[] condData = conditions[cond];
        Integer value = readAddBit(condData[CNDI_READ_AREA], condData[CNDI_READ_BIT]);
        return value == null ? ConditionState.OFF : value != 0 ? ConditionState.YES : ConditionState.NO;
    }


    /**
     *  Resets all latched conditions.
     */
    public void resetLatches()
    {
        writeBit(RESET_BIT / 16, RESET_BIT & 0x0f, 1);
        try {
            Thread.sleep(500);
        }
        catch (InterruptedException e) {}
        writeBit(RESET_BIT / 16, RESET_BIT & 0x0f, 0);
    }


    /**
     *  Gets the PLC error code.
     * 
     *  @return  The error code
     */
    public int getErrorCode()
    {
        return readAddWord(11, 0);
    }

}
