package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.BypassValveState;
import org.lsst.ccs.subsystem.refrig.constants.CompAlert;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.CoolantValveState;
import org.lsst.ccs.subsystem.refrig.constants.OilSepHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.OrificeValveState;
import org.lsst.ccs.subsystem.refrig.constants.SurgeHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CryoCompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Controls a refrigeration cryo compressor.
 *
 *  @author Owen Saxton
 */
public class CryoCompressor extends Compressor {

    @LookupName
    private String name;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private CryoCompLimits compLimits;    // Created by RefrigMain
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private CompMaq20PWMControl pwmControl;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private AgentStateService stateService;

    private String suctPressChan, phaseSepTempChan, oilLevelChan;  // From Groovy file

    private static final Logger LOG = Logger.getLogger(CryoCompressor.class.getName());
    private double dryerHyst = 1.0;
    private Channel suctPress;
    private boolean checkDryer = false;


    /**
     *  Constructor.
     */
    public CryoCompressor() {
        super(new CryoCompState());
    }
            

    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Register valve & heater states
        stateService.registerState(OrificeValveState.class, "Orifice valve state", this);
        stateService.updateAgentComponentState(this, OrificeValveState.OFFLINE);
        stateService.registerState(BypassValveState.class, "Bypass valve state", this);
        stateService.updateAgentComponentState(this, BypassValveState.OFFLINE);
        stateService.registerState(CoolantValveState.class, "Coolant valve state", this);
        stateService.updateAgentComponentState(this, CoolantValveState.OFFLINE);
        stateService.registerState(SurgeHeaterState.class, "Surge tank heater state", this);
        stateService.updateAgentComponentState(this, SurgeHeaterState.OFFLINE);
        stateService.registerState(OilSepHeaterState.class, "Oil separator heater state", this);
        stateService.updateAgentComponentState(this, OilSepHeaterState.OFFLINE);
    }


    /**
     *  Registers the possible alerts.
     */
    @Override
    public void init()
    {
        super.init();
        alertService.registerAlert(CompAlert.CHECK_FILTER_DRYERS.newAlert(name));                        
    }


    /**
     *  Initializes the cryo compressor control.
     */
    @Override
    public void postInit()
    {
        super.postInit();
        Channel phaseSepTemp = (phaseSepTempChan != null) ? channelMap.get(phaseSepTempChan) : null;
        if (phaseSepTemp == null) {
            ErrorUtils.reportConfigError(LOG, name, "phaseSepTempChan", "not specified or not defined");
        }
        condData[CompSwConds.SWC_PHASE_SEP_TEMP].channel = phaseSepTemp;
        suctPress = (suctPressChan != null) ? channelMap.get(suctPressChan) : null;
        if (suctPress == null) {
            ErrorUtils.reportConfigError(LOG, name, "suctPressChan", "not specified or not defined");
        }
        condData[CompSwConds.SWC_DISC_TEMP_LOW].channel = discTemp;
        condData[CompSwConds.SWC_DISC_TEMP_LOW].noShutoff = true;
        condData[CompSwConds.SWC_PRESS_DIFF].channel = discPress;
        condData[CompSwConds.SWC_PRESS_DIFF].channel2 = suctPress;
        condData[CompSwConds.SWC_PRESS_DIFF].noShutoff = true;
        Channel oilLevel = (oilLevelChan != null) ? channelMap.get(oilLevelChan) : null;
        if (oilLevel == null) {
            ErrorUtils.reportConfigError(LOG, name, "oilLevelChan", "not specified or not defined");
        }
        condData[CompSwConds.SWC_OIL_LEVEL].channel = oilLevel;
        condData[CompSwConds.SWC_CHECK_DRYERS].isWarning = true;
        condData[CompSwConds.SWC_CHECK_DRYERS].immedCond = false;
        condData[CompSwConds.SWC_CHECK_DRYERS].noShutoff = true;

        switchDevices[CompSwitches.SW_HEATER] = plutoDevc;
        switchChannels[CompSwitches.SW_HEATER] = CompPlutoDevice.SW_HEATER;
        switchDevices[CompSwitches.SW_ORIFICE_VALVE] = maq20Devc;
        switchChannels[CompSwitches.SW_ORIFICE_VALVE] = CompMaq20Device.SW_ORIFICE_VALVE;
        switchDevices[CompSwitches.SW_COOLANT_VALVE] = maq20Devc;
        switchChannels[CompSwitches.SW_COOLANT_VALVE] = CompMaq20Device.SW_COOLANT_VALVE;
        switchDevices[CompSwitches.SW_BYPASS_VALVE] = maq20Devc;
        switchChannels[CompSwitches.SW_BYPASS_VALVE] = CompMaq20Device.SW_BYPASS_VALVE;
        switchDevices[CompSwitches.SW_SURGE_HEATER] = maq20Devc;
        switchChannels[CompSwitches.SW_SURGE_HEATER] = CompMaq20Device.SW_SURGE_HEATER;
        switchDevices[CompSwitches.SW_CABINET_LIGHT] = maq20Devc;
        switchChannels[CompSwitches.SW_CABINET_LIGHT] = CompMaq20Device.SW_CABINET_LIGHT;

        if (compLimits == null) {
            ErrorUtils.reportConfigError(LOG, name, "CryoCompLimits", "not defined");
        }
        limitData = compLimits.getLimitData();
    }


    /**
     *  Updates the compressor state.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @return  Whether any value changed
     */
    @Override
    protected synchronized boolean updateState()
    {
        checkFilterDryers();
        boolean changed = super.updateState();
        if (changed) {
            SwitchState swState = state.getSwitchState(CompSwitches.SW_ORIFICE_VALVE);
            OrificeValveState ovState = swState == SwitchState.ON ? OrificeValveState.OPEN :
                                        swState == SwitchState.OFF ? OrificeValveState.SHUT : OrificeValveState.OFFLINE;
            swState = state.getSwitchState(CompSwitches.SW_BYPASS_VALVE);
            BypassValveState bvState = swState == SwitchState.ON ? BypassValveState.OPEN :
                                       swState == SwitchState.OFF ? BypassValveState.SHUT : BypassValveState.OFFLINE;
            swState = state.getSwitchState(CompSwitches.SW_COOLANT_VALVE);
            CoolantValveState cvState = swState == SwitchState.ON ? CoolantValveState.SHUT :
                                        swState == SwitchState.OFF ? CoolantValveState.OPEN : CoolantValveState.OFFLINE;
            swState = state.getSwitchState(CompSwitches.SW_HEATER);
            OilSepHeaterState oshState = swState == SwitchState.ON ? OilSepHeaterState.ON :
                                         swState == SwitchState.OFF ? OilSepHeaterState.OFF : OilSepHeaterState.OFFLINE;
            swState = state.getSwitchState(CompSwitches.SW_SURGE_HEATER);
            SurgeHeaterState shState = swState == SwitchState.ON ? SurgeHeaterState.ON :
                                       swState == SwitchState.OFF ? SurgeHeaterState.OFF : SurgeHeaterState.OFFLINE;
            stateService.updateAgentComponentState(this, ovState, bvState, cvState, oshState, shState);
        }
        boolean compRunning = state.getCompressorState() == CompressorState.RUNNING || state.getCompressorState() == CompressorState.RUN_WRN;
        controlValves(compRunning, stateChanged);
        controlFan(compRunning, stateChanged);
        controlHeater();
        return changed;
    }


    /**
     *  Controls the cryo valves.
     * 
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @param  compOn   Whether compressor is on
     *  @param  changed  Whether compressor state has changed
     */
    private void controlValves(boolean compOn, boolean changed)
    {
        try {
            if (compOn) {
                double press = condData[CompSwConds.SWC_DISC_PRESS].channel.getValue();
                if (press > limitData[CompSwConds.SWC_ORF_OFF_PRESS].immedLimit) {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, false);
                }
                else if (press < limitData[CompSwConds.SWC_ORF_ON_PRESS].immedLimit) {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, true);
                }
            }
            if (changed) {
                if (compOn) {
                    setSwitch(CompSwitches.SW_BYPASS_VALVE, false);
                }
                else {
                    setSwitch(CompSwitches.SW_ORIFICE_VALVE, false);
                }
                setSwitch(CompSwitches.SW_COOLANT_VALVE, !compOn);  // Closed when activated: open iff compressor running
            }
        }
        catch (RefrigException e) {
            LOG.log(Level.SEVERE, "Error operating {0} compressor valves: {1}", new Object[]{name, e});
        }
    }


    /**
     *  Controls the cryo heater.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     */
    private void controlHeater()
    {
        double temp = discTemp.getValue();
        if (temp > limitData[CompSwConds.SWC_HEATER_TEMP].immedLimit && state.getSwitchState(CompSwitches.SW_HEATER) == SwitchState.ON) {
            LOG.log(Level.WARNING, "Turning {0} compressor heater off - discharge temperature ({1}) > {2}",
                    new Object[]{name, temp, limitData[CompSwConds.SWC_HEATER_TEMP].immedLimit});
            try {
                setSwitch(CompSwitches.SW_HEATER, false);
            }
            catch (RefrigException e) {
                LOG.log(Level.SEVERE, "Error operating {0} compressor heater: {1}", new Object[]{name, e});
            }
        }
    }


    /**
     *  Controls the cryo fan.
     *
     *  @param  on  Whether compressor is on
     */
    private void controlFan(boolean compOn, boolean changed)
    {
        if (changed && pwmControl != null) {
            if (compOn) {
                pwmControl.enable();
            }
            else {
                pwmControl.disable();
            }
        }
    }


    /**
     *  Checks whether filter dryers need replacing.
     */
    private void checkFilterDryers()
    {
        Boolean check = null;
        double highPress = discPress.getValue(), lowPress = suctPress.getValue();
        if (highPress >= limitData[CompSwConds.SWC_DRYER_HI_PRESS].immedLimit
              && lowPress <= limitData[CompSwConds.SWC_DRYER_LO_PRESS].immedLimit
              && state.getConditionState(CompConditions.COND_CMP_ON_8HRS) == ConditionState.YES) {
            check = true;
        }
        else if (highPress < limitData[CompSwConds.SWC_DRYER_HI_PRESS].immedLimit - dryerHyst
                   || lowPress > limitData[CompSwConds.SWC_DRYER_LO_PRESS].immedLimit + dryerHyst) {
            check = false;
        }
        if (check != null && check != checkDryer) {
            checkDryer = check;
            if (check) {
                raiseAlert(CompAlert.CHECK_FILTER_DRYERS, AlertState.WARNING, name + " filter dryers need checking");
                condData[CompSwConds.SWC_CHECK_DRYERS].newImmedCond = true;
            }
            else {
                lowerAlert(CompAlert.CHECK_FILTER_DRYERS, name + " filter dryers no longer need checking");
                condData[CompSwConds.SWC_CHECK_DRYERS].newImmedCond = false;
            }
        }
    }

}
