package org.lsst.ccs.subsystem.refrig;

import java.util.Arrays;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.devices.pluto.PlutoDevice;
import org.lsst.ccs.subsystem.refrig.constants.PcpConditions;
import org.lsst.ccs.subsystem.refrig.constants.PcpLatches;
import org.lsst.ccs.subsystem.refrig.constants.PcpLimits;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitchState;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitches;

/**
 *  Handles the protection system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class PcpPlutoDevice extends PlutoDevice {

    /**
     *  Constants.
     */
    private static final int
        NUM_AREAS = 30,
        SWDI_ON_BIT = 0,
        SWDI_OFF_BIT = 1,
        SWDI_READ_AREA = 2,
        SWDI_READ_BIT = 3,
        LTDI_RESET_BIT = 0,
        LTDI_READ_AREA = 1,
        LTDI_READ_BIT = 2,
        LTDI_PEND_AREA = 3,
        LTDI_PEND_BIT = 4,
        CNDI_READ_AREA = 0,
        CNDI_READ_BIT = 1;
    private static final int
        PLC_STATUS_MASK = 0x01,
        ERROR_AREA = 6;
    private static final int
        OFFSET_CELSIUS = -273;

    /**
     *  Private lookup maps, etc.
     */
    private static final int[][] switches = new int[PcpSwitches.NUM_SWITCHES_ALL][];
    static {                              // ON bit, OFF bit, read area, read bit 
        switches[PcpSwitches.SW_BLOCK_COLD_HEAT] = new int[]{2, 3, 6, 6};
        switches[PcpSwitches.SW_BLOCK_COLD_REFG] = new int[]{4, 5, 7, 6};
        switches[PcpSwitches.SW_ENABLE_HEAT_PS1] = new int[]{7, 8, 0, 0};
        switches[PcpSwitches.SW_ENABLE_HEAT_PS2] = new int[]{9, 10, 0, 0};
        switches[PcpSwitches.SW_ENABLE_HEAT_PS3] = new int[]{11, 12, 0, 0};
    }
    private static final int[][] latches = new int[PcpLatches.NUM_LATCHES][];
    static {                                        // Reset bit, on area, on bit, ind area, ind bit
        latches[PcpLatches.LATCH_SMOKE_DETC]     = new int[]{6, 8, 2, 8, 9};
        latches[PcpLatches.LATCH_COLD_TEMP_HIGH] = new int[]{0, 6, 2, 6, 4};
        latches[PcpLatches.LATCH_COLD_TEMP_LOW]  = new int[]{1, 7, 2, 7, 4};
    }
    private static final int[][] conditions = new int[PcpConditions.NUM_CONDITIONS][];
    static {                                                // Area, bit
        conditions[PcpConditions.COND_COLD_REFG] = new int[]{7, 9};
        conditions[PcpConditions.COND_HEAT_PS1]  = new int[]{6, 12};
        conditions[PcpConditions.COND_HEAT_PS2]  = new int[]{6, 13};
        conditions[PcpConditions.COND_HEAT_PS3]  = new int[]{6, 14};
    }

    /**
     *  Data fields.
     */
    private final int[] tempLimits = new int[PcpLimits.NUM_LIMITS];
    private boolean limitsRead = false;
    private final boolean[] swIsAlarm = new boolean[PcpSwitches.NUM_SWITCHES_ALL];


    /**
     *   Constructor.
     */
    public PcpPlutoDevice()
    {
        super(NUM_AREAS);
        Arrays.fill(tempLimits, Integer.MAX_VALUE);
    }


    /**
     *  Tests whether the PLC is active
     *
     *  @return  Whether the PLC is active, or null if offline
     */
    public Boolean isPlcActive()
    {
        try {
            int status = plu.readModuleStatus();
            return (status & PLC_STATUS_MASK) != 0;
        }
        catch (DriverException e) {
            return null;
        }
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the protection Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    protected void setSwitchOn(int sw, boolean on)
    {
        int bitNum = switches[sw][on ? SWDI_ON_BIT : SWDI_OFF_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Sets a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     *  @param  alarm  Whether being called on behalf of remote alarm
     */
    public void setSwitchOn(int sw, boolean on, boolean alarm)
    {
        if (alarm) {
            swIsAlarm[sw] = on;
            if (!on) return;
        }
        else {
            if (!on && swIsAlarm[sw]) return;
        }
        setSwitchOn(sw, on);
    }


    /**
     *  Gets the state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  The switch state
     */
    public PcpSwitchState getSwitchState(int sw)
    {
        Boolean isOn = isSwitchOn(sw);
        return isOn != null ? isOn ? swIsAlarm[sw] ? PcpSwitchState.ALARM : PcpSwitchState.ON
                         : PcpSwitchState.OFF : PcpSwitchState.OFFLINE;
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The on/off state is not the state of the bit that was toggled, but is read back
     *  directly from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    protected Boolean isSwitchOn(int sw)
    {
        int[] swData = switches[sw];
        Integer value = readAddBit(swData[SWDI_READ_AREA], swData[SWDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    public Boolean isLatchActive(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_READ_AREA], condData[LTDI_READ_BIT] + 16);
        return value != null ? value == 0 : null;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    public Boolean isLatchLatched(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_PEND_AREA], condData[LTDI_PEND_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    public void clearLatch(int cond)
    {
        int bitNum = latches[cond][LTDI_RESET_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    public Boolean isConditionActive(int cond)
    {
        int[] condData = conditions[cond];
        Integer value = readAddBit(condData[CNDI_READ_AREA], condData[CNDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets the temperature limits.
     *
     *  @return  The 4-element array of temperature limits
     */
    public int[] getTempLimits()
    {
        if (!limitsRead && isOnline()) {
            tempLimits[PcpLimits.LIMIT_COLD_HIGH] = readAddWord(4, 0) + OFFSET_CELSIUS;
            tempLimits[PcpLimits.LIMIT_COLD_LOW] = readAddWord(4, 1) + OFFSET_CELSIUS;
            limitsRead = true;
        }
        return tempLimits;
    }


    public int getErrorCode()
    {
        return readAddWord(ERROR_AREA, 0);
    }

}
