package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.constants.ChillerAlerts;
import org.lsst.ccs.subsystem.refrig.constants.ChillerState;

/**
 *  Listen for those alerts from chiller subsystem which should block
 *  chiller flow when raised at the alarm level; upon detection, if
 *  chiller is in operation (with coolant flow) shut off flow.
 */

public class ChillerBlockingListener implements AlertListener, HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private ChillerSubsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupName
    private String name;

    private static final Logger LOG = Logger.getLogger(ChillerSubsystem.class.getName());

    private List<Alert> alerts = new ArrayList<>();

   /**
    *  Init phase.
    */
    @Override
    public void init()
    {
        for (ChillerAlerts ca : subsys.blockingAlerts) {
            alerts.add(ca.newAlert());
        }

        LOG.log(Level.INFO, name + " listening to Alerts from " + subsys.getName());
    }

   /**
    *  Start phase.
    */
    @Override
    public void start()
    {
         alertService.addListener(this);
    }

   /**
    *  Shutdown phase.
    */
    @Override
    public void shutdown()
    {
        //Remove alert listener
        alertService.removeListener(this);
    }

   /**
    *  Handler for received AlertEvent
    *
    *  If one of blocking Alerts is raised at Alarm level, and Chiller
    *  is in a controlling state (with fluid flow), quit control and flow.
    *
    *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event) {
        if (event.getType() == AlertEvent.AlertEventType.ALERT_RAISED) {
            if (alerts.contains(event.getAlert()) &&
                event.getLevel() == AlertState.ALARM) {
                LOG.fine(name + " received " + event.getAlert().toString());
		ChillerState cs = (ChillerState) stateService.getState(ChillerState.class);
                if (cs == ChillerState.SETPOINT || cs == ChillerState.CONTROLLING) {
                    LOG.info("Quitting control due to " + event.getAlert().toString());
                    try {
                        subsys.quitControllingTemperature();
                    }
                    catch (Exception e) {
                        LOG.severe("Exception while trying to quit: " + e);
                    }
                }
	    }
	}
    }
}
