package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.agilent.DL34970;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles an Agilent 34970A data logger.
 *
 *  @author Owen Saxton
 */
public class DL34970Device extends Device {

    /**
     *  Constants.
     */
    private static final int BAUD_RATE = 9600;
    private static final String tcTypes = "JKTERSBN";

    /**
     *  Private lookup maps.
     */
    private static final Map<String, DL34970.ConnType> cTypeMap = new HashMap<>();
    static {
        cTypeMap.put("SERIAL", DL34970.ConnType.SERIAL);
    }

    private final static Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   Channel.TYPE_TEMP);
        typeMap.put("VOLT",   Channel.TYPE_VOLTS);
    }

    private final static Map<String, DL34970.TcType> tcMap = new HashMap<>();
    static {
        tcMap.put("J", DL34970.TcType.J);
        tcMap.put("K", DL34970.TcType.K);
        tcMap.put("T", DL34970.TcType.T);
        tcMap.put("E", DL34970.TcType.E);
        tcMap.put("R", DL34970.TcType.R);
        tcMap.put("S", DL34970.TcType.S);
        tcMap.put("B", DL34970.TcType.B);
        tcMap.put("N", DL34970.TcType.N);
    }

    /**
     *  Data fields.
     */
    private String connType;            // Connection type string
    private String devcId;              // Device parameter
    private int baudRate = BAUD_RATE;   // Baud rate

    private static final Logger LOG = Logger.getLogger(DL34970Device.class.getName());
    private DL34970.ConnType connTypeC;  // Connection type
    private final DL34970 dl = new DL34970();  // Associated DL34970 object
    private final Set<Integer> hwChanSet = new TreeSet<>();
                                        // Sorted set of channel numbers
    private final Map<Integer, Integer> hwChanMap = new HashMap<>();
                                        // Map of channel number to index in hwChanSet
    private double chanValues[];        // Read channel data
    private boolean initError = false;  // True if last initialize attempt failed


    /**
     *  Constructor.
     *
     *  @param  connType  The device type (serial)
     *  @param  devcId    The device parameter (device name)
     *  @param  baudRate  The baud rate (0 for default)
     */
    public DL34970Device(String connType, String devcId, int baudRate)
    {
        this.connType = connType;
        this.devcId = devcId;
        this.baudRate = (baudRate == 0) ? BAUD_RATE : baudRate;
    }


    public DL34970Device() {
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (connType == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "connType", "is missing");
        }
        connTypeC = cTypeMap.get(connType);
        if (connTypeC == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "connType", "is invalid: " + connType);
        }
        if (devcId == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "devcId", "is missing");
        }
        fullName = "DL34970 Logger (" + devcId + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            dl.open(connTypeC, devcId, baudRate);
            initSensors();
            if (!hwChanSet.isEmpty()) {
                int index = 0;
                int[] chans = new int[hwChanSet.size()];
                for (int chan : hwChanSet) {
                    hwChanMap.put(chan, index);
                    chans[index++] = chan;
                }
                dl.setScanList(chans);
            }
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException | ChannelInitializationException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                dl.close();
            }
            catch (DriverException ce) {
                // Can happen if open failed
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            dl.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  This is called before the device has been initialized.
     *
     *  @param  ch  The Channel to check
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        String type = ch.getTypeStr();
        Integer iType = typeMap.get(type.toUpperCase());
        int iSubtype = 0;
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "type", type);
        }
        String subtype = ch.getSubTypeStr();
        if (iType == Channel.TYPE_TEMP) {
            iSubtype = subtype.length() == 1 ? tcTypes.indexOf(subtype) : -1;
            if (iSubtype < 0) {
                ErrorUtils.reportChannelError(LOG, ch.getPath(), "subtype", subtype);
            }
        }
        else if (iType == Channel.TYPE_VOLTS) {
            try {
                double range = Math.abs(Double.valueOf(subtype));
                double exp = Math.floor(Math.log10(range));
                double fract = range / Math.pow(10, exp);
                iSubtype = ((int)(exp - 3) << 16) | (int)(1000 * fract);
            }
            catch (NumberFormatException e) {
                ErrorUtils.reportChannelError(LOG, ch.getPath(), "subtype", subtype);
            }
        }
        hwChanSet.add(ch.getHwChan());

        return new int[]{iType, iSubtype};
    }


    /**
     *  Initializes a channel.
     *
     *  @param  ch  The Channel to initialize
     */
    @Override
    protected void initChannel(Channel ch)
    {
        try {
            int type = ch.getType();
            int subtype = ch.getSubType();
            int chan = ch.getHwChan();
            if (type == Channel.TYPE_TEMP) {
                dl.configTC(tcMap.get(tcTypes.substring(subtype, subtype + 1)), new int[]{chan});
            }
            else if (type == Channel.TYPE_VOLTS) {
                dl.configVolts(Math.pow(10.0, subtype >> 16) * (subtype & 0xffff), new int[]{chan});
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error configuring {0}: {1}", new Object[]{fullName, e});
            throw new ChannelInitializationException(e);
        }
    }

    private class ChannelInitializationException extends RuntimeException {
        
        public ChannelInitializationException(DriverException ex) {
            super(ex);
        }
    }


    /**
     *  Gets the group for a channel.
     *
     *  @param  ch  The channel
     *  @return  The group name, or null if not in a group
     */
    @Override
    protected String getGroupForChannel(Channel ch) {
        if (hwChanSet.contains(ch.getHwChan())) {
            return "data";
        }
        return null;
    }


    /**
     *  Reads Channels for the provided group.
     * 
     *  @param group The group to read.
     */
    @Override
    protected void readChannelGroup(String group)
    {
        if (!isOnline()) return;
        try {
            if (!hwChanSet.isEmpty()) {
                chanValues = dl.readData();
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading data logger: {0}", e);
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  The channels have been read in ascending channel number order (this
     *  is a feature of the data logger), so the map is used to locate where
     *  in the data array a particular channel is located.
     *
     *  @param  ch  The Channel to read.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(Channel ch)
    {
        if (isOnline()) {
            try {
                return chanValues[hwChanMap.get(ch.getHwChan())];
            }
            catch (Exception e) {  // Not enough data was read
                LOG.log(Level.SEVERE, "Error reading data logger: {0}", e);
                setOnline(false);
            }
        }
        return Double.NaN;
    }

}
