package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.io.Serializable;
import java.time.Duration;
import java.util.function.Predicate;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.data.RunMode;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.chiller.Chiller.ErrorWords;
import org.lsst.ccs.drivers.chiller.Chiller.FParam;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Monitor;
import org.lsst.ccs.monitor.MonitorUpdateTask;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.HeatTransferChannel;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.refrig.constants.ChillerAlerts;
import org.lsst.ccs.subsystem.refrig.constants.ChillerLatches;
import org.lsst.ccs.subsystem.refrig.constants.ChillerSwitches;
import org.lsst.ccs.subsystem.refrig.constants.ChillerState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.ChillerControlState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.lsst.ccs.subsystem.refrig.data.RefrigUtils;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/* Pre-defined command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ROUTINE;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ADVANCED;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_EXPERT;

/**
 *  Implements the chiller subsystem.
 *
 */
public class ChillerSubsystem extends Subsystem implements HasLifecycle, StatusMessageListener, AgentPresenceListener {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private InTESTChillerDevice devChiller;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private ChillerPlutoDevice devPluto;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private ChillerMaq20Device devMaq20;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private List<Channel> chanList = new ArrayList<>();
    @LookupField(strategy=LookupField.Strategy.TREE)
    protected Monitor mon;; 
    @LookupField(strategy=LookupField.Strategy.TREE)
    private List<HeatTransferChannel> heatXferList = new ArrayList<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private FanControl fanControl;
    @LookupName
    private String name;

    @ConfigurationParameter(isFinal = true, description = "whether this is the chiller providing fluid to the camera", units = "unitless")
    protected volatile boolean connectedToCamera;

    @ConfigurationParameter(isFinal = true, description = "Which refrig subsystem to listen to (RefrigAgentProperties)", units = "unitless")
    protected volatile String listenTo;

    @ConfigurationParameter(isFinal = true, description = "path of coldplate temperature Channel to listen for", units = "unitless")
    protected volatile String coldplateChannelPath;

    @ConfigurationParameter(isFinal = true, description = "path of cold inlet temperature Channel to listen for", units = "unitless")
    protected volatile String coldInletChannelPath;

    @ConfigurationParameter(isFinal = true, description = "path of cold outlet temperature Channel to listen for", units = "unitless")
    protected volatile String coldOutletChannelPath;

    // Factor to lengthen temperature-set timeout beyond estimated value
    @ConfigurationParameter(description="Factor on temperature-setting timeout",
                            range = "1.0..1.4", units = "unitless")
    protected volatile double temperatureTimeoutFactor;

    // @ConfigurationParameter(desvription = "timeout for setting flow",
    //                         units = "s")
    // protected volatile double flowTimeout;

    @ConfigurationParameter(isFinal = true, description = "cooling by glycol (true)/water (false)", units = "unitless")
    protected volatile boolean glycolCooling;

    @ConfigurationParameter(isFinal = true, description = "Burst-disc presssure to stop flow", units = "psig")
    protected volatile double maxPressure;;

    private static final Logger LOG = Logger.getLogger(ChillerSubsystem.class.getName());

    private final static int emoMask = 0x20; // emergency switch bit, ErrorWord2
    // Additive correction from specified ramp value to actual ramp
    private static final double rampCorrection = -0.04;  // deg-C/minute
    // Heat-transfer factors: density * specific heat * (min/s) from Brian Qiu
    // Density is in kg/gallon
    private static final double waterCoolFactor =  3.7854 * 4.182 / 60.;
    private static final double glycolCoolFactor=  3.9764 * 3.572 / 60.;
    // HFE 7100
    private static final double coolantXferFactor = 6.480 * 1.033 / 60.;

    private final Object listeningLock = new Object();
    // "Thermal" is shorthand for a subsystem providing temperature data
    private volatile boolean isListeningToThermal = false;
    private volatile boolean isListeningToHex = false;
    private volatile String thermalName = null;
    private volatile String hexName = null;
    private volatile boolean setTempInProgress = false;
    private volatile boolean setFlowInProgress = false;
    private volatile String refrigAgentProperty;
    private String refrigAgentHexProperty = RefrigAgentProperties.HEX_TYPE;
    private boolean listenAlarm = false;
    private Channel chanFlow;            // flow-rate channel
    private MonitorUpdateTask taskFlow;  // Task updating flow-rate Channel
    private Channel chanPSupply, chanPReturn;  
    private Channel chanFlowSet, chanTempSet, chanTankSet;  // needed for GUI
    private HeatTransferChannel chanGlycXfer, chanCoolantXfer, chanColdXfer;
    private volatile CCSTimeStamp timeLastData;   // Most-recent received data
    private volatile boolean lastControlMode;
    private volatile AlertState lastDataAlert = AlertState.NOMINAL;
    private double lastRampValue;

    /* Array of Chiller Alerts which block flow when at ALARM level */
    final ChillerAlerts[] blockingAlerts = {
        ChillerAlerts.VACUUM_LIMIT,
        ChillerAlerts.PRESSURE_LIMIT,
        ChillerAlerts.COLD_TEMP_LOW,
        ChillerAlerts.COLD_TEMP_HIGH
    };
    private String blockingCause;

    /* Threads launched from this class  */

    private Thread checkTemp;
    private Thread checkFlow;
    private static final Duration intervalCheckTemp = Duration.ofSeconds(30);
    private static final Duration intervalCheckFlow = Duration.ofSeconds(5);
    private static final Duration setFlowTimeout = Duration.ofSeconds(45);
    private static final double flowTolerance = 0.1;     // gpm

    /* Periodic Tasks  */

    private AgentPeriodicTask checkDataArrival;
    private static final Duration periodCheckData = Duration.ofMillis(500);
    private static long timeDataWarning = 3000;  // ms
    private static long timeDataAlarm = 20000;   // ms
    private AgentPeriodicTask updateCtrlState;
    private static final int UPDATE_CONTROL_INTVL = 1000;   // ms
    private AgentPeriodicTask sendDUTTemperature;
    private static final Duration periodDUT = Duration.ofMillis(1000);
    private AgentPeriodicTask checkBurstDiscP;
    private static final Duration periodBurst = Duration.ofMillis(500);
    private AgentPeriodicTask publishParams;
    private static final Duration periodParams = Duration.ofMinutes(30);

    private MonitorTaskControl monitorControl;
    private final ChillerControlState controlState = new ChillerControlState();
    private boolean gotCommand;

    /**
     *  Constructor.
     */
    public ChillerSubsystem() {
        super("chiller", AgentInfo.AgentType.WORKER);
    }

    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, RefrigConstants.MONITOR_CONTROL);

        /* Set up AgentPeriodicTask to update chiller control state for gui */
        updateCtrlState = new AgentPeriodicTask("updateCtrlState", () -> updateControlState()).withPeriod(Duration.ofMillis(UPDATE_CONTROL_INTVL));
        periodicTaskService.scheduleAgentPeriodicTask(updateCtrlState);

        /* Set up periodic task to check time since last listned-for data */
        Runnable checkData = new Runnable() {
            @Override
            public void run() {
                boolean controlMode = devChiller.getTempControlMode();
                AlertState dataAlert;
                String msg = "";
                if (controlMode) {       // DUT mode
                    long dt = System.currentTimeMillis() - timeLastData.getUTCInstant().toEpochMilli();
                    if (controlMode != lastControlMode || dt <= timeDataWarning) {
                        dataAlert = AlertState.NOMINAL;
                    } else if (dt <= timeDataAlarm) {
                        dataAlert = AlertState.WARNING;
                    } else {
                        dataAlert = AlertState.ALARM;
                    }
                    msg = "time since last data = " + Long.toString(dt) + " ms";
		} else {                 // Normal mode
                    dataAlert = AlertState.NOMINAL;
                }
                if (dataAlert != lastDataAlert) {
                    alertService.raiseAlert(ChillerAlerts.MISSING_THERMAL.newAlert(), dataAlert, msg);
                    lastDataAlert = dataAlert;
                }
                lastControlMode = controlMode;
            }
	};
        checkDataArrival = new AgentPeriodicTask("checkDataArrival",
            checkData).withPeriod(periodCheckData);
        periodicTaskService.scheduleAgentPeriodicTask(checkDataArrival);

        /* Set up periodic task to send DUT temperature data to Chiller  */
        Runnable sendDUT = new Runnable() {
            @Override
            public void run() {
                if (devChiller.getTempControlMode()) {
                    devChiller.sendDUTData();
                }
            }
	};
        sendDUTTemperature = new AgentPeriodicTask("sendDutTemperature",
            sendDUT).withPeriod(periodDUT);
        periodicTaskService.scheduleAgentPeriodicTask(sendDUTTemperature);

        /* Set up periodic task to check for burst-disc pressure problem */
        Runnable checkBurst = new Runnable() {
            @Override
            public void run() {
                double pSupply = chanPSupply.getValue();
                double pReturn = chanPReturn.getValue();
                if (pSupply >= maxPressure || pReturn >= maxPressure) {
                    if (devChiller.isOnline()) {
                        // Now done in ChillerBlockingListener
                        // try {
                        //     quitControllingTemperature();
                        // } catch (DriverException ex) {
                        //     LOG.log(Level.SEVERE, "DriverException while trying to stop chiller flow because of burs-disc pressureb:" + ex);
                        // }
                    }
                    alertService.raiseAlert(ChillerAlerts.PRESSURE_LIMIT.newAlert(), AlertState.ALARM,"Burst-disc pressures = " + pSupply + ", " + pReturn + " over limit " + maxPressure + " psig", AlertService.RaiseAlertStrategy.ON_SEVERITY_CHANGE);
                }
            }
	};
	checkBurstDiscP = new AgentPeriodicTask("checkBurstDiscP",
            checkBurst).withPeriod(periodBurst);
        periodicTaskService.scheduleAgentPeriodicTask(checkBurstDiscP);

        /* Set up periodic task to publish chiller FParam's  */
        Runnable publishFParams = new Runnable() {
            @Override
            public void run() {
                devChiller.publishParameters();
            }
	};
        publishParams = new AgentPeriodicTask("publishParams", publishFParams).withPeriod(periodParams);
        periodicTaskService.scheduleAgentPeriodicTask(publishParams);

    }

   /**
    *  init() method
    */
    @Override
    public void init() {

       /**
        *  Anonymous class to handle alert-clearing
        */
        ClearAlertHandler chillerClearAlertHandler = new ClearAlertHandler() {
           /**
            * Test whether alert can be cleared.
            *
            * @param Alert - the Alert being testsd
            * @param AlertState - state of Alert being tested
            *
            * @return ClearAlertCode
            */
            public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                String alertId = alert.getAlertId();
                if (alertId.equals(ChillerAlerts.TEMP_TIMEOUT.getId()) ||
                    alertId.equals(ChillerAlerts.FLOW_TIMEOUT.getId()) ) {
                    return ClearAlertCode.CLEAR_ALERT;
		}
                if (alertId.equals(ChillerAlerts.PRESSURE_LIMIT.getId())) {
                    return ClearAlertCode.DONT_CLEAR_ALERT;
	        }
                return ClearAlertCode.UNKNOWN_ALERT;
            }
	};

        // Set a property to define that this Agent is a chiller subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.CHILLER_TYPE, ChillerSubsystem.class.getCanonicalName());

        // Register alerts
        alertService.registerAlert(ChillerAlerts.TEMP_TIMEOUT.newAlert(),
                                   chillerClearAlertHandler);
        alertService.registerAlert(ChillerAlerts.FLOW_TIMEOUT.newAlert(),
                                   chillerClearAlertHandler);
        alertService.registerAlert(ChillerAlerts.LOST_LISTENED.newAlert());
        alertService.registerAlert(ChillerAlerts.MISSING_THERMAL.newAlert());
        for (ChillerAlerts ca : blockingAlerts) {
            alertService.registerAlert(ca.newAlert());
        }

        //Register self as an AgentPresenceListener to choose when to start
        //listening to the thermal subsystem.
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);

        //Register self as a StatusMessageListener
        Predicate<BusMessage<? extends Serializable, ?>> filter = 
            BusMessageFilterFactory.messageClass(StatusSubsystemData.class); 
	getMessagingAccess().addStatusMessageListener(this,filter);

        //Check that required devices have been defined
        if (devChiller == null) {
            ErrorUtils.reportConfigError(LOG, name, "Chiller device", "not specified");
        }
        if (devPluto == null) {
            ErrorUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }

        // Get task which updates the fluid FlowRate channel
        List<MonitorUpdateTask> listTasks =
            mon.getMonitorUpdateTasksForDevice (devChiller);
	for (MonitorUpdateTask task : listTasks) {
            List<Channel> chList = task.getAllChannels();
            for (Channel ch : chList) {
                if (chanFlow==null && ch.getSubTypeStr().equals("FLOW_RATE")) {
                    chanFlow = ch;
                    taskFlow = task;
                }
            }
        }
        // Check all Channels to find additional Channels needed
        for (Channel ch : chanList) {
            if (chanFlowSet==null && ch.getSubTypeStr().equals("FLOW_SETPT")) {
                chanFlowSet = ch;
            } else if (chanTempSet==null && ch.getSubTypeStr().equals("SET_POINT")) {
                chanTempSet = ch;
            } else if (chanTankSet==null && ch.getSubTypeStr().equals("TANK_P_SET")) {
                chanTankSet = ch;
            } else if (chanPSupply==null && ch.getName().contains("PBurstSupply")) {
                chanPSupply = ch;
            } else if (chanPReturn==null && ch.getName().contains("PBurstReturn")) {
                chanPReturn = ch;
	    }
        }
        for (HeatTransferChannel ch : heatXferList) {
            if (chanGlycXfer == null && ch.getName().contains("Glyc")) {
		chanGlycXfer = ch;
	    } else if (chanCoolantXfer == null && ch.getName().contains("Coolant")) {
		chanCoolantXfer = ch;
	    } else if (chanColdXfer == null && ch.getName().contains("Cold")) {
		chanColdXfer = ch;
            }
        }

        if (taskFlow == null) {
            ErrorUtils.reportConfigError(LOG, name, "taskFlow", "no update task found with Channel subtype FLOW_RATE");
        }
        if (chanPSupply == null) {
            ErrorUtils.reportConfigError(LOG, name, "chanPSupply", "no update task found with Channel PBurstSupply");
        }
        if (chanPReturn == null) {
            ErrorUtils.reportConfigError(LOG, name, "chanPReturn", "no update task found with Channel PBurstReturn");
        }

        // Set up heat transfer calculations
        if (chanGlycXfer != null) {
            if (glycolCooling) {
                chanGlycXfer.setFactor(glycolCoolFactor);
            } else {
                chanGlycXfer.setFactor(waterCoolFactor);
            }
        }
        if (chanCoolantXfer != null) {
            chanCoolantXfer.setFactor(coolantXferFactor);
        }
        if (chanColdXfer != null) {
            chanColdXfer.setFactor(coolantXferFactor);
        }
        
        // Initial value for use in periodic task checkDataArrival
        timeLastData = CCSTimeStamp.currentTime();
    }
    
    /**
     * postStart (must follow Device.start())
     */
    @Override
    public void postStart() {
        // if (!devChiller.isOnline()) {
        //     throw new RuntimeException("InTESTChillerDevice not online");
        // }

        /* Start dummy threads, so Thread objects will not be null */
        checkTemp = new Thread("dummyT"){};
        checkFlow = new Thread("dummyF"){};

        if (fanControl != null) {
            LOG.log(Level.INFO, "Starting fan speed controller for chiller");
            fanControl.startLoop();
        }
    }

    /*  shutdown method  */
    @Override
    public void shutdown() {
        // Unregister self from Listener (AgentPresence and StatusMessage)
        // notifications.
        getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
        getMessagingAccess().removeStatusMessageListener(this);

	// If in DUT mode, that must be shut down.
        try {
            devChiller.shutdownDUT();
	} catch (DriverException ex) {
            throw new RuntimeException("DriverException in shutdownDUT() "+ex);
        }
    }

   /**
    *  Select refrig subsystem to listen to for average coldplate temperature
    *
    *  @param String value (case-insensitive)
    */
    @ConfigurationParameterChanger(propertyName = "listenTo")
    public void setListenTo(String value) {
        if (value.toLowerCase().equals("thermal")) {
            refrigAgentProperty = RefrigAgentProperties.THERMAL_TYPE;
            listenTo = value;
        } else if (value.toLowerCase().equals("pcp")) {
            refrigAgentProperty = RefrigAgentProperties.PCP_TYPE;
            listenTo = value;
	} else {
            ErrorUtils.reportConfigError(LOG, name, "listenTo", "Must specify either Thermal or Pcp subsystem");
        }
    }

   /**
    *  Veto transition to normal mode if Chiller state is not at setpoint,
    *  or if subsysgtem is in DUT mode but not listening for relevant input
    *
    *  @return  String:  null if not vetoed, explanation if vetoed
    */
    @Override
    public String vetoTransitionToNormalMode() {
        String reply = "";
        if (stateService.getState(ChillerState.class) != 
            ChillerState.SETPOINT) {
            reply += "Chiller not controlling at setpoint  ";
        }
        if (devChiller.getTempControlMode() && !isListeningToThermal) {
            reply += "Chiller subayatem in DUT mode but not listening to Thermal input  ";
        }
        String lights = showLights();
        if (lights.contains("on")) {
            reply += ("Chiller cabinet lights " + lights + ", must be off  ");
        }
        if (reply.equals("")) {
            if (!devChiller.isGuiLocked()) {
                try {
                    devChiller.setGuiLock(true);
                    LOG.log(Level.INFO, "Locking Chiller controller Gui for transition to Normal mde");
                } 
                catch (DriverException ex) {
                    LOG.log(Level.SEVERE, "DriverException on setting Gui lock " + ex);
                    reply += "DriverException on setting Chiller Gui lock";
                }
            }
        }
        return reply.equals("") ? null : reply;
    }

   /**
    *  Determine if any Alerts intended to block flow (blockingAlerts) 
    *  are raised at the ALARM level.
    *
    *  @return boolean
    *  String blockingCause is used to itemize any such.
    */
    private boolean blockFlow() {
        boolean block = false;
        blockingCause = "";
        for (ChillerAlerts ca : blockingAlerts) {
            RaisedAlertHistory alertHistory = alertService.getRaisedAlertSummary().getRaisedAlert(ca.getId());
            if (alertHistory != null &&
                alertHistory.getLatestAlertState() == AlertState.ALARM) {
                block = true;
                blockingCause += (ca.getId() + " ");
            }
        }
        if (block) {
            blockingCause += " raised at ALARM level";
	}
        return block;
    }

   /**
    *  Set chiller temperature and go there using default ramp
    *
    *  @param  double temperature in degrees
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE,
             name = "setTemperature", autoAck = false,
             description = "go to temperature setting using default ramp")
    public void setTemperature(@Argument(description="tempeature in degrees")
                               double temperature) throws DriverException {
        helper()
	.precondition(!blockFlow(), "operattion blocked due to "+blockingCause)
        .precondition(devChiller.getLastErrorAlert() != AlertState.ALARM,
                      "operation blocked due to Chiller Error")
        .precondition(!setTempInProgress, "Previous set command in progress,"
                      + " quitControllingTemperature must be issued first")
        .action(() -> {
	    gotCommand = true;
	    double ramp =  Double.parseDouble(devChiller.readNamedParameter(FParam.RAMP_DEFAULT));
            devChiller.setTemperature(temperature);
            setTempInProgress = true;
            lastRampValue = ramp;
            waitForTemp(temperature, ramp);
	});
    }

   /**
    *  Set chiller temperature and go there using provided ramp rate
    *
    *  @param  double temperature in degrees
    *  @param  double ramp in degrees/minute
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE,
             name = "setTemperatureWithRamp", autoAck = false,
             description = "go to temperature setting using provided ramp")
    public void setTemperatureWithRamp(
        @Argument(description="tempeature in degrees") double temperature,
        @Argument(description="ramp in degrees/min") double ramp)
        throws DriverException {
        helper()
	.precondition(!blockFlow(), "operattion blocked due to "+blockingCause)
        .precondition(devChiller.getLastErrorAlert() != AlertState.ALARM,
                      "operation blocked due to Chiller Error")
        .precondition(!setTempInProgress, "Previous set command in progress,"
                      + " quitControllingTemperature must be issued first")
	.precondition(ramp+rampCorrection >= 0.01, "Ramp + correction (" + rampCorrection + ") must be at least 0.01")
        .action(() -> {
	    gotCommand = true;
            devChiller.setTemperatureWithRamp(temperature, ramp);
            setTempInProgress = true;
            lastRampValue = ramp;
            waitForTemp(temperature, ramp);
	});
    }

   /**
    *  Wait for temperature to reach set-point or until timeout.
    *  Timeout value is estimated time to completion multiplied by a factor.
    *
    *  @param  double  Setpoint temperature (deg-C)
    *  @param  double  Ramp (deg-C/minute)
    *  @throws DriverException
    */
    private void waitForTemp(double temp, double ramp) throws DriverException {
        Double currentTemp = (devChiller.getTempControlMode()) ? 
            mon.getChannelValue("Chiller/ImportedColdTemp") :
            mon.getChannelValue("Chiller/FluidTemperature");
        // Estimated time in seconds to reach set-point
        long timeEst = Math.round(60.*Math.abs(currentTemp - temp)/(ramp + rampCorrection));
        LOG.info("Initiating temperature change to " + Double.toString(temp) +
                 " deg, estimated duration " + Double.toString(timeEst) +
                 " seconds.");
        checkTemp = new Thread("checkTemp") {
            @Override
            public void run() {
                long timeStart = System.currentTimeMillis();
                long timeout = Math.round(temperatureTimeoutFactor * 1000. * timeEst); //ms
                while (true) {
                    try {
                        Thread.sleep(intervalCheckTemp.toMillis());
                    } catch (InterruptedException ex) {
                        throw new RuntimeException("Unexpected interrupt while waiting for requested temperature",ex);
                    }
                    if (stateService.getState(ChillerState.class) ==
                        ChillerState.SETPOINT) {
                        setTempInProgress = false;
                        LOG.info("Chiller temperature is at requested setpoint");
                        break;
		    } else if (stateService.getState(ChillerState.class) !=
			     ChillerState.CONTROLLING) {
                        LOG.info("Chiller no longer pumping, stop waiting for temperature setting");
                        setTempInProgress = false;
                        break;
                    }
                    if (System.currentTimeMillis() - timeStart > timeout) {
                        alertService.raiseAlert(ChillerAlerts.TEMP_TIMEOUT.newAlert(), AlertState.ALARM,"Set temperature timed out at " + Long.toString(timeout/1000) + " s");
                        setTempInProgress = false;
                        break;
                    }
                }
            }
	};
        checkTemp.setDaemon(true);
        checkTemp.start();
    }

   /**
    *  Quit controlling temperature
    *
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE,
             name="quitControllingTemperature",
             description="stop controlling temperature")
    public void quitControllingTemperature() throws DriverException {
        devChiller.quitControllingTemperature();
        setTempInProgress = false;  
    }

   /**
    *  Set flow rate
    *  Command is accepted even if pumps are off, still setting flow setpoint.
    *
    *  @param  double flow  Requested flow rate in gallons per minute
    *  @return String action taken
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE,
             name = "setFlow",
             description = "set flow rate of chilled fluid")
    public String setFlow(@Argument(description="flow rate in gpm")
                           double flow) throws DriverException {
        gotCommand = true;
        devChiller.setFlow(flow);
        ChillerState state = (ChillerState) stateService.getState(ChillerState.class);
        if (state != ChillerState.SETPOINT &&
            state != ChillerState.CONTROLLING) {
            return "setFlow command rejected because chiller pumps are off";
	} 

        /* Wait for flow-setting to complete, or for timeout.  */

        setFlowInProgress = true;
        checkFlow = new Thread("checkFlow") {
            @Override
            public void run() {
                long timeStart = System.currentTimeMillis();
                long timeout = setFlowTimeout.toMillis();
                // Speed up monitor data publication while setting flow.
                taskFlow.forceDataPublicationForDuration(setFlowTimeout);
		while (true) {
                    try {
                        Thread.sleep(intervalCheckFlow.toMillis());
                    } catch (InterruptedException ex) {
                        throw new RuntimeException("Unexpected interrupt while waiting for requested flow",ex);
                    }
                    if (Math.abs(chanFlow.getValue() - flow) < flowTolerance) {
                        LOG.info("Chiller flpw is at requested setpoint");
                        taskFlow.resetForcedDataPublication();
                        break;
                    }
                    if (System.currentTimeMillis() - timeStart > timeout) {
                        alertService.raiseAlert(ChillerAlerts.FLOW_TIMEOUT.newAlert(), AlertState.ALARM,"Set flow timed out at " + Long.toString(timeout/1000) + " s");
			break;
                    }
                }
                setFlowInProgress = false;
            }
	};
        checkFlow.setDaemon(true);
        checkFlow.start();
        return "Chilled fluid flow setting initiated";
    }

    /* Methods to listen to messages from specified subsystem and act on them */

    /**
    *  Command to set chiller tank pressure
    *
    *  @param  value Tank pressure setting (psig)
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name="setTankPressure",description="Set tank pressure in psig")
    public void setTankPressure(@Argument(description = "Tank pressure set value in psig") double value) throws DriverException {
        gotCommand = true;
        devChiller.setTankPressure(value);
    }

   /**
    *  Implementation of StatusMessageListener:
    *  If message is StatusSubsystemData from monitoring, search for
    *  Channels containing needed coldplate temperature data, and save data.
    *  It checks specific subsystems specified by the connected method
    *  of AgentPresenceListener.
    *
    *  @parmm  StatusMessage
    */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        String agentName = msg.getOriginAgentInfo().getName();
        if (!agentName.equals(thermalName) && !agentName.equals(hexName)) {
            return;
        }
        StatusSubsystemData ssd = (StatusSubsystemData) msg;
        KeyValueData kvd = ssd.getSubsystemData();
        if ( kvd instanceof KeyValueDataList && 
             ssd.getDataKey().equals(MonitorUpdateTask.PUBLICATION_KEY) ) {
            KeyValueDataList kvdl = (KeyValueDataList) kvd;
            // Find requested Channels 
            if (agentName.equals(thermalName)) {
                for (KeyValueData data : kvdl.getListOfKeyValueData()) {
                    timeLastData = data.getCCSTimeStamp();
                    if (data.getKey().equals(coldplateChannelPath)) {
                        devChiller.updateColdplateTemp((double)data.getValue(), 
                                                       timeLastData);
                        break;
		    }
                }
            }
            if (agentName.equals(connectedToCamera ? hexName : thermalName)) {
                for (KeyValueData data : kvdl.getListOfKeyValueData()) {
                    timeLastData = data.getCCSTimeStamp();
                    if (data.getKey().equals(coldInletChannelPath)) {
                        devChiller.updateColdInletTemp((double)data.getValue(), 
                                                       timeLastData);
	        
                    } else if (data.getKey().equals(coldOutletChannelPath)) {
                        devChiller.updateColdOutletTemp((double)data.getValue(), 
                                                       timeLastData);
                    }
                }
            }
        }
    }

   /**
    * Implementation of AgentPresenceListener:
    * Checks for appropriate agents of appropriate thermal (or equivalent)
    * type, and sets subsystem names (thermalName and hexName) which
    * are then used by StatusMessageListener.  It accepts only the first
    * appropriate agent connected of each type.
    */
    @Override
    public void connected(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            String runMode = agent.getAgentProperty("runMode", "shouldNeverGetTheDefaultValue");
            // Thermal or equivalent tyoe
            if (agent.hasAgentProperty(refrigAgentProperty) &&
                (RunMode.isSimulation() ? runMode.equals("simulation") :
		 runMode.equals("normal"))) {
                // If reconnecting, only accept subsystem originally connected
                if ( !isListeningToThermal && (thermalName == null 
		    || thermalName.equals(agent.getName())) ) {
                    isListeningToThermal = true;
                    thermalName = agent.getName();
                    LOG.log(Level.INFO,"Starting to listen to messages from " + listenTo.toUpperCase() + " subsystem {0}.",thermalName);
                    if (listenAlarm) {
                        alertService.raiseAlert(ChillerAlerts.LOST_LISTENED.newAlert(), AlertState.NOMINAL, "Subsystem " + thermalName + " reconnected");
                        listenAlarm = false;
                    }
                } else {
                    //We should probably raise an exception here.
                    LOG.log(Level.SEVERE,"More than one thermal subsystem on the buses!!! Currently listening to {0} and just connected {1}!!", new Object[]{thermalName,agent.getName()});
                }
            }

            // Hex tyoe
            if (agent.hasAgentProperty(refrigAgentHexProperty) &&
                (RunMode.isSimulation() ? runMode.equals("simulation") :
		 runMode.equals("normal"))) {
                // If reconnecting, only accept subsystem originally connected
                if ( !isListeningToHex && (hexName == null 
		    || hexName.equals(agent.getName())) ) {
                    isListeningToHex = true;
                    hexName = agent.getName();
                    LOG.log(Level.INFO,"Starting to listen to messages from HEX subsystem {0}.",hexName);
                } else {
                    //We should probably raise an exception here.
                    LOG.log(Level.SEVERE,"More than one hex subsystem on the buses!!! Currently listening to {0} and just connected {1}!!", new Object[]{hexName,agent.getName()});
                }
            }
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            // Disconnect of subsystem thermalName
            if ( isListeningToThermal && agent.getName().equals(thermalName) ) {
                String msg = "Subsystem " + thermalName + "disconnected";
	        if (devChiller.getTempControlMode()) {
                    alertService.raiseAlert(ChillerAlerts.LOST_LISTENED.newAlert(), AlertState.ALARM, msg);
                    listenAlarm = true;
                }
                LOG.log(Level.INFO,msg+", no longer listening to its messages");
                isListeningToThermal = false;
            } 
            // Disconnect of subsystem hexName
            if ( isListeningToHex && agent.getName().equals(hexName) ) {
                String msg = "Subsystem " + hexName + "disconnected";
                LOG.log(Level.INFO,msg+", no longer listening to its messages");
                isListeningToHex = false;
            } 
        }        
    }    
    
  /**
   *  Gets the state of the chiller system.
   *
   *  This is intended to be called by GUIs during initialization
   *
   *  @return  The thermal control state
   */
    @Command(type=Command.CommandType.QUERY, level=NORMAL,
             name="getControlState", description="Get the chiller state")
    public ChillerControlState getControlState()
    {
        controlState.setFastPeriod(monitorControl.getFastPeriod());
        return controlState;
    }

   /**
    *  Turn chiller-cabinet lights on or off
    *
    *  @param  boolen on  <true/false> for <on/off>
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name="setLights",
             description="Turn chiller-cabinet lights on or off")
    public void setLights(@Argument(description="<true/false> for <on/off>") boolean on) throws DriverException {
        devMaq20.setLightOn(ChillerMaq20Device.LightNumber.LIGHT1, on);
        devMaq20.setLightOn(ChillerMaq20Device.LightNumber.LIGHT2, on);
    }

   /**
    *  Get status of cabinet lighta
    *
    *  @return SwitchState[] for array over lights
    */
    private SwitchState[] getLights() {
	int idx = -1;
	SwitchState[] val = {SwitchState.OFFLINE, SwitchState.OFFLINE};
	for (ChillerMaq20Device.LightNumber light : ChillerMaq20Device.LightNumber.values()) {
	    Boolean ilo = devMaq20.isLightOn(light);
	    if (ilo != null) {
	        val[++idx] = ilo ? SwitchState.ON : SwitchState.OFF;
	    }
	}
	return val;
    }

   /**
    *  Show status of cabinet lighta
    *
    *  @return String  "on" or "off" for each light
    */
    @Command(type=Command.CommandType.QUERY, level=NORMAL, 
             name="showLights", description="Get status of cabinet lights")
    public String showLights() {
        String status = "";
        for (ChillerMaq20Device.LightNumber light : ChillerMaq20Device.LightNumber.values()) {
            status += (light + " = " + (devMaq20.isLightOn(light) ? "on  " : "off "));
        }
        return status;
    }

    /**
     *  Tell PLC to enables/disables the chiller operation
     * 
     *  @param  on  Whether enabling
     */
    @Command(type=Command.CommandType.ACTION, level=ENGINEERING_ROUTINE,
             description="Enable or disable chiller operation")
    public void enableChiller(boolean on)
    {
        gotCommand = true;
        devPluto.setSwitchOn(ChillerSwitches.SW_ENABLE_CHILLER, on);
    }

    /**
     *  Gets the list of latched PLC condition names.
     *
     *  @return  The condition names.
     */
    @Command(type=Command.CommandType.QUERY, level=NORMAL,
             description="Get latched PLC condition names")
    public List<String> getPlcLatchNames()
    {
        return ChillerLatches.getNames();
    }

    /**
     *  Clears a (named) latched PLC condition.
     *
     *  @param  cond  The condition name.
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, level=ENGINEERING_ROUTINE,
             description="Clear a latched PLC condition")
    public void clearPlcLatch(@Argument(description="The condition name") String cond) throws RefrigException
    {
        gotCommand = true;
        devPluto.clearLatch(ChillerLatches.getId(cond));
    }

    /**
     *  Clears all latched PLC conditions.
     */
    @Command(type=Command.CommandType.ACTION, level=ENGINEERING_ROUTINE,
             description="Clear all latched PLC conditions")
    public void clearAllPlcLatches()
    {
        gotCommand = true;
        devPluto.clearAllLatches();
    }

    /**
     *  Updates the chiller control state periodically.
     */
    private void updateControlState()
    {
        boolean changed = monitorControl.hasPeriodChanged() || gotCommand;
        gotCommand = false;

        ChillerState state = devChiller.getChillerState();
        if (state != controlState.getChillerState()) {
            controlState.setChillerState(state);
            changed = true;
        }
        double flow = chanFlowSet.getValue();
        double oldFlow = controlState.getFlowSet();
        if ((!Double.isNaN(flow) || !Double.isNaN(oldFlow)) && flow !=oldFlow) {
            controlState.setFlowSet(flow);
            changed = true;
        }
        double temp = chanTempSet.getValue();
        double oldTemp = controlState.getSetPoint();
        if ((!Double.isNaN(temp) || !Double.isNaN(oldTemp)) && temp !=oldTemp) {
            controlState.setSetPoint(temp);
            changed = true;
        }
        double tank = chanTankSet.getValue();
        double oldTank = controlState.getTankSet();
        if ((!Double.isNaN(tank) || !Double.isNaN(oldTank)) && tank !=oldTank) {
            controlState.setTankSet(tank);
            changed = true;
        }
        double oldLastRamp = controlState.getLastRamp();
        if ((!Double.isNaN(lastRampValue) || !Double.isNaN(oldLastRamp)) && lastRampValue !=oldLastRamp) {
            controlState.setLastRamp(lastRampValue);
            changed = true;
        }
        double ramp = devChiller.getDefaultRamp();
        double oldRamp = controlState.getDefaultRamp();
        if ((!Double.isNaN(ramp) || !Double.isNaN(oldRamp)) && ramp !=oldRamp) {
            controlState.setDefaultRamp(ramp);
            changed = true;
        }
        String mode = devChiller.getTempControlMode() ? "DUT" : "RTD1";
        String oldMode = controlState.getTemperatureMode();
        if (!mode.equals(oldMode)) {
            controlState.setTemperatureMode(mode);
            changed = true;
        }
        ErrorWords ew = devChiller.getLastErrorWords();
        if (ew != null) {
            boolean errorEMO = (ew.getError2() & emoMask) != 0;
            boolean oldErrorEMO = controlState.getChillerErrorEMO();
            if (errorEMO != oldErrorEMO) {
                controlState.setChillerErrorEMO(errorEMO);
                changed = true;
	    }
            boolean errorOther = ew.getError1() != 0 || 
		                 (ew.getError2() & ~emoMask) != 0;
            boolean oldErrorOther = controlState.getChillerErrorOther();
            if (errorOther != oldErrorOther) {
                controlState.setChillerErrorOther(errorOther);
                changed = true;
	    }
            boolean warning = ew.getWarning1() != 0 || ew.getWarning1() != 0;
            boolean oldWarning = controlState.getChillerWarning();
            if (warning != oldWarning) {
                controlState.setChillerWarning(warning);
                changed = true;
	    }
        }
        SwitchState[] oldLights = controlState.getLightState();
	SwitchState[] lights = getLights();
        if (lights != oldLights) {
            controlState.setLightState(lights);
            changed = true;
        }
	
        changed |= devPluto.updateState(controlState.getPlcState());
        if (changed) {
            publishControlState();
        }
    }

    /**
     *  Publishes the state of the chiller subsystem.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishControlState()
    {
        controlState.setFastPeriod(monitorControl.getFastPeriod());
        publishSubsystemDataOnStatusBus(new KeyValueData(ChillerControlState.KEY, controlState));
    }    

   /**
    *  Is chiller connected to camera
    *
    *  @eturn boolean
    */
    boolean getConnectedToCamera() {
        return connectedToCamera;
    }

}
