package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.wattsup.WattsUp;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a WattsUp power meter.
 *
 *  @author Owen Saxton
 */
public class WUDevice extends Device implements WattsUp.Listener {

    /**
     *  Constants.
     */
    private static final int
        CHAN_WATTS  = 0,
        CHAN_VOLTS  = 1,
        CHAN_FREQ   = 2,
        N_DEV_CHANS = 3,
        LOG_PERIOD  = 1;

    /**
     *  Data fields.
     */
    private String   node;        // Name of the node attached to the meter
    private String   devcName;    // The serial device name

    private static final Logger LOG = Logger.getLogger(WUDevice.class.getName());
    private final WattsUp wtu = new WattsUp();  // WattsUp hardware object
    private final double[] value = new double[N_DEV_CHANS];  // Current values
    private boolean initError = false;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        wtu.addListener(this);
        fullName = "WattsUp meter (" + (node == null ? "local" : node)
                     + (devcName == null ? "" : ":" + devcName) + ")";
    }


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            wtu.openSerial(devcName, 0);
            wtu.restart();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
        }
    }


    /**
     *  Closes the connection.
     *
     *  This is a no-op
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The Channel to read
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        int hwChan = ch.getHwChan();
        if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
            ErrorUtils.reportChannelError(LOG, ch.getPath(), "hw channel number", hwChan);
        }

        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  ch  The Channel to read.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(Channel ch)
    {        
        return isOnline() ? value[ch.getHwChan()] : Double.NaN;
    }


    /**
     *  Changes the WattsUp? meter powered state.
     *
     *  Performs meter setup upon power-on.
     *
     *  @param  on  Whether the meter is powered on or not
     */
    @Override
    public void setPowered(boolean on)
    {
        if (on) {
            try {
                wtu.setLoggedFields((1 << WattsUp.FLD_WATTS) |
                                    (1 << WattsUp.FLD_VOLTS) |
                                    (1 << WattsUp.FLD_FREQUENCY));
                wtu.setExternalLogging(LOG_PERIOD);
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error configuring {0}: {1}", new Object[]{fullName, e});
            }
        }
        else {
            for (int j = 0; j < N_DEV_CHANS; j++) {
                value[j] = 0F;
            }
        }
    }


    /**
     *  Changes the WattsUp? meter open state.
     */
    @Override
    public void setClosed()
    {
        setOnline(false);
    }


    /**
     *  Receives data periodically from the WattsUp? meter.
     *
     *  @param  data  The array of data from the meter.
     */
    @Override
    public void processData(double[] data)
    {
        value[CHAN_WATTS] = data[WattsUp.FLD_WATTS];
        value[CHAN_VOLTS] = data[WattsUp.FLD_VOLTS];
        value[CHAN_FREQ]  = data[WattsUp.FLD_FREQUENCY];
    }

}
