package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.services.alert.AlertEvent;
import org.lsst.ccs.services.alert.AlertListener;
import org.lsst.ccs.subsystem.common.devices.pluto.PlutoDevice;
import org.lsst.ccs.subsystem.refrig.constants.ChillerConditions;
import org.lsst.ccs.subsystem.refrig.constants.ChillerLatches;
import org.lsst.ccs.subsystem.refrig.constants.ChillerPLCState;
import org.lsst.ccs.subsystem.refrig.constants.ChillerPlcAlert;
import org.lsst.ccs.subsystem.refrig.constants.ChillerSwitches;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.ChillerPlcState;

/**
 *  Handles the chiller system Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class ChillerPlutoDevice extends PlutoDevice implements AlertListener {

    /**
     *  Constants.
     */
    private static final int
        NUM_AREAS = 14,
        SWDI_ON_BIT = 0,
        SWDI_OFF_BIT = 1,
        SWDI_READ_AREA = 2,
        SWDI_READ_BIT = 3,
        LTDI_RESET_BIT = 0,
        LTDI_READ_AREA = 1,
        LTDI_READ_BIT = 2,
        LTDI_PEND_AREA = 3,
        LTDI_PEND_BIT = 4,
        CNDI_READ_AREA = 0,
        CNDI_READ_BIT = 1;
    private static final int
        CLEAR_ALL_LATCHES_BIT = 5,
        PLC_STATUS_MASK = 0x01,
        ERROR_AREA = 11;

    /**
     *  Private lookup maps, etc.
     */
    private static final int[][] switches = new int[ChillerSwitches.NUM_SWITCHES][];
    static {                              // ON bit, OFF bit, read area, read bit 
        switches[ChillerSwitches.SW_ENABLE_CHILLER] = new int[]{0, 1, 13, 4};
    }
    private static final int[][] latches = new int[ChillerLatches.NUM_LATCHES][];
    static {                                        // Reset bit, on area, on bit, ind area, ind bit
        latches[ChillerLatches.LATCH_SMOKE_DETC]  = new int[]{2, 10, 1, 10, 3};
        latches[ChillerLatches.LATCH_PERMIT]      = new int[]{3, 10, 5, 10, 7};
        latches[ChillerLatches.LATCH_EXT_EMO]     = new int[]{7, 10, 9, 10, 11};
        latches[ChillerLatches.LATCH_BD_SUPPLY_P] = new int[]{4, 12, 1, 12, 3};
        latches[ChillerLatches.LATCH_BD_RETURN_P] = new int[]{6, 12, 9, 12, 11};
        latches[ChillerLatches.LATCH_LEAK_DETC]   = new int[]{8, 11, 3, 11, 5};
        latches[ChillerLatches.LATCH_LEAK_FAULT]  = new int[]{9, 11, 11, 11, 13};
    }
    private static final int[][] conditions = new int[ChillerConditions.NUM_CONDITIONS][];
    static {                                                // Area, bit
        conditions[ChillerConditions.COND_PERMIT]         = new int[]{13, 0};
        conditions[ChillerConditions.COND_KEY_SWITCH]     = new int[]{13, 1};
        conditions[ChillerConditions.COND_EMO_READBACK]   = new int[]{13, 2};
        conditions[ChillerConditions.COND_SMOKE_OK]       = new int[]{13, 3};
        conditions[ChillerConditions.COND_CLP_REF_PERM]   = new int[]{13, 4};
        conditions[ChillerConditions.COND_ALLOWED]        = new int[]{13, 5};
        conditions[ChillerConditions.COND_CHILLER_EMO]    = new int[]{13, 6};
        conditions[ChillerConditions.COND_MASTER_RESET]   = new int[]{13, 7};
        conditions[ChillerConditions.COND_BD_SUPP_NO_WRN] = new int[]{13, 8};
        conditions[ChillerConditions.COND_BD_SUPP_NO_ERR] = new int[]{13, 9};
        conditions[ChillerConditions.COND_BD_RETN_NO_WRN] = new int[]{13, 10};
        conditions[ChillerConditions.COND_BD_RETN_NO_ERR] = new int[]{13, 11};
    }
    private static final Map<Integer, ChillerPlcAlert> latchAlertMap = new HashMap<>();
    static {
        latchAlertMap.put(ChillerLatches.LATCH_SMOKE_DETC, ChillerPlcAlert.SMOKE_DETC);
        latchAlertMap.put(ChillerLatches.LATCH_PERMIT, ChillerPlcAlert.NO_PERMIT);
        latchAlertMap.put(ChillerLatches.LATCH_EXT_EMO, ChillerPlcAlert.EXT_EMO);
        latchAlertMap.put(ChillerLatches.LATCH_BD_SUPPLY_P, ChillerPlcAlert.BD_SUPPLY_ERROR);
        latchAlertMap.put(ChillerLatches.LATCH_BD_RETURN_P, ChillerPlcAlert.BD_RETURN_ERROR);
    }
    private static final Map<String, Integer> revLatchAlertMap = new HashMap<>();
    static {
        for (Map.Entry e : latchAlertMap.entrySet()) {
            revLatchAlertMap.put(((ChillerPlcAlert)e.getValue()).getId(), (int)e.getKey());
        }
    }
    private static final Map<Integer, ChillerPlcAlert> condAlertMap = new HashMap<>();
    static {
        condAlertMap.put(ChillerConditions.COND_BD_SUPP_NO_WRN, ChillerPlcAlert.BD_SUPPLY_WARN);
        condAlertMap.put(ChillerConditions.COND_BD_RETN_NO_WRN, ChillerPlcAlert.BD_RETURN_WARN);
    }


    /**
     *   Constructor.
     */
    public ChillerPlutoDevice()
    {
        super(NUM_AREAS);
    }


    /**
     *  Init phase.
     */
    @Override
    public void init()
    {
        super.init();

        // Register Alerts raised by the PLC
        for (ChillerPlcAlert alert : ChillerPlcAlert.values()) {
            alertService.registerAlert(alert.getAlert());            
        }

        // Add alert listener
        alertService.addListener(this);
    }


    /**
     *  Tests whether the PLC is active
     *
     *  @return  Whether the PLC is active, or null if offline
     */
    public Boolean isPlcActive()
    {
        try {
            int status = plu.readModuleStatus();
            return (status & PLC_STATUS_MASK) != 0;
        }
        catch (DriverException e) {
            return null;
        }
    }


    /**
     *  Gets the PLC state.
     * 
     *  @return  The state
     */
    public ChillerPLCState getPlcState()
    {
        Boolean isActive = isPlcActive();
        return isActive == null ? ChillerPLCState.OFFLINE : !isActive ? ChillerPLCState.PLC_DEAD :
               isConditionActive(ChillerConditions.COND_CHILLER_EMO) ? ChillerPLCState.RUNNING : ChillerPLCState.STOPPED;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the protection Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    protected void setSwitchOn(int sw, boolean on)
    {
        int bitNum = switches[sw][on ? SWDI_ON_BIT : SWDI_OFF_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Gets the state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  The switch state
     */
    public SwitchState getSwitchState(int sw)
    {
        Boolean isOn = isSwitchOn(sw);
        return isOn != null ? isOn ? SwitchState.ON : SwitchState.OFF : SwitchState.OFFLINE;
    }


    /**
     *  Gets the on state of a switch.
     *
     *  The on/off state is not the state of the bit that was toggled, but is read back
     *  directly from the PLC output line.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    protected Boolean isSwitchOn(int sw)
    {
        int[] swData = switches[sw];
        Integer value = readAddBit(swData[SWDI_READ_AREA], swData[SWDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Gets whether a latched condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 0
     */
    public Boolean isLatchActive(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_READ_AREA], condData[LTDI_READ_BIT] + 16);
        return value != null ? value == 0 : null;
    }


    /**
     *  Gets whether a latched condition is latched.
     * 
     *  @param  cond  The condition number
     *  @return  Whether latched - indicated by the bit being 1
     */
    public Boolean isLatchLatched(int cond)
    {
        int[] condData = latches[cond];
        Integer value = readAddBit(condData[LTDI_PEND_AREA], condData[LTDI_PEND_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    /**
     *  Clears a latched condition.
     * 
     *  @param  cond  The condition number
     */
    public void clearLatch(int cond)
    {
        int bitNum = latches[cond][LTDI_RESET_BIT];
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Clears all latched conditions.
     */                                           
    public void clearAllLatches()
    {
        int bitNum = CLEAR_ALL_LATCHES_BIT;
        toggleBit(bitNum / 16, bitNum & 0x0f);
    }


    /**
     *  Gets whether a condition is active.
     * 
     *  @param  cond  The condition number
     *  @return  Whether active - indicated by the bit being 1
     */
    public Boolean isConditionActive(int cond)
    {
        int[] condData = conditions[cond];
        Integer value = readAddBit(condData[CNDI_READ_AREA], condData[CNDI_READ_BIT] + 16);
        return value != null ? value != 0 : null;
    }


    public int getErrorCode()
    {
        return readAddWord(ERROR_AREA, 0);
    }


    /**
     *  Updates the PLC state.
     *
     *  This is to  be called periodically to keep the GUI updated
     * 
     *  @param  plcState  The PLC state to be updated
     *  @return  Whether the state changed
     */
    public boolean updateState(ChillerPlcState plcState)
    {
        boolean changed = false;

        ChillerPLCState newPlcState = getPlcState();
        ChillerPLCState oldPlcState = plcState.getPlcState();
        if (newPlcState != oldPlcState) {
            changed = true;
            plcState.setPlcState(newPlcState);
            ChillerPlcAlert alert = ChillerPlcAlert.PLC_NOT_ALIVE;
            String plcDesc = "Chiller protection PLC ";
            if (newPlcState == ChillerPLCState.STOPPED || newPlcState == ChillerPLCState.RUNNING) {
                if (!(oldPlcState == ChillerPLCState.STOPPED || oldPlcState == ChillerPLCState.RUNNING)) {
                    lowerAlert(alert, plcDesc + "is alive");
                }
            }
            else {
                String errDesc = newPlcState == ChillerPLCState.PLC_DEAD ? "has died: error = " + getErrorCode() : "is offline"; 
                raiseAlarm(alert, plcDesc + errDesc);
            }
        }

        for (int sw = 0; sw < ChillerSwitches.NUM_SWITCHES; sw++) {
            SwitchState state = getSwitchState(sw);
            if (state != plcState.getSwitchState(sw)) {
                plcState.setSwitchState(sw, state);
                changed = true;
            }
        }

        for (int cond = 0; cond < ChillerLatches.NUM_LATCHES; cond++) {
            Boolean active = isLatchActive(cond);
            Boolean latched = isLatchLatched(cond);
            LatchState state = active == null || latched == null ? LatchState.OFFLINE :
                               latched ? LatchState.LATCHED :
                               active ? LatchState.ACTIVE : LatchState.CLEAR;
            LatchState oldState = plcState.getLatch(cond); 
            if (state != oldState) {
                plcState.setLatch(cond, state);
                ChillerPlcAlert alert = latchAlertMap.get(cond);
                if (state == LatchState.ACTIVE) {
                    raiseAlarm(alert, "Chiller protection PLC error condition set");
                }
                else if (state != LatchState.OFFLINE) {
                    if (oldState != LatchState.ACTIVE && state == LatchState.LATCHED) {
                        raiseAlarm(alert, "Chiller protection PLC error condition set");
                    }
                    if (oldState == LatchState.ACTIVE || state == LatchState.LATCHED) {
                        lowerAlert(alert, "Chiller protection PLC error condition cleared");
                    }
                }
                changed = true;
            }
        }

        for (int cond = 0; cond < ChillerConditions.NUM_CONDITIONS; cond++) {
            Boolean active = isConditionActive(cond);
            ConditionState state = active == null ? ConditionState.OFF :
                                   active ? ConditionState.YES : ConditionState.NO;
            if (state != plcState.getCondition(cond)) {
                plcState.setCondition(cond, state);
                ChillerPlcAlert alert = condAlertMap.get(cond);
                if (alert != null) {
                    if (state == ConditionState.NO) {
                        raiseWarning(alert, "Chiller protection PLC warning condition set");
                    }
                    else if (state == ConditionState.YES) {
                        lowerAlert(alert, "Chiller protection PLC warning condition cleared");
                    }
                }
                changed = true;
            }
        }

        return changed;
    }


    /**
     *  Raises an alarm alert.
     *
     *  @param  alert  The protection alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseAlarm(ChillerPlcAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.ALARM, cond);
    }


    /**
     *  Raises a warning alert.
     *
     *  @param  alert  The protection alert to raise
     *  @param  cond   The alert condition
     */
    private void raiseWarning(ChillerPlcAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.WARNING, cond);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The protection alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(ChillerPlcAlert alert, String cond)
    {
        alertService.raiseAlert(alert.getAlert(), AlertState.NOMINAL, cond);
    }


    /**
     *  Alert event handler.
     *
     *  Resets PLC latch when corresponding alert is cleared.
     *
     *  @param  event  The alert event
     */
    @Override
    public void onAlert(AlertEvent event)
    {
        if (event.getType() != AlertEvent.AlertEventType.ALERT_CLEARED) return;
        for (String id : event.getClearedIds()) {
            Integer cond = revLatchAlertMap.get(id);
            if (cond != null) {
                clearLatch(cond);
            }
        }
    }

}
