package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20AnalogControl;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;

/**
 *  Handles compressor MAQ20 DIOL and VO modules.
 *
 *  @author Owen Saxton
 */
public class CompMaq20Device extends Maq20Device implements Compressor.SwitchDevice {

    /**
     *  Constants.
     */
    public static final int
        SW_OIL_SEP_HEATER  = 1,
        NUM_COLD_SWITCHES  = 1,   // Used in simulation
        SW_COOLANT_VALVE   = 0x11,
        SW_ORIFICE_VALVE   = 0x02,
        SW_BYPASS_VALVE    = 0x03,
        SW_SURGE_HEATER    = 0x04,
        SW_CABINET_LIGHT   = 0x10,
        NUM_CRYO_SWITCHES  = 4,   // Used in simulation
        CHAN_HGB_VALVE     = 0,
        CHAN_EEPR_VALVE    = 1,
        CHAN_COOLANT_VALVE = 2;

    /**
     *  Data fields.
     */
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<CompMaq20DiscControl> discCtrlDef = new ArrayList<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Maq20AnalogControl anlgCtrl;
    
    private static final Logger LOG = Logger.getLogger(CompMaq20Device.class.getName());
    private CompMaq20DiscControl[] discCtrl;
    private boolean gotValveError = false;


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    public void setType(int type)
    {
        if (discCtrlDef.isEmpty()) {
            ErrorUtils.reportConfigError(LOG, name, "MAQ20 discrete control", "not defined");
        }
        int modExptd = (type == CompTypes.TYPE_COLD ? 1 : 2);
        if (discCtrlDef.size() < modExptd) {
            ErrorUtils.reportConfigError(LOG, name, "MAQ20 discrete module count", "must be at least " + modExptd);
        }
        discCtrl = new CompMaq20DiscControl[discCtrlDef.size()];
        for (CompMaq20DiscControl ctrl : discCtrlDef) {
            int index = ctrl.getIndex();
            if (index < 0 || index >= discCtrl.length) {
                ErrorUtils.reportConfigError(LOG, ctrl.getPath(), "index", "is negative or exceeds discrete module count");
            }
            if (discCtrl[index] != null) {
                ErrorUtils.reportConfigError(LOG, ctrl.getPath(), "index", "is already used by another discrete module");
            }
            discCtrl[index] = ctrl;
        }
        if (type == CompTypes.TYPE_COLD && anlgCtrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "MAQ20 analog control", "not defined");
        }
    }


    /**
     *  Sets a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        try {
            discCtrl[sw >> 4].setLineOn(sw & 0x0f, on);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting switch for {0}: {1}", new Object[]{getPath(), e});
            setOnline(false);
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return discCtrl[sw >> 4].isLineOn(sw & 0x0f);
    }


    /**
     *  Sets a valve position.
     *
     *  @param  chan  The valve channel number
     *  @param  posn  Requested position (0 - 1)
     */
    public void setValvePosition(int chan, double posn)
    {
        double value = 10.0 * Math.min(Math.max(posn, 0.0), 1.0);
        try {
            anlgCtrl.writeValue(chan, value);
            gotValveError = false;
        }
        catch (DriverException e) {
            if (!gotValveError) {
                LOG.log(Level.SEVERE, "Error setting valve position for {0}: {1}", new Object[]{getPath(), e});
                setOnline(false);
            }
            gotValveError = true;
        }
    }


    /**
     *  Gets a valve position.
     *
     *  @param  chan  The valve channel number
     *  @return  Valve position (0 - 1), or NaN if error
     */
    public double getValvePosition(int chan)
    {
        try {
            double value = anlgCtrl.readValue(chan) / 10.0;
            gotValveError = false;
            return value;
        }
        catch (DriverException e) {
            if (!gotValveError) {
                LOG.log(Level.SEVERE, "Error getting valve position for {0}: {1}", new Object[]{getPath(), e});
                setOnline(false);
            }
            gotValveError = true;
            return Double.NaN;
        }
    }

}
