package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.data.Measurement;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.chiller.Chiller;
import org.lsst.ccs.drivers.chiller.Chiller.ErrorWords;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.ChillerAlerts;
import org.lsst.ccs.subsystem.refrig.constants.ChillerErrorTables;
import org.lsst.ccs.subsystem.refrig.constants.ChillerState;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/* Pre-defined command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ROUTINE;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_ADVANCED;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING_EXPERT;

/**
 *  Interfaces with driver for inTEST chiller
 *
 *  Version adapted to Chiller driver version (JIRA LSStCCSDRIVERS-432)
 *  which sets chiller parameters one at a time, rather than in groups.
 *
 *  @author Al Eisner
 */
public class InTESTChillerDevice extends Device implements HasDataProviderInfos{

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alert;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent subsys;

   /**
    *  Data fields.
    */

    private final Chiller chiller = new Chiller();  // Associated driver

   /**
    *  The Channel type in the groovy file identifies a Channel category"
    *  "Temp" (temperature), "Flow" (flow rate), "Pres"(pressure). 
    *  "Life" (lifetime), "HC" (heat and cool percentages), "Register" 
    *  (a Chiller register), "ColdPlate"(imported).  
    *
    *  The specific Channel is identified by subtype.
    *
    *  chanKey is a numeric Channel counter, returned via checkChannel.
    */
    private Integer chanKey;
    private boolean readHeatCool = false;
    private double[] heatCool;
    private ErrorWords lastErrorWords;
    // private boolean clearAgain = false;
    private volatile AlertState lastErrorAlert;
    private volatile ChillerState lastState;
    private volatile boolean expectSetpoint = false;
    private boolean departedSetPoint = false;
    private volatile Measurement coldplateTemperature;  // from Listener
    private volatile Measurement coldInletTemperature;  // from Listener
    private volatile Measurement coldOutletTemperature; // from Listener
    private volatile boolean dutMode;
    private volatile int sentDut;
    private volatile double lastSet, lastDyn;
    private volatile boolean guiLocked;

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter(isFinal = true, description = "IP host", units = "unitless")
    protected volatile String host;

    private static final Logger LOG = Logger.getLogger(InTESTChillerDevice.class.getName());

    private static final int WARN_RAMP_TIMEOUT = 0x10;

    /* Hard-coded hardware-initialization parameters

    // to be added; example from turbo:
    // private final int rotfreq_set_304 = 1010;

  /**
   *  Register publication of named chiller parameters
   *
    * @return List<DataProviderInfo>
    */
    @Override
    public List<DataProviderInfo> getDataProviderInfos() {
        //List of DataProviderInfo objects to register
        List<DataProviderInfo> list = new ArrayList();

        Map attributes = new HashMap();
        attributes.put(DataProviderInfo.Attribute.TYPE,"double");
        for (Chiller.FParam param : Chiller.FParam.values()) {
            String path = "FParam/" + param.toString();
            String units = param.getUnits();
            if (units.equals("")) {
                units = "unitless";
            }
            String descr = String.format("F%02d: %s", param.getNumber(),
                                         param.getDescription());
            attributes.put(DataProviderInfo.Attribute.UNITS, units);
            attributes.put(DataProviderInfo.Attribute.DESCRIPTION, 
                           descr);
            if (param.getTolerance() == 1.0) {
                attributes.put(DataProviderInfo.Attribute.FORMAT,"%.0f");
            } else {
                attributes.put(DataProviderInfo.Attribute.FORMAT,"%.1f");
            }
                
            DataProviderInfo info = new DataProviderInfo(path, DataProviderInfo.Type.TRENDING, path, attributes);
            list.add(info);
        }
        return list;
    }

   /**
    *  Performs basic initialization.  
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        fullName = "InTESTChiller " + name + " (" + host + ")";
        chanKey = 0;
        lastErrorAlert = AlertState.NOMINAL;
        lastErrorWords = new ErrorWords(0,0,0,0);
        lastState = ChillerState.OFFLINE;

        if (host == null) {
            ErrorUtils.reportConfigError(LOG, name, "host", "is missing");
        }

        /* Registe alerts and state */

        alert.registerAlert(ChillerAlerts.CHILLER_ERR.newAlert());
        alert.registerAlert(ChillerAlerts.DEPARTED_SETPT.newAlert());
        stateService.registerState(ChillerState.class, "Chiller State", subsys);
        stateService.updateAgentState(lastState);   // Initial value
    }

   /**
    *  Define Channel groups
    *
     * @param ch The Channel object
     * @return A String representing the group this channel belongs to.
     *         A null is returned if the Channel belongs to no group.
     */
    @Override
    protected String getGroupForChannel(Channel ch) {
        if (ch.getTypeStr().equals("HC")) {
            LOG.info("\n Assign heatCool group for Channel "+ch.getName());
            return "heatCool";
        }
        return null;
    }

   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize() {

        try {
            chiller.open(host);

            /* Hardware initialization */

            if (!inited) {
                setGuiLock(false);
            }
            shutdownDUT();      // Temporary, later DUT mode to be configured
            initSensors();
            setOnline(true);
            LOG.info("\n Connected to " + fullName);

            LOG.info("Read and publish chiller parameters");
            publishParameters();
        }

        catch (DriverException e) {
            if (!inited) {
                LOG.severe("Error connecting to or initializaing " 
                          + fullName + ": " + e);
            }
            close();
        }

        inited = true;
    }

   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            chiller.close();
        }
        catch (DriverException e) {
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *  The numeric type returned is just an ordinal Channel index;
    *  the numeric subtyoe is not used, just set to 0.
    *
    *  @param   Channel ch    The channel
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        String type = ch.getTypeStr();
	String subtype = ch.getSubTypeStr();
	String name = ch.getName();
        try {
            switch (type) {
                case "Temp":
                    Chiller.Query qu = Chiller.Query.valueOf(subtype);
                    if (!qu.getIsTemp()) {
		        throw new IllegalArgumentException("bad subtype");
                    }
                    break;
	        case "Flow":
                    Chiller.Query.valueOf(subtype);
                    break;
	        case "Pres":
                    Chiller.Query.valueOf(subtype);
                    break;
	        case "Fill":
                    Chiller.Query.valueOf(subtype);
                    break;
	        case "HC":
                    if (!subtype.equals("Heat") && !subtype.equals("Cool")) {
			throw new IllegalArgumentException("bad subtype");
		    } else {
		        readHeatCool = true;
		    }
                    break;
                case "Life":
                    Chiller.Life.valueOf(subtype);
                    break;
	        case "Register":
                    if (!subtype.equals("Status")) {
			throw new IllegalArgumentException("bad subtype");
                    }
                    break;
	        case "ColdPlate":
                    if (!subtype.equals("Average") && !subtype.equals("Inlet")
                        && !subtype.equals("Outlet")) {
			throw new IllegalArgumentException("bad subtype");
                    }
		    break;
                default:
                    ErrorUtils.reportChannelError(LOG, name, "type", type);
            }
        }
        catch (IllegalArgumentException e) {
            ErrorUtils.reportChannelError(LOG, name, type + " subtype ",
					subtype);
        }
        chanKey++;
        return new int[]{chanKey, 0};
    }

   /**
    *  Update a coldplate temperature (from a listened-to subsystem)
    *
    *  @param  double temperature
    *  @param  CCSTimestamp
    */
    void updateColdplateTemp(double value, CCSTimeStamp ts) {
        coldplateTemperature = new Measurement(value, ts);
    }
    void updateColdInletTemp(double value, CCSTimeStamp ts) {
        coldInletTemperature = new Measurement(value, ts);
    }
    void updateColdOutletTemp(double value, CCSTimeStamp ts) {
        coldOutletTemperature = new Measurement(value, ts);
    }

   /**
    *  Reads the monitor channel group.
    *
    *  This method is to be overridden for devices that reads some (or all)
    *  of its monitor channels at one time.  Results should be saved
    *  and then returned (one at a time) by readChannel method.
    *
    * @param group The group to be read;
    */
    @Override
    protected void readChannelGroup(String group)
    {
        if (!isOnline()) return;
        if ( group.equals("heatCool") ) {
            if (readHeatCool) {
                try {
                    heatCool = chiller.getHeatCool();
	        }
                catch (DriverTimeoutException et) {
                    LOG.severe("Chiller timeout reading HeatCool:  " + et);
                    setOnline(false);
                    lastState = ChillerState.OFFLINE;
                    if (lastState !=stateService.getState(ChillerState.class)) {
                        stateService.updateAgentState(lastState);
		    }
                }
                catch (DriverException e) {
                    LOG.severe("Chiller exception reading HeatCool:  " + e);
                }
            }
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  Channel ch  
    *  @return The read value or NaN
    */
    @Override
	protected double readChannel(Channel ch)
    {
        double value = super.readChannel(ch);   //NaN
        if (!isOnline()) return value;
        String itemType = ch.getTypeStr();
        String item = ch.getSubTypeStr();
        if (itemType.equals("HC")) {     // value read in readChannelGroup()
            if (item.equals("Heat")) {
                value = heatCool[0];
            } else if (item.equals("Cool")) {
                value = heatCool[1];
            }
        } else {
            try {
    	        switch (itemType) {
                    case "Temp":
                        // Set probe-2 value only in DUT mode
                        // and only after second time DUT is sent to chiller.
                        if (!item.equals("T_PROBE2") || sentDut >= 2) {
                            double tempValue = 
                            chiller.getTemperature(Chiller.Query.valueOf(item));
                            if (tempValue < 699.) {   // 700. means bad value
                                value = tempValue;
                            }
                            if (item.equals("SET_POINT")) {
                                lastSet = value;
			    }
                            if (item.equals("DYNAMIC_SET")) {
                                lastDyn = value;
			    }
                        }
                        break;
                    case "Flow":
                        value = chiller.getFlow(Chiller.Query.valueOf(item));
                        break;
                    case "Pres":
                        value = chiller.getPressure(Chiller.Query.valueOf(item));
                        break;
                    case "Fill":
                        value = chiller.getFill(Chiller.Query.valueOf(item));
                        break;
                    case "Life":
                        value = chiller.getLifetime(Chiller.Life.valueOf(item));
                        break;
		    case "ColdPlate":
                        if (item.equals("Average") && coldplateTemperature != null) {
                            value = coldplateTemperature.getValue();
                        }
                        if (item.equals("Inlet") && coldInletTemperature != null) {
                            value = coldInletTemperature.getValue();
                        }
                        if (item.equals("Outlet") && coldOutletTemperature != null) {
                            value = coldOutletTemperature.getValue();
                        }
                        break;
		    case "Register":
                        if (item.equals("Status")) {
                            int status = chiller.getStatusReg();
                            value = (double) status;
                            ErrorWords errorWords = chiller.getErrorWords();
                            // Temporary:  
                            // turn off ramp timeout warning if not ramping
                            if (lastDyn == lastSet) {
                                errorWords = new ErrorWords(errorWords.getError1(), errorWords.getError2(), errorWords.getWarning1()&~WARN_RAMP_TIMEOUT, errorWords.getWarning2());
                            }
                            checkForErrors(status, errorWords);
                            ChillerState st = getState(status);
                            if (st != lastState) {
                                stateService.updateAgentState(st); 
                                lastState = st;
                                if (st == ChillerState.SETPOINT) {
                                    // Chiller state newly at setpoint
                                    expectSetpoint = true;
                                    if (departedSetPoint) {
                                        alert.raiseAlert(ChillerAlerts.DEPARTED_SETPT.newAlert(), AlertState.NOMINAL, "");
                                        departedSetPoint = false;
                                    }
                                } else if (expectSetpoint) {
                                    // Chiller unexpectedly departed setpoint
                                    alert.raiseAlert(ChillerAlerts.DEPARTED_SETPT.newAlert(), AlertState.WARNING, "Chiller left SETPOINT for unknown reason");
                                    departedSetPoint = true;
                                    expectSetpoint = false;
                                }
                            }
                        }
                        break;
    	        }
            }        
            catch (DriverTimeoutException et) {
                LOG.severe("Chiller timeout reading data " + item + ": " + et);
                setOnline(false);
                lastState = ChillerState.OFFLINE;
                if (lastState != stateService.getState(ChillerState.class)) {
                    stateService.updateAgentState(lastState);
		}
            }
            catch (DriverException e) {
                LOG.severe("Chiller exception reading data " + item + ": " + e);
            }
        }
        return value;
    }

   /**
    *  Send latest cold plate temperature as DUT data to Chiller
    *
    */
    void sendDUTData() {
        if (isOnline()) {
             try {
                 chiller.writeDUT(coldplateTemperature.getValue());
                 sentDut++;
                 if (sentDut == 1) {
                     LOG.info("Sending first DUT temperature to Chiller, " +
			      Double.toString(coldplateTemperature.getValue()));
		 }
	     }
             catch (DriverException ex) {
		 LOG.severe("Driver Exception "+ ex +" while sending DUT data");
                 setOnline(false);
	     }
        }
    }

   /**
    *  Get current chiller state
    *
    *  @param  (int) ChillerStatus read from Chiller
    *  #return ChillerState
    */
    private ChillerState getState(int status) {
        ChillerState state = ChillerState.OFF;
        if ((status & Chiller.StatusRegister.AT_SETPT.getMask()) != 0) {
            state = ChillerState.SETPOINT;
	} else if ((status & Chiller.StatusRegister.T_CONTROL.getMask()) != 0) {
            state = ChillerState.CONTROLLING;
	} else if ((status & Chiller.StatusRegister.CMPRS_ON.getMask()) != 0) {
            state = ChillerState.IDLE;
        }
        return state;
    }

   /**
    *  Get temperatire-control mode
    *
    *  @return boolean  true if DUT mode, false if Normal ode
    */
    public boolean getTempControlMode() {
        return dutMode;
    }

   /**
    *  Check for chiller errors or warnings; if changed. raise Alert.
    *
    *  Clear error/warning latches if any are set.  After Alert returns
    *  to NOMINAL, clear error screen on Chiller controller.
    *
    *  @param  Current StatusRegister value
    *  @param  Current value of ErrorWords
    */
    private void checkForErrors(int status, ErrorWords errorWords) {
        boolean error = (status & Chiller.StatusRegister.ERROR.getMask())!=0;
        boolean warning = (status & Chiller.StatusRegister.WARNING.getMask())!=0
            && (errorWords.getWarning1() !=0 ||
                errorWords.getWarning2() !=0);
        // Second requirement allows for temporary fix to WARN_RAMP_TIMEOUT bit
        AlertState errorAlert;
        if (error) {
            errorAlert = AlertState.ALARM;
	} else if (warning) {
            errorAlert = AlertState.WARNING;
        } else {
            errorAlert = AlertState.NOMINAL;
        }
        boolean raise =  (errorAlert != lastErrorAlert);
        if (!raise && (errorAlert != AlertState.NOMINAL)) {
            if (!errorWords.equals(lastErrorWords)) {
	        raise = true;
            }
        }
        
        if (raise) {
            alert.raiseAlert(ChillerAlerts.CHILLER_ERR.newAlert(),
                             errorAlert, "Chiller " + errorWords.toString());
            if (errorAlert == AlertState.NOMINAL) {
                try {
                    chiller.clearGuiErrorScreen();  // clear error screen
                }
                catch (DriverException e) {
                    LOG.severe("Chiller exception clearing error screen: "+ e);
                }
            }
        }
        if (errorAlert != AlertState.NOMINAL) {
            try {
                chiller.clearErrors();  // Removes stale error conditions
            }
            catch (DriverException e) {
                LOG.severe("Chiller exception clearing errors: " + e);
            }
        }

        lastErrorWords = errorWords;
        lastErrorAlert = errorAlert;
        if (error) expectSetpoint = false;
        return;
    }

    /* Query commands for command-line usage */

   /**
    *  List queries
    *
    *  @return  Table listing enumeraed queries with descriptions
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL,
             name = "listQueries", description = "List available queries")
    public String listQueries() {
        return chiller.listQueries();
    }

   /**
    *  List status bits
    *
    *  @return  Table listing enumeraed status bits with descriptions
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL,
             name = "listStatusBits", description = "List status register bits")
    public String listStatusBits() {
        return chiller.listStatusBits();
    }

   /**
    *  Get last chiller alert (CHILLER_ERR) state
    *
    *  @return AlertState
    */
    public AlertState getLastErrorAlert() {
        return lastErrorAlert;
    }

   /**
    *  Get last chiller error words
    *
    *  @return ErrorWords
    */
    public ErrorWords getLastErrorWords() {
        return lastErrorWords;
    }

   /**
    *  Send query to Chiller and show response
    *
    *  @param  quantity (enumerated Query)
    *  @return String
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, name="queryChiller", 
             description = "send specified query to chiller")
    public String queryChiller(@Argument(description = "Read Enumerated quantity from Chiller") Chiller.Query quantity) throws DriverException {
        String response = "";
        try {
            response = chiller.queryChiller(quantity);
	} 
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
        return response;
    }

   /**
    *  Read a specified Chiller setup parameter
    *
    *  @param  index of setup parameter
    *  @return String (just the parameter value)
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL,
             name = "readAnyParameter", 
             description = "Read value of specified parameter")
    public String readAnyParameter(@Argument(description = "index of parameter") int index) throws DriverException {
        String response = "";
        try {
            String str = chiller.readParameter(index);
            response = str.substring(str.lastIndexOf(' ')+1);
	} 
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
        return response;
    }

   /**
    *  List settable (enumerated) Chiller parameters
    *
    *  @return  Table listing enumeraed parameter-setting commands
    */
    @Command(type=Command.CommandType.QUERY, level=NORMAL,
             name = "listSettableParameters",
             description = "List parameters available for public setting")
    public String listSettableParameters() {
        return chiller.listNamedParameters();
    }

   /**
    *  Read a Chiller parmeter specified by name
    *
    *  @param  FParam (parameter name)
    *  @return String
    *  @throws Driver Exception
    */
    @Command(type=Command.CommandType.QUERY, level=NORMAL,
             name = "readNamedParameter",
             description = "Read value of named parameter")
    public String readNamedParameter(@Argument(description = "enum of parameter")
     Chiller.FParam param) throws DriverException {
        String response = "";
        try {
            String str = chiller.readNamedParameter(param);
            response = str.substring(str.lastIndexOf(' ')+1);
        }
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
	return response;
    }

   /**
    *  Show event register bits.  Warning: this clears that register.
    *
    *  @return String listing any set bits
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_EXPERT, 
             name = "showEvtReg", description = "Show decoded Event Register")
    public String showEvtReg() throws DriverException {
        return chiller.decodeEvtReg();
    }

   /**
    *  Show status register bits
    *
    *  @return String listing any set bits
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name = "showStatusReg", description="Show decoded Status Register")
    public String showStatusReg() throws DriverException {
        return chiller.decodeStatusReg();
    }

   /**
    *  Show error and warning words in hex Ascii
    *
    *  @return String
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name = "showErrorWords", 
             description = "Show error and warning words in hex")
    public String showErrorWords() throws DriverException {
        return chiller.getErrorWords().toString();
    }

   /**
    *  Get the current ChillerState
    *
    *  @return ChillerState
    */
    public ChillerState getChillerState() {
        return lastState;
    }

   /**
    *  Show current chiller state and whether Gui is locked
    *
    *  @return String
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name = "showChillerState", description = "Show ChillerState")
    public String showChillerState() {
        ChillerState st = getChillerState();
        return (st.toString() + ":  " + st.getDescription() +
                ",   Controller Gui " + (guiLocked ? "locked" : "unlocked")); 
    }

   /**
    *  Commands to list bit assignments for chiller errors and warnings
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name="listError1Bits")
    public String listError1Bits() {
        return ChillerErrorTables.getError1Table();
    }

    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name="listError2Bits")
    public String listError2Bits() {
        return ChillerErrorTables.getError2Table();
    }

    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name="listWarning1Bits")
    public String listWarning1Bits() {
        return ChillerErrorTables.getWarning1Table();
    }

    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name="listWarning2Bits")
    public String listWarning2Bits() {
        return ChillerErrorTables.getWarning2Table();
    }

   /**
    *  Gets the default ramp rate.
    *  Temporary:  block parmeter-read if device not online.
    * 
    *  @return The ramp rate (deg/minute)
    */
    public double getDefaultRamp() {
        double value = Double.NaN;
        if (isOnline()) {
            try {
                value = Double.valueOf(readNamedParameter(Chiller.FParam.RAMP_DEFAULT));
            }
            catch (DriverException | NumberFormatException e) {
            }
        }
	return value;
    }

    /**
    *  Method to check for Chiller command validity after a timeout
    *
    *  Because a rejected command to the Chiller results in no response,
    *  the only symptom may be a DriverTimeoutException.  This method
    *  tests whether the flags specifying either an invalid command
    *  (probably a coding mistake) or bad arguments are set.  If so, it
    *  throws a new Exception indicating that; if not, it rethrows
    *  the DriverTimeoutException.  Note that the latter may be null,
    *  meaning the timeout was a wait time rather than an Exception.
    *
    *  @param  DriverTimeoutException
    *  @throws DriverException
    */
    private void checkCommand(DriverTimeoutException ex) throws DriverException
    {
        String eventReg = "";
        try {
            eventReg = chiller.decodeEvtReg();
        }
        catch (Exception e) {}   //Ignore Exception on reading event register
        if (eventReg.contains("BAD_CMND")) {
            throw new DriverException("Chiller command not recognizw");
	} else if (eventReg.contains("BAD_ARG")) {
            throw new IllegalArgumentException("Bad argument(s) to Chiller comand");
        } else if (ex != null) {
            throw ex;
        }
    }

    /*  Parameter-setting Commands  */

   /**
    *  Command to execute a parameter-setting command, enumerared in
    *  Chiller.FParam, checking that value gets set as requested.
    *
    *  @param  Chiller.FParam (specifies parameter)
    *  @param  double (value to set)
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name = "setParam",
             description = "Invoke Chiller command to set named parameter")
     public void setParam(
      @Argument(description = "enumerated parameter") Chiller.FParam param,
      @Argument(description = "parameter value") double value)
     throws DriverException {
	try {
            chiller.setParamCommand(param, value);
        }
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
        // After setting, publish all named parameters
        publishParameters();
        return;
    }

   /**
    *  Command to set chiller tank pressure
    *
    *  @param  double Tank pressure setting (psig)
    *  @throws DriverException
    */
    //@Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
    //         name="setTankPressure",description="Set tank pressure in psig")
    public void setTankPressure(@Argument(description = "Tank pressure set value in psig") double value) throws DriverException {
        setParam(Chiller.FParam.TANK_SETPOINT, value);
    }

   /**
    *  Select DUT vs Normal temperature-control mode
    *
    *  @param  boolean true for DUT, false for Normal
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name = "selectDUT", autoAck = false,
             description = "Select DUT vs Normal temperature-control mode")
    public void selectDUT(@Argument(description="true for DUT") boolean dut)
     throws DriverException {
        subsys.helper()
	.precondition(!(lastState==ChillerState.SETPOINT || 
                        lastState==ChillerState.CONTROLLING), 
                      "Isssue quitControllingTemperature before "
                      + "changing temperature-control mode")
        .action(() -> {
            setParam(Chiller.FParam.T_CTRL_MODE, dut ? 1. : 0.);
            if (dut && !dutMode) {
                sendDUTData();   // Send an initial valid temperature to Chiller
            } else if (!dut && dutMode) {
                sentDut = 0;     // restart sendDUTData counter
            }
            dutMode = dut;
        });
    }

    /**
     * If in DUT mode, this method stops controlling and turns off DUT mode
     *
     * @throws DriverException
     */
    void shutdownDUT() throws DriverException {
        if (dutMode) {
            quitControllingTemperature();
            setParam(Chiller.FParam.T_CTRL_MODE, 0);
            dutMode = false;
            sentDut = 0;
        }
    }

   /**
    *  Command to save current Chiller parameters to active non-volatile memory
    *
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name="saveChillerParameters",
             description="save all current chiller F# parameters to active NVM")
    public void saveChillerParameters() throws DriverException {
        chiller.saveParams();
    }

   /**
    *  Command to load all Chiller parameters from active non-volatile memory
    *
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name="loadChillerParameters",
             description="load all chiller F# parameters from active NVM")
    public void loadChillerParameters() throws DriverException {
        chiller.loadParams();
    }

    /*  Chiller-control Commands  */

   /**
    *  Set chiller temperature and go there using default ramp
    *
    *  @param  double temperature in degrees
    *  @throws DriverException
    */
    //@Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
    //         name = "setTemperature",
    //         description = "go to temperature setting using default ramp")
    public void setTemperature(@Argument(description="tempeature in degrees")
                               double temperature) throws DriverException {
        expectSetpoint = false;
        try {
            chiller.setTemperature(temperature);
        }
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
    }

   /**
    *  Set chiller temperature and go there using provided ramp rate
    *
    *  @param  double temperature in degrees
    *  @param  double ramp in degrees/minute
    *  @throws DriverException
    */
    //@Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
    //         name = "setTemperatureWithRamp",
    //         description = "go to temperature setting using provided ramp")
    public void setTemperatureWithRamp(
        @Argument(description="tempeature in degrees") double temperature,
        @Argument(description="ramp in degrees/min") double ramp)
        throws DriverException {
        expectSetpoint = false;
        try {
            chiller.setTemperatureWithRamp(temperature, ramp);
        }
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
    }

   /**
    *  Quit controlling temperature
    *
    *  @throws DriverException
    */
    //@Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
    //         name="quitControllingTemperature",
    //         description="stop controlling temperature")
    public void quitControllingTemperature() throws DriverException {
        expectSetpoint = false;
        chiller.quitControl();
    }

   /**
    *  Set chiller flow rate
    *
    *  @param  double flow (e.g., gallons per monute)
    *  @throws DriverException
    */
    //@Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
    //         name = "setFlow",
    //         description = "set flow rate (gpm) of chilled fluie")
    public void setFlow(@Argument(description="flow rate in gpm")
                        double flow) throws DriverException {
        try {
            chiller.setFlow(flow);
        }
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
    }

   /**
    *  Lock or unlock controller Gui.
    *
    *  @param  boolean
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name = "setGuiLock", description = "lock or unlock controller Gui")
    public void setGuiLock(@Argument(description="<true|false> to <lock|unlock Chiller controller Gui") boolean lock) throws DriverException {
        chiller.lockGui(lock);
        guiLocked = lock;
    }

   /**
    *  Get state of Gui lock
    *
    *  @return boolean
    */
    boolean isGuiLocked() {
        return guiLocked;
    }

   /**
    *  Clear chiller errors and warnings (for emergency use only)
    *  [Monitoring routinely clears errors, this might interfere.]
    *
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING_ROUTINE, 
             name="emergencyCleaErrors",
             description="clear chiller errors and warnings")
    public void emergencyClearErrors() throws DriverException {
        chiller.clearErrors();
        chiller.clearGuiErrorScreen();
    }

   /**
    *  Publish chiller parameter values 
    */
    public void publishParameters() {
        KeyValueDataList list = new KeyValueDataList();
        try {
            for (Chiller.FParam param : Chiller.FParam.values()) {
                String path = "FParam/" + param.toString();
                String str = chiller.readNamedParameter(param);
                String substr = str.substring(str.lastIndexOf(' ')+1);
                double value = Double.parseDouble(substr);
                list.addData(path, value);
            }
            subsys.publishSubsystemDataOnStatusBus(list);
        } 
        catch (DriverException ex) {
            LOG.severe("Driver exception while reading chiller parameters for publication: " + ex);
	}
    }

}
