package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.List;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20DiscControl;
import org.lsst.ccs.subsystem.refrig.constants.ChillerLatches;
import org.lsst.ccs.subsystem.refrig.constants.ChillerSwitches;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.ChillerPlcTestState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Implements the chiller PLC test subsystem.
 *
 */
public class ChillerPlcTest extends Subsystem implements HasLifecycle, AgentPresenceListener {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private ChillerPlutoDevice devPluto;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Maq20DiscControl maq20Ctrl[];

    private static final Logger LOG = Logger.getLogger(ChillerPlcTest.class.getName());
    private final ChillerPlcTestState controlState = new ChillerPlcTestState();
    private boolean gotCommand;

    /**
     *  Constructor.
     */
    public ChillerPlcTest() {
        super("chiller", AgentInfo.AgentType.WORKER);
    }

    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        /* Set up AgentPeriodicTask to update chiller control state for gui */
        AgentPeriodicTask ucs = new AgentPeriodicTask("updateCtrlState", () -> updateControlState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(ucs);

    }

   /**
    *  init() method
    */
    @Override
    public void init() {
        // Set a property to define that this Agent is a PLC test subsystem.
        propertiesService.setAgentProperty("ChillerPlcTest", ChillerPlcTest.class.getCanonicalName());

        //Check that required devices have been defined
        if (devPluto == null) {
            ErrorUtils.reportConfigError(LOG, name, "Pluto device", "not specified");
        }
        if (maq20Ctrl == null) {
            ErrorUtils.reportConfigError(LOG, name, "Maq20 controls", "not specified");
        }
        if (maq20Ctrl.length < 2) {
            ErrorUtils.reportConfigError(LOG, name, "Maq20 controls", "must have at least two elements");
        }
    }
    
    /**
     * postStart (must follow Device.start())
     */
    @Override
    public void postStart() {
    }

    /*  shutdown method  */
    @Override
    public void shutdown() {
    }

    /**
     *  Gets the state of the chiller system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The thermal control state
     */
    @Command(type=Command.CommandType.QUERY, description="Get the chiller state", level=0)
    public ChillerPlcTestState getControlState()
    {
        return controlState;
    }    

    /**
     *  Blocks/enables the chiller operation
     * 
     *  @param  on  Whether enabling
     */
    @Command(type=Command.CommandType.ACTION, description="Block or enable chiller operation")
    public void blockChiller(boolean on)
    {
        gotCommand = true;
        devPluto.setSwitchOn(ChillerSwitches.SW_ENABLE_CHILLER, on);
    }

    /**
     *  Gets the list of latched PLC condition names.
     *
     *  @return  The condition names.
     */
    @Command(type=Command.CommandType.QUERY, description="Get latched PLC condition names", level=0)
    public List<String> getPlcLatchNames()
    {
        return ChillerLatches.getNames();
    }

    /**
     *  Clears a (named) latched PLC condition.
     *
     *  @param  cond  The condition name.
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Clear a latched PLC condition")
    public void clearPlcLatch(@Argument(description="The condition name") String cond) throws RefrigException
    {
        gotCommand = true;
        devPluto.clearLatch(ChillerLatches.getId(cond));
    }

    /**
     *  Set a latched PLC condition via a MAQ20 DIOL module
     * 
     *  @param  cond  The condition number (0 - 9)
     *  @param  on    Whether to set
     *  @throws  RefrigException 
     */
    @Command(type=Command.CommandType.ACTION, description="Set a latched PLC condition")
    public void setPlcLatch(@Argument(description="The condition number") int cond,
                            @Argument(description="Whether on or off") boolean on) throws RefrigException
    {
        gotCommand = true;
        try {
            maq20Ctrl[cond / 5].setLineOn(cond % 5, on);
        }
        catch (DriverException e) {
            throw new RefrigException("Error setting MAQ20 line: " + e);
        }
    }
    
    /**
     *  Updates the chiller control state periodically.
     */
    private void updateControlState()
    {
        boolean changed = gotCommand;
        for (int cond = 0; cond < 10; cond++) {
            Boolean on = maq20Ctrl[cond / 5].isLineOn(cond % 5);
            SwitchState state = on == null ? SwitchState.OFFLINE : on ? SwitchState.ON : SwitchState.OFF;
            if (state != controlState.getSwitchState(cond)) {
                controlState.setSwitchState(cond, state);
                changed = true;
            }
        }
        changed |= devPluto.updateState(controlState.getPlcState());
        if (changed) {
            publishControlState();
        }
        gotCommand = false;
    }

    /**
     *  Publishes the state of the chiller subsystem.
     *
     *  This is intended to be called whenever any element of the state is changed.
     */
    private void publishControlState()
    {
        publishSubsystemDataOnStatusBus(new KeyValueData(ChillerPlcTestState.KEY, controlState));
    }

}
