package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;

/**
 *  Manages refrigeration cold compressor limits.
 *
 *  @author Owen Saxton
 */
public class ColdCompLimits extends CompLimits {

    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile int discPressDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double liquidTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double liquidTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile int liquidTempDelayTime;


    /**
     *  Sets the liquid temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempImmedLimit(double value)
    {
        liquidTempImmedLimit = value;
        limitData[CompSwConds.SWC_LIQUID_TEMP].immedLimit = value;
    }


    /**
     *  Sets the liquid temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayLimit(double value)
    {
        liquidTempDelayLimit = value;
        limitData[CompSwConds.SWC_LIQUID_TEMP].delayLimit = value;
    }


    /**
     *  Sets the liquid temperature delay time.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayTime(int value)
    {
        liquidTempDelayTime = value;
        limitData[CompSwConds.SWC_LIQUID_TEMP].delayTime = 1000 * value;
    }


    /**
     *  Sets the discharge pressure delay time.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressDelayTime(int value)
    {
        discPressDelayTime = value;
        limitData[CompSwConds.SWC_DISC_PRESS].delayTime = 1000 * value;
    }

}
