package org.lsst.ccs.subsystem.refrig;

import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;

/**
 *  Manages refrigeration compressor limits.
 *
 *  @author Owen Saxton
 */
public class CompLimits implements HasLifecycle {

    public static class LimitData {

        protected boolean isLower = false;   // Is lower limit, not upper
        protected boolean noShutoff = false; // Doesn't shut off the compressor
        protected double immedLimit = 0.0;   // The immediate limit, or NaN if not applicable
        protected double delayLimit = 0.0;   // The delayed limit, or NaN if not applicable
        protected int delayTime = 0;         // The delay time (ms), or -1 if 8-hour on

    }

    /**
     *  Constants.
     */
    protected static final String COMP_LIMITS = "CompLimits";

    /**
     *  Data fields.
     */
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double discPressImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double discPressDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double discTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double discTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile int discTempDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double suctTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double cmprPowerImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double cmprPowerDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile int cmprPowerDelayTime;

    protected final LimitData[] limitData = new LimitData[CompSwConds.NUM_SW_CONDITIONS];


    /**
     *  Constructor.
     */
    public CompLimits()
    {
        for (int j = 0; j < CompSwConds.NUM_SW_CONDITIONS; j++) {
            limitData[j] = new LimitData();
        }
    }


    /**
     *  Initializes the compressor limits.
     */
    @Override
    public void postInit()
    {
        limitData[CompSwConds.SWC_SUCT_TEMP].isLower = true;
        limitData[CompSwConds.SWC_SUCT_TEMP].delayLimit = Double.NaN;
    }


    /**
     *  Sets the discharge pressure immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressImmedLimit(double value)
    {
        discPressImmedLimit = value;
        limitData[CompSwConds.SWC_DISC_PRESS].immedLimit = value;
    }


    /**
     *  Sets the discharge pressure delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressDelayLimit(double value)
    {
        discPressDelayLimit = value;
        limitData[CompSwConds.SWC_DISC_PRESS].delayLimit = value;
    }


    /**
     *  Sets the discharge temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempImmedLimit(double value)
    {
        discTempImmedLimit = value;
        limitData[CompSwConds.SWC_DISC_TEMP].immedLimit = value;
    }


    /**
     *  Sets the discharge temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempDelayLimit(double value)
    {
        discTempDelayLimit = value;
        limitData[CompSwConds.SWC_DISC_TEMP].delayLimit = value;
    }


    /**
     *  Sets the discharge temperature delay time.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscTempDelayTime(int value)
    {
        discTempDelayTime = value;
        limitData[CompSwConds.SWC_DISC_TEMP].delayTime = 1000 * value;
    }


    /**
     *  Sets the suction temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setSuctTempImmedLimit(double value)
    {
        suctTempImmedLimit = value;
        limitData[CompSwConds.SWC_SUCT_TEMP].immedLimit = value;
    }


    /**
     *  Sets the compressor power immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerImmedLimit(double value)
    {
        cmprPowerImmedLimit = value;
        limitData[CompSwConds.SWC_CMPR_POWER].immedLimit = value;
    }


    /**
     *  Sets the compressor power delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerDelayLimit(double value)
    {
        cmprPowerDelayLimit = value;
        limitData[CompSwConds.SWC_CMPR_POWER].delayLimit = value;
    }


    /**
     *  Sets the compressor power delay time.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setCmprPowerDelayTime(int value)
    {
        cmprPowerDelayTime = value;
        limitData[CompSwConds.SWC_CMPR_POWER].delayTime = 1000 * value;
    }

   
    /**
     *  Gets the limit data
     * 
     *  @return  The array of limit data
     */
    public LimitData[] getLimitData()
    {
        return limitData;
    }

}
