package org.lsst.ccs.subsystem.refrig;

import java.util.Map;
import java.util.TreeMap;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Handles the simulated MAQ20 device.
 *
 *  @author Owen Saxton
 */
public class SimMaq20Device extends Maq20Device {

    private final Map<String, Integer> channelMap = new TreeMap<>();


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
        super.initDevice();
        Maq20.ModuleType[] modTypes = new Maq20.ModuleType[getModuleCount()];
        for (int j = 0; j < modTypes.length; j++) {
            modTypes[j] = getModuleData(j).modDef.type;
        }
        maq.initSimulation(modTypes, serials);
        fullName = path + " (Simulated Maq20 system)";
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  ch  The Channel to check
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if there is a parameter error
     */
    @Override
    protected int[] checkChannel(Channel ch) throws Exception
    {
        int[] result = super.checkChannel(ch);
        channelMap.put(ch.getPath(), ch.getHwChan());
        return result;
    }


    @Command(type=Command.CommandType.ACTION, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chanName,
                                @Argument(description="Channel value") double value) throws RefrigException
    {
        Channel chan = mon.getChannel(chanName);
        Integer hwChan = channelMap.get(chanName);
        if (chan == null || hwChan ==  null) {
            throw new RefrigException("Invalid channel name: " + chanName);
        }
        double offset = chan.convertRawValue(0.0);
        double scale = chan.convertRawValue(1.0) - offset;
        try {
            maq.setSimChannelValue(getModuleData(chan.getType() & 0xff).modId, hwChan, (value - offset) / scale);
        }
        catch (DriverException e) {
            throw new RefrigException(e);
        }
    }

    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
