package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;
import org.lsst.ccs.subsystem.refrig.constants.PcpSwitches;

/**
 *  Handles a SLAC heater power supply.
 *
 *  @author Owen Saxton
 */
public class PcpPowerDevice extends PowerDevice {

    /**
     *  Constants.
     */
    public static final int
        N_HW_CHANS = 3;

    private static final int
        TYPE_VOLTAGE     = 0,
        TYPE_CURRENT     = 1,
        TYPE_POWER       = 2;

    /**
     *  Private lookup maps.
     */
    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("VOLTAGE", TYPE_VOLTAGE);
        typeMap.put("CURRENT", TYPE_CURRENT);
        typeMap.put("POWER", TYPE_POWER);
    }

    private static final Map<Integer, Integer> plutoSwitches = new HashMap<>();
    static {
        plutoSwitches.put(0, PcpSwitches.SW_ENABLE_HEAT_PS1);
        plutoSwitches.put(1, PcpSwitches.SW_ENABLE_HEAT_PS2);
        plutoSwitches.put(2, PcpSwitches.SW_ENABLE_HEAT_PS3);
    }

    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected Maq20Device maqDevc;
    @LookupField(strategy=LookupField.Strategy.SIBLINGS)
    private final List<PcpPowerControl> controlList = new ArrayList<>();
    @LookupField(strategy=LookupField.Strategy.TREE)
    private PcpPlutoDevice plutoDevc;

    private static final Logger LOG = Logger.getLogger(PcpPowerDevice.class.getName());
    private final PcpPowerControl[] controls = new PcpPowerControl[N_HW_CHANS];
    //private final double[] setVoltages = new double[N_HW_CHANS]; // Set voltages
    private final double[] voltages = new double[N_HW_CHANS]; // Read voltages
    private final double[] currents = new double[N_HW_CHANS]; // Read currents
    //private final boolean[] outputs = new boolean[N_HW_CHANS]; // Set output states


    /**
     *  Constructor.
     */
    public PcpPowerDevice()
    {
        super("PcpPower", null, 0, 0, N_HW_CHANS - 1);
        connType = DriverConstants.ConnType.NET;
        devcId = "";
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        for (PcpPowerControl ctrl : controlList) {
            controls[ctrl.getIdent()] = ctrl;
        }
        for (int j = 0; j < controls.length; j++) {
            if (controls[j] == null) {
                ErrorUtils.reportConfigError(LOG, path, "PcpPowerControl object with ident " + j, "has not been defined");
            }
        }
        if (plutoDevc == null) {
            ErrorUtils.reportConfigError(LOG, path, "PcpPlutoDevice object", "has not been defined");
        }
        fullName = getPath() + " (Pcp load power supply)";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(maqDevc.isOnline());
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string (not used)
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        if (hwChan < 0 || hwChan >= N_HW_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "HW channel", hwChan);
        }
        return new int[]{iType, 0};
    }


    /**
     *  Reads all heater channels as a group.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!isOnline()) return;
        try {
            for (int chan = 0; chan < N_HW_CHANS; chan++) {
                PcpPowerControl ctrl = controls[chan];
                currents[chan] = ctrl.readCurrent();
                voltages[chan] = ctrl.readVoltage();
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading currents & voltages from {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (isOnline()) {
            switch (type) {
            case TYPE_VOLTAGE:
                value = voltages[hwChan];
                break;
            case TYPE_CURRENT:
                value = currents[hwChan];
                break;
            case TYPE_POWER:
                value = voltages[hwChan] * currents[hwChan];
                break;
            }
        }
        return value;
    }


    /**
     *  Sets the output state.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The output state to set, true or false
     *  @throws  DriverException
     */
    @Command (type = Command.CommandType.ACTION) 
    @Override
    public void setOutput(int chan, boolean value) throws DriverException
    {
        plutoDevc.setSwitchOn(plutoSwitches.get(chan), value);
    }


    /**
     *  Gets the output state.
     *
     *  @param  chan  The hardware channel
     *  @return  The output state
     *  @throws  DriverException
     */
    @Command (type = Command.CommandType.ACTION) 
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return controls[chan].isOutputOn();
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The voltage to set
     *  @throws  DriverException
     */
    @Command (type = Command.CommandType.ACTION) 
    @Override
    public void setVoltage(int chan, double value) throws DriverException
    {
        controls[chan].setVoltage(value);
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chan  The hardware channel
     *  @return  The read voltage
     *  @throws  DriverException
     */
    @Command (type = Command.CommandType.ACTION) 
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return controls[chan].readVoltage();
    }


    /**
     *  Sets the current.
     * 
     *  @param  chan   The hardware channel
     *  @param  value  The current to set
     *  @throws  DriverException
     */
    @Command (type = Command.CommandType.ACTION) 
    @Override
    public void setCurrent(int chan, double value) throws DriverException
    {
        controls[chan].setCurrent(value);
    }


    /**
     *  Reads the current.
     *
     *  @param  chan  The hardware channel
     *  @return  The read current
     *  @throws  DriverException
     */
    @Command (type = Command.CommandType.ACTION) 
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return controls[chan].readCurrent();
    }

}
