package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.dataforth.Maq20;
import org.lsst.ccs.drivers.dataforth.Maq20Analog;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.devices.dataforth.Maq20Device;

/**
 *  Controls a Maq20-controlled power supply.
 *
 *  @author Owen Saxton
 */
public class PcpPowerControl implements HasLifecycle, Maq20Device.Maq20Control {

    private static class ChannelDesc {

        Maq20Device.ModuleData moduleData;
        int hwChannel;
        int range;
        double scale;

        private ChannelDesc(Maq20Device.ModuleData moduleData, int hwChannel, int range, double scale)
        {
            this.moduleData = moduleData;
            this.hwChannel = hwChannel;
            this.range = range;
            this.scale = scale;
        }

    }

    private static final int
        CHAN_IN_VOLTAGE  = 0,
        CHAN_OUT_VOLTAGE = 1,
        CHAN_IN_CURRENT  = 2,
        CHAN_OUT_CURRENT = 3,
        CHAN_IN_ENABLED  = 4,
        NUM_CHANNELS = 5;

    private static final Map<Integer, Double> inScaleMap = new HashMap<>();
    static {
        inScaleMap.put(Maq20Analog.RANGE_VOLT_60, 60.0);
        inScaleMap.put(Maq20Analog.RANGE_VOLT_40, 40.0);
        inScaleMap.put(Maq20Analog.RANGE_VOLT_20, 20.0);
        inScaleMap.put(Maq20Analog.RANGE_VOLT_10, 10.0);
        inScaleMap.put(Maq20Analog.RANGE_VOLT_5, 5.0);
    }

    private static final Map<Integer, Double> outScaleMap = new HashMap<>();
    static {
        outScaleMap.put(Maq20Analog.RANGE_VOUT_PM10, 10.0);
        outScaleMap.put(Maq20Analog.RANGE_VOUT_PM5, 5.0);
        outScaleMap.put(Maq20Analog.RANGE_VOUT_PM2, 2.0);
        outScaleMap.put(Maq20Analog.RANGE_VOUT_P10, 10.0);
        outScaleMap.put(Maq20Analog.RANGE_VOUT_P5, 5.0);
        outScaleMap.put(Maq20Analog.RANGE_VOUT_P2, 2.0);
    }

    @LookupName
    protected String name;
    @LookupPath
    protected String path;
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    protected Maq20Device maqDevc;

    private Integer ident;  // Identification
    private String inVoltage, outVoltage, inCurrent, outCurrent, inEnabled;  // Descriptors: modIndex:hwChan:range:fullScale
    private Double maxCurrent;  // Maximum current

    private static final Logger LOG = Logger.getLogger(PcpPowerControl.class.getName());
    private final ChannelDesc[] channelDescs = new ChannelDesc[NUM_CHANNELS];


    /**
     *  Life-cycle initialization.
     * 
     *  Checks the configuration
     */
    @Override
    public void init()
    {
        if (ident == null) {
            ErrorUtils.reportConfigError(LOG, path, "ident", "has not been specified");
        }
        if (ident < 0 || ident >= PcpPowerDevice.N_HW_CHANS) {
            ErrorUtils.reportConfigError(LOG, path, "ident", "must be non-negative and less than " + PcpPowerDevice.N_HW_CHANS);
        }
        channelDescs[CHAN_IN_VOLTAGE] = decodeChannel("inVoltage", inVoltage, false);
        channelDescs[CHAN_OUT_VOLTAGE] = decodeChannel("outVoltage", outVoltage, true);
        channelDescs[CHAN_IN_CURRENT] = decodeChannel("inCurrent", inCurrent, false);
        channelDescs[CHAN_OUT_CURRENT] = decodeChannel("outCurrent", outCurrent, true);
        channelDescs[CHAN_IN_ENABLED] = decodeChannel("inEnabled", inEnabled, false);
        if (maxCurrent == null) {
            ErrorUtils.reportConfigError(LOG, path, "currentLimit", "has not been specified");
        }
    }


    /**
     *  Post-connection initialization.
     *
     *  @throws  DriverException
     */
    @Override
    public void initialize() throws DriverException
    {
        for (ChannelDesc chanDesc : channelDescs) {
            chanDesc.moduleData.maqAna.setRange(chanDesc.hwChannel, chanDesc.range);
        }
        setCurrent(maxCurrent);
    }


    /**
     *  Pre-closure cleanup.
     */
    @Override
    public void close()
    {
    }


    /**
     *  Reads a voltage.
     *
     *  @return  The value
     *  @throws  DriverException
     */
    public double readVoltage() throws DriverException
    {
        ChannelDesc chanDesc = channelDescs[CHAN_IN_VOLTAGE];
        return chanDesc.moduleData.maqAnaIn.readValue(chanDesc.hwChannel) / chanDesc.scale;
    }


    /**
     *  Sets a voltage.
     *
     *  @param  value  The value
     *  @throws  DriverException
     */
    public void setVoltage(double value) throws DriverException
    {
        ChannelDesc chanDesc = channelDescs[CHAN_OUT_VOLTAGE];
        chanDesc.moduleData.maqAnaOut.writeValue(chanDesc.hwChannel, chanDesc.scale * value);
    }


    /**
     *  Reads a current.
     *
     *  @return  The value
     *  @throws  DriverException
     */
    public double readCurrent() throws DriverException
    {
        ChannelDesc chanDesc = channelDescs[CHAN_IN_CURRENT];
        return chanDesc.moduleData.maqAnaIn.readValue(chanDesc.hwChannel) / chanDesc.scale;
    }


    /**
     *  Sets a current.
     *
     *  @param  value  The value
     *  @throws  DriverException
     */
    public void setCurrent(double value) throws DriverException
    {
        ChannelDesc chanDesc = channelDescs[CHAN_OUT_CURRENT];
        chanDesc.moduleData.maqAnaOut.writeValue(chanDesc.hwChannel, chanDesc.scale * value);
    }


    /**
     *  Gets whether output enabled.
     *
     *  @return  Whether enabled
     *  @throws  DriverException
     */
    public boolean isOutputOn() throws DriverException
    {
        ChannelDesc chanDesc = channelDescs[CHAN_IN_ENABLED];
        return chanDesc.moduleData.maqAnaIn.readValue(chanDesc.hwChannel) < 0.5;
    }


    /**
     *  Gets the name.
     * 
     *  @return  The name
     */
    public String getName()
    {
        return name;
    }


    /**
     *  Gets the path.
     * 
     *  @return  The path
     */
    public String getPath()
    {
        return path;
    }


    /**
     *  Gets the identification.
     * 
     *  @return  The identification
     */
    public int getIdent()
    {
        return ident;
    }


    /**
     *  Decodes a channel specification string.
     * 
     *  @param  chanName  The channel name
     *  @param  channel  The channel specification
     *  @param  isOut  Whether output channel
     *  @return  A channel descriptor
     */
    private ChannelDesc decodeChannel(String chanName, String channel, boolean isOut)
    {
        int hwChannel = 0;
        Integer range;
        double fullScale = 0;
        Maq20Device.ModuleData modData = null;
        String[] words = channel.split(":");
        if (words.length != 4) {
            ErrorUtils.reportConfigError(LOG, path, chanName, "does not contain exactly four elements");
        }
        try {
            int modIndex = Integer.valueOf(words[0]);
            modData = maqDevc.getModuleData(modIndex);
            if (modData == null) {
                ErrorUtils.reportConfigError(LOG, path, chanName + " modIndex (" + modIndex + ")", "specifies non-existent module");
            }
            Maq20Device.ModuleDef modDef = modData.modDef;
            if (modDef.opType != (isOut ? Maq20.OPER_ANALOUT : Maq20.OPER_ANALOG)) {
                ErrorUtils.reportConfigError(LOG, path, chanName + " modIndex (" + modIndex + ")",
                                             "does not specify analog " + (isOut ? "output" : "input") + " module");
            }
        }
        catch (NumberFormatException e) {
            ErrorUtils.reportConfigError(LOG, path, chanName + " modIndex (" + words[0] + ")", "is not an integer");
        }
        try {
            hwChannel = Integer.valueOf(words[1]);
        }
        catch (NumberFormatException e) {
            ErrorUtils.reportConfigError(LOG, path, chanName + " hwChannel (" + words[1] + ")", "is not an integer");
        }
        range = modData.modDef.rangeMap.get(words[2]);
        if (range == null) {
            ErrorUtils.reportConfigError(LOG, path, chanName + " range (" + words[2] + ")", "is not a valid range");
        }
        
        try {
            fullScale = Double.valueOf(words[3]);
        }
        catch (NumberFormatException e) {
            ErrorUtils.reportConfigError(LOG, path, chanName + " fullScale (" + words[3] + ")", "is not a floating-point number");
        }
        return new ChannelDesc(modData, hwChannel, range, (isOut ? outScaleMap : inScaleMap).get(range) / fullScale);
    }

}
