package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.chiller.Chiller;
import org.lsst.ccs.drivers.chiller.Chiller.ErrorWords;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.commons.DriverTimeoutException;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.ChillerAlerts;
import org.lsst.ccs.subsystem.refrig.constants.ChillerState;

/* Pre-defined command levels */
import static org.lsst.ccs.command.annotations.Command.NORMAL;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING1;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING2;
import static org.lsst.ccs.command.annotations.Command.ENGINEERING3;

/**
 *  Interfaces with driver for inTEST chiller
 *
 *  @author Al Eisner
 */
public class InTESTChillerDevice extends Device {

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alert;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService stateService;
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent subsys;

   /**
    *  Data fields.
    */

    private final Chiller chiller = new Chiller();  // Associated driver

   /**
    *  Private lookup maps for driver read commands used in monitoring.
    *  The key is a numeric identifier of a monitor channel defined
    *  in the groovy file.
    *
    *  typeMap identifies the type of a quantity to be read, e.g.,
    *  "Temp" (temperature), "Flow" (flow rate), "Pres"(pressure). 
    *  "Life" (lifetime).
    *  It is entered as the type parameter in the groovy file.
    *
    *  itemMap identifies the specific enumeraed quantity ro be read. 
    *  It is taken from the subtype parameter in the groovy file.
    */
    private final Map<Integer, String> typeMap = new HashMap<>();
    private final Map<Integer, String> itemMap = new HashMap<>();
    private Integer chanKey;
    private boolean readHeatCool = false;
    private double[] heatCool;
    private ErrorWords lastErrorWords;
    private AlertState lastErrorAlert;
    private volatile ChillerState lastState;

    /* Configuration parameters related to hardware settings */

    @ConfigurationParameter(isFinal = true, description = "IP host")
    protected volatile String host;

    private static final Logger LOG = Logger.getLogger(InTESTChillerDevice.class.getName());

    /* Hard-coded hardware-initialization parameters

    // to be added; example from turbo:
    // private final int rotfreq_set_304 = 1010;

   /**
    *  Performs basic initialization.  
    *  Verify that all ConfigurationParameters are set.
    */
    @Override
    protected void initDevice() {

        fullName = "InTESTChiller " + name + " (" + host + ")";
        chanKey = 0;
        lastErrorAlert = AlertState.NOMINAL;
        lastErrorWords = new ErrorWords(0,0,0,0);
        lastState = ChillerState.CHILLER_OFF;

        if (host == null) {
            ErrorUtils.reportConfigError(LOG, name, "host", "is missing");
        }

        /* Registe alerts and state */

        alert.registerAlert(ChillerAlerts.CHILLER_ERR.newAlert());
        stateService.registerState(ChillerState.class, "Chiller State", subsys);
        stateService.updateAgentState(lastState);   // Initial value
    }


   /**
    *  Performs full initialization.
    */
    @Override
    protected void initialize() {

        try {
            chiller.open(host);

            /* Hardware initialization */

            initSensors();
            setOnline(true);
            LOG.info("\n Connected to " + fullName);
        }

        catch (DriverException e) {
            if (!inited) {
                LOG.severe("Error connecting to or initializaing " 
                          + fullName + ": " + e);
            }
            close();
        }

        inited = true;
    }

   /**
    *  Closes the connection.
    */
    @Override
    protected void close() {
        try {
            chiller.close();
        }
        catch (DriverException e) {
        }
    }

   /**
    *  Checks a channel's parameters for validity.
    *
    *  @param  name     The channel name
    *  @param  hwChan   The hardware channel number
    *  @param  type     The channel type string
    *  @param  subtype  The channel subtype string
    *  @return  A two-element array containing the encoded type [0] and
    *           subtype [1] values.
    *  @throws  Exception if any errors found in the parameters.
    */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, 
                                 String subtype) throws Exception
    {
        try {
            switch (type) {
                case "Temp":
                    Chiller.Query qu = Chiller.Query.valueOf(subtype);
                    if (!qu.getIsTemp()) {
		        throw new IllegalArgumentException("bad subtype");
                    }
                    break;
	        case "Flow":
                    Chiller.Query.valueOf(subtype);
                    break;
	        case "Pres":
                    Chiller.Query.valueOf(subtype);
                    break;
	        case "HC":
                    if (!subtype.equals("Heat") && !subtype.equals("Cool")) {
			throw new IllegalArgumentException("bad subtype");
		    } else {
		        readHeatCool = true;
		    }
                    break;
                case "Life":
                    Chiller.Life.valueOf(subtype);
                    break;
	        case "Register":
                    if (!subtype.equals("Status")) {
			throw new IllegalArgumentException("bad subtype");
                    }
                    break;
                default:
                    ErrorUtils.reportChannelError(LOG, name, "type", type);
            }
        }
        catch (IllegalArgumentException e) {
            ErrorUtils.reportChannelError(LOG, name, type + " subtype ",
					subtype);
        }
        chanKey++;
        typeMap.put(chanKey,type);
        itemMap.put(chanKey,subtype);
        return new int[]{chanKey, 0};
    }

   /**
    *  Reads the monitor channel group.
    *
    *  This method is to be overridden for devices that reads some (or all)
    *  of its monitor channels at one time.  Results should be saved
    *  and then returned (one at a time) by readChannel method.
    */
    @Override
    protected void readChannelGroup()
    {
        if (!isOnline()) return;
        if (readHeatCool) {
            try {
                heatCool = chiller.getHeatCool();
	    }
            catch (DriverTimeoutException et) {
                LOG.severe("Chiller timeout reading HeatCool:  " + et);
                setOnline(false);
            }
            catch (DriverException e) {
                LOG.severe("Chiller exception reading HeatCool:  " + e);
            }
        }
    }


   /**
    *  Reads a channel.
    *
    *  @param  hwChan   The hardware channel number.
    *  @param  type     The encoded channel type returned by checkChannel.
    *  @return The read value
    */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);   //NaN
        if (!isOnline()) return value;
        String itemType = typeMap.get(type);
        String item = itemMap.get(type);
        if (itemType.equals("HC")) {     // value read in readChannelGroup()
            if (item.equals("Heat")) {
                value = heatCool[0];
            } else if (item.equals("Cool")) {
                value = heatCool[1];
            }
        } else {
            try {
    	        switch (itemType) {
                    case "Temp":
                        value = chiller.getTemperature(Chiller.Query.valueOf(item));
                        break;
                    case "Flow":
                        value = chiller.getFlow(Chiller.Query.valueOf(item));
                        break;
                    case "Pres":
                        value = chiller.getPressure(Chiller.Query.valueOf(item));
                        break;
                    case "Life":
                        value = chiller.getLifetime(Chiller.Life.valueOf(item));
                        break;
		    case "Register":
                        if (item.equals("Status")) {
                            int status = chiller.getStatusReg();
                            value = (double) status;
                            ErrorWords errorWords = chiller.getErrorWords();
                            checkForErrors(status, errorWords);
                            ChillerState st = getState(status);
                            if (st != lastState) {
                                stateService.updateAgentState(st); 
                                lastState = st;
                            }
                        }
                        break;
    	        }
            }        
            catch (DriverTimeoutException et) {
                LOG.severe("Chiller timeout reading data " + item + ": " + et);
                setOnline(false);
            }
            catch (DriverException e) {
                LOG.severe("Chiller exception reading data " + item + ": " + e);
            }
        }
        return value;
    }

   /**
    *  Get current chiller state
    *
    *  @param  (int) ChillerStatus read from Chiller
    *  #return ChillerState
    */
    private ChillerState getState(int status) {
        ChillerState state = ChillerState.CHILLER_OFF;
        if ((status & Chiller.StatusRegister.AT_SETPT.getMask()) != 0) {
            state = ChillerState.CHILLER_SETPOINT;
	} else if ((status & Chiller.StatusRegister.T_CONTROL.getMask()) != 0) {
            state = ChillerState.CHILLER_CONTROLLING;
	} else if ((status & Chiller.StatusRegister.CMPRS_ON.getMask()) != 0) {
            state = ChillerState.CHILLER_IDLE;
        }
        return state;
    }

   /**
    *  Check for chiller errors or warnings; if changed. raise Alert
    *
    *  @param  Current StatusRegister value
    *  @param  Current value of ErrorWords
    */
    private void checkForErrors(int status, ErrorWords errorWords) {
        boolean error = (status & Chiller.StatusRegister.ERROR.getMask())!=0;
        boolean warning = (status & Chiller.StatusRegister.WARNING.getMask())!=0;
        AlertState errorAlert;
        if (error) {
            errorAlert = AlertState.ALARM;
	} else if (warning) {
            errorAlert = AlertState.WARNING;
        } else {
            errorAlert = AlertState.NOMINAL;
        }
        boolean raise =  (errorAlert != lastErrorAlert);
        if (!raise && (errorAlert != AlertState.NOMINAL)) {
            if (!errorWords.equals(lastErrorWords)) {
	        raise = true;;
            }
        }
        if (raise) {
            alert.raiseAlert(ChillerAlerts.CHILLER_ERR.newAlert(),
                             errorAlert, "Chiller " + errorWords.toString());
        }
        lastErrorWords = errorWords;
        lastErrorAlert = errorAlert;
        return;
    }

    /* Query commands for command-line usage */

   /**
    *  List queries
    *
    *  @return  Table listing enumeraed queries with descroptions
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL,
             name = "listQueries", description = "List available queries")
    public String listQueries() {
        return chiller.listQueries();
    }

   /**
    *  List status bits
    *
    *  @return  Table listing enumeraed status bits with descroptions
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL,
             name = "listStatusBits", description = "List status register bits")
    public String listStatusBits() {
        return chiller.listStatusBits();
    }

   /**
    *  Send query to Chiller and show response
    *
    *  @param  quantity (enumerated Query)
    *  @return String
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, name="queryChiller", 
             description = "send specified query to chiller")
    public String queryChiller(@Argument(description = "Read Enumerated quantity from Chiller") Chiller.Query quantity) throws DriverException {
        String response = "";
        try {
            response = chiller.queryChiller(quantity);
	} 
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
        return response;
    }

   /**
    *  Read a specified Chiller setup parameter
    *
    *  @param  index of setup parameter
    *  @return String
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL,name = "readParameter", 
             description = "Read value of specified parameter")
    public String readParameter(@Argument(description = "index of parameter") int index) throws DriverException {
        String response = "";
        try {
            response = chiller.readParameter(index);
	} 
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}
        return response;
    }

   /**
    *  Show event register bits.  Warning: this clears that register.
    *
    *  @return String listing any set bits
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING1, 
             name = "showEvtReg", description = "Show decoded Event Register")
    public String showEvtReg() throws DriverException {
        return chiller.decodeEvtReg();
    }

   /**
    *  Show status register bits
    *
    *  @return String listing any set bits
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name = "showStatusReg", description="Show decoded Event Register")
    public String showStatusReg() throws DriverException {
        return chiller.decodeStatusReg();
    }

   /**
    *  Show error and warning words in hex Ascii
    *
    *  @return String
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name = "showErrorWords", 
             description = "Show error and warning words in hex")
    public String showErrorWords() throws DriverException {
        return chiller.getErrorWords().toString();
    }

   /**
    *  Show current chiller state
    *
    *  @return String
    */
    @Command(type=Command.CommandType.QUERY,level=NORMAL, 
             name = "showChillerState", description = "Show ChillerState")
    public String showChillerState() {
        return (lastState + ":  " + lastState.getDescription());
    }

   /**
    *  Command to execute a parameter-setting command, enumerared in
    *  Chiller.SetParam, checking that value gets set as requested.
    *
    *  @param  Chiller.SetParam (specifies command to Chiller controller)
    *  @param  Arguments to send to Chiller
    *  @throws DriverException
    */
    @Command(type=Command.CommandType.ACTION,level=ENGINEERING1, 
             name = "setParams",
             description = "Invoke Chiller command tp set parameter(s)")
    public void setParams(
     @Argument(description = "enumerated command") Chiller.SetParam setParam,
     @Argument(description = "parameter value(s)") double... values)
     throws DriverException {
	try {
            chiller.setParamCommand(setParam, values);
        }
        catch (DriverTimeoutException ex) {
            checkCommand(ex);
	}

        return;
    }

   /**
    *  Method to check for Chiller command validity after a timeout
    *
    *  Because a rejected command to the Chiller results in no response,
    *  the only symptom may be a DriverTimeoutException.  This method
    *  tests whether the flags specifying either an invalid command
    *  (probably a coding mistake) or bad arguments are set.  If so, it
    *  throws a new Exception indicating that; if not, it rethrows
    *  the DriverTimeoutException.  Note that the latter may be null,
    *  meaning the timeout was a wait time rather than an Exception.
    *
    *  @param  DriverTimeoutException
    *  @throws DriverException
    */
    private void checkCommand(DriverTimeoutException ex) throws DriverException
    {
        String eventReg = "";
        try {
            eventReg = chiller.decodeEvtReg();
        }
        catch (Exception e) {}   //Ignore Exception on reading event register
        if (eventReg.contains("BAD_CMND")) {
            throw new DriverException("Chiller command not recognizw");
	} else if (eventReg.contains("BAD_ARG")) {
            throw new IllegalArgumentException("Bad argument(s) to Chiller comand");
        } else if (ex != null) {
            throw ex;
        }
    }

}
