package org.lsst.ccs.subsystem.refrig;

import java.io.Serializable;
import java.util.function.Predicate;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;

/**
 *  Implements the chiller subsystem.
 *
 */
public class ChillerSubsystem extends Subsystem implements HasLifecycle, StatusMessageListener, AgentPresenceListener {

    private static final Logger LOG = Logger.getLogger(ChillerSubsystem.class.getName());

    private volatile boolean isListeningToThermal = false;
    private volatile String thermalName = null;
    private final Object listeningLock = new Object();

    /**
     *  Constructor.
     */
    public ChillerSubsystem() {
        super("chiller", AgentInfo.AgentType.WORKER);
    }

    @Override
    public void init() {
        //Register self as an AgentPresenceListener to choose when to start
        //listening to the thermal subsystem.
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
    }
    
    @Override
    public void shutdown() {
        //Un-register self from listening to AgentPresenceManager notifications.
        getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);        
    }
    
    
    @Override
    public void onStatusMessage(StatusMessage msg) {
        //Do something here.
    }

    @Override
    public void connected(AgentInfo... agents) {
        //If the thermal subsystem connects start listening to its
        //StatusSubsystemData messages.
        //Also check that we have only one thermal subsystem
        for ( AgentInfo agent : agents ) {
            if ( agent.hasAgentProperty(RefrigAgentProperties.THERMAL_TYPE) ) {
                synchronized(listeningLock) {
                    if ( !isListeningToThermal ) {
                        isListeningToThermal = true;
                        thermalName = agent.getName();
                        LOG.log(Level.INFO,"Starting to listen to messages from thermal subsystem {0}.",thermalName);
                        Predicate<BusMessage<? extends Serializable, ?>> filter = 
                                BusMessageFilterFactory.messageOrigin(thermalName).and(BusMessageFilterFactory.messageClass(StatusSubsystemData.class));                        
                        getMessagingAccess().addStatusMessageListener(this,filter);
                    } else {
                        //We should probably raise an exception here.
                        LOG.log(Level.SEVERE,"More than one thermal subsystem on the buses!!! Currently listening to {0} and just connected {1}!!", new Object[]{thermalName,agent.getName()});
                    }
                }                
            }
        }
    }

    @Override
    public void disconnected(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            synchronized(listeningLock) {
                if ( isListeningToThermal && agent.getName().equals(thermalName) ) {
                    LOG.log(Level.INFO,"Thermal subsystem {0} disconnected. No longer listening to its messages.",thermalName);
                    getMessagingAccess().removeStatusMessageListener(this);
                    isListeningToThermal = false;
                    thermalName = null;
                } 
            }
        }        
    }    
    
    @Override
    public String vetoTransitionToNormalMode() {
        //We should not transition to Normal mode if we are not
        //listening to the thermal subsystem
        return isListeningToThermal ? null : "Chiller is not listening to Thermal.";
    }

}
