package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.refrig.data.RefrigAction;

/**
 * This class manages a refrig action client.
 *
 * It uses the AgentPresenceManager to detect which Agent on the buses is available
 * for handling action alerts.
 *
 * @author The LSST CCS Team
 */
public class RefrigActionClient implements HasLifecycle, AgentPresenceListener {

    private static final Logger LOG = Logger.getLogger(RefrigActionClient.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent agent;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @Override
    public void init()
    {
        // Agent presence listener
        agent.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        // Set a property to define agent as valid refrig client.
        propertiesService.setAgentProperty(RefrigAction.getClientName(), "");
    }

    @Override
    public void shutdown()
    {
        //Remove agent presence and alert listeners
        agent.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
    }

    @Override
    public void connecting(AgentInfo... agents)
    {
        for (AgentInfo ai : agents) {
            if (ai.hasAgentProperty(RefrigAction.getManagerName())) {
                LOG.log(Level.INFO, "Publishing raised alert summary for agent {0}", ai.getName());
                agent.getScheduler().execute(() -> {
                    alertService.publishRaisedAlertSummary();
                });
                break;
            }
        }
    }

}
