package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.refrig.constants.ColdState;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.constants.ThermalAlert;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterOpState;
import org.lsst.ccs.subsystem.refrig.data.RefrigAction;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;
import org.lsst.ccs.subsystem.refrig.data.RefrigUtils;

/**
 *  Implements the refrigeration thermal control subsystem.
 *
 *  @author Owen Saxton
 */
public class ThermalMain extends Subsystem implements HasLifecycle, AgentPresenceListener, StatusMessageListener {

    /**
     *  Inner class for holding trim heater parameters.
     */
    static class TrimHeater {

        TrimHeaterControl tempCtrl;
        PowerDevice powerDevc;
        int powerChan;
        boolean inited = false;

        TrimHeater(TrimHeaterControl tempCtrl, PowerDevice powerDevc, int powerChan)
        {
            this.tempCtrl = tempCtrl;
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }

    /**
     *  Constants.
     */
    private static final String THERMAL_LIMITS = "ThermalLimits";

    private static final int
        UPDATE_STATE_INTVL = 1000,
        TEMP_REPEAT_COUNT = 3;

    private static final List<Set<Integer>> validTrimHeaterSets = new ArrayList<>();
    static {
        Set<Integer> trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
        trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYE);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_C);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYE);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
        trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYE);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYC);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYC);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYE);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
    }

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, TrimHeaterControl> tempControlMap = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private HeaterPsDevice heaterPs;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, Channel> allChannels = new HashMap<>();

    @ConfigurationParameter(category=THERMAL_LIMITS, isFinal=true)
    private volatile double coldTempLimit = 40.0;
    @ConfigurationParameter(category=THERMAL_LIMITS, isFinal=true)
    private volatile double coldTempLowLimit;
    @ConfigurationParameter(category=THERMAL_LIMITS, isFinal=true)
    private volatile double cryoTempLowLimit;

    private String[] minusColdTempChans;   // Names of the minus-end cold plate temperature channels
    private String[] centerColdTempChans;  // Names of the center cold plate temperature channels
    private String[] plusColdTempChans;    // Names of the plus-end cold plate temperature channels
    private String[] cryoTempChans;        // Names of the cryo plate temperature channels
    private String minusColdComp;          // Name of the minus-end cold compressor
    private String plusColdComp;           // Name of the plus-end cold compressor

    private static final Logger LOG = Logger.getLogger(ThermalMain.class.getName());
    private Set<Integer> trimHeaterSet;
    private final TrimHeater[] trimHeaters = new TrimHeater[ThermalState.NUM_TRIM_HEATERS];
    private final List<Channel> coldTemps = new ArrayList<>(), cryoTemps = new ArrayList<>();
    private final Set<Channel> minusColdTemps = new HashSet<>(), centerColdTemps = new HashSet<>(), plusColdTemps = new HashSet<>();
    private final ThermalState thermalState = new ThermalState();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private final List<StickyCondition> coldLowCondition = new ArrayList<>(), coldHighCondition = new ArrayList<>(),
                                        cryoLowCondition = new ArrayList<>();
    private ColdState refrigColdState = ColdState.UNKNOWN;
    private boolean coldHigh = false, minusColdLow = false, centerColdLow = false, plusColdLow = false, cryoLow = false;
    private String refrigGroup = null;
    private MonitorTaskControl monitorControl;
    private boolean inited = false;


    /**
     *  Constructor
     */
        public ThermalMain() {
        super("thermal", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, RefrigConstants.MONITOR_CONTROL);

        //Create and schedule an AgentPeriodicTask to update the refrigeration state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("thermal-state", () -> updateThermalState()).withPeriod(Duration.ofMillis(UPDATE_STATE_INTVL));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the thermal subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a thermal subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.THERMAL_TYPE, ThermalMain.class.getCanonicalName());

        // Add an agent present listener
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);

        // General initialization
        refrigGroup = RefrigUtils.getGroupName(subsys.getAgentInfo());

        // Initialize trim heater control
        Set<Integer> heaters = new HashSet<>();
        for (TrimHeaterControl ctrl : tempControlMap.values()) {
            int htr = ctrl.getPowerChannel();
            heaters.add(htr);
            trimHeaters[htr] = new TrimHeater(ctrl, ctrl.getPowerDevice(), htr);
        }
        for (Set heaterSet : validTrimHeaterSets) {
            if (heaterSet.equals(heaters)) {
                trimHeaterSet = heaterSet;
                break;
            }
        }
        if (trimHeaterSet == null) {
            throw new RuntimeException("Trim heater loop set is not a valid one");
        }
        thermalState.setTrimHeaters(trimHeaterSet.toArray(thermalState.getTrimHeaters()));

        // Initialize cold temperature channel list
        if (minusColdTempChans == null) {
            throw new RuntimeException("No minus-end cold-plate temperature monitoring channels specified");
        }
        for (String chanName : minusColdTempChans) {
            Channel chan = allChannels.get(chanName);
            if (chan == null) {
                throw new RuntimeException("Parameter minusColdTempChans contains non-channel item: " + chanName);
            }
            minusColdTemps.add(chan);
            coldTemps.add(chan);
        }
        if (centerColdTempChans == null) {
            throw new RuntimeException("No center cold-plate temperature monitoring channels specified");
        }
        for (String chanName : centerColdTempChans) {
            Channel chan = allChannels.get(chanName);
            if (chan == null) {
                throw new RuntimeException("Parameter centerColdTempChans contains non-channel item: " + chanName);
            }
            centerColdTemps.add(chan);
            coldTemps.add(chan);
        }
        if (plusColdTempChans == null) {
            throw new RuntimeException("No plus-end cold-plate temperature monitoring channels specified");
        }
        for (String chanName : plusColdTempChans) {
            Channel chan = allChannels.get(chanName);
            if (chan == null) {
                throw new RuntimeException("Parameter plusColdTempChans contains non-channel item: " + chanName);
            }
            plusColdTemps.add(chan);
            coldTemps.add(chan);
        }
        for (Channel channel : coldTemps) {
            coldLowCondition.add(new StickyCondition(TEMP_REPEAT_COUNT));
            coldHighCondition.add(new StickyCondition(TEMP_REPEAT_COUNT));
        }
        
        // Initialize cryo temperature channel list
        if (cryoTempChans == null) {
            throw new RuntimeException("No cryo-plate temperature monitoring channels specified");
        }
        for (String chanName : cryoTempChans) {
            Channel chan = allChannels.get(chanName);
            if (chan == null) {
                throw new RuntimeException("Parameter cryoTempChans contains non-channel item: " + chanName);
            }
            cryoTemps.add(chan);
        }
        for (Channel channel : cryoTemps) {
            cryoLowCondition.add(new StickyCondition(TEMP_REPEAT_COUNT));
        }
        
        //Register the Alerts raised by this subsystem
        alertService.registerAlert(ThermalAlert.COLD_TEMP_HIGH.newAlert());
        alertService.registerAlert(ThermalAlert.COLD_COMP_OFF.newAlert());
        alertService.registerAlert(ThermalAlert.COLD_MINUS_TEMP_LOW.newAlert());
        alertService.registerAlert(ThermalAlert.COLD_CENTER_TEMP_LOW.newAlert());
        alertService.registerAlert(ThermalAlert.COLD_PLUS_TEMP_LOW.newAlert());
        alertService.registerAlert(ThermalAlert.CRYO_TEMP_LOW.newAlert());
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        publishState();
        inited = true;
        LOG.info("Thermal control system started");
    }


    /**
     *  Listens for the arrival of the companion refrigeration subsystem.
     *
     *  @param  agents  Array of agents present
     */
    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
                String agentName = agent.getName();
                if (refrigGroup.equals(RefrigUtils.getGroupName(agent))) {
                    subsys.getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName));
                    break;
                }
            }
        }
    }


    /**
     *  Listens for the departure of the companion refrigeration subsystem.
     *
     *  @param  agents  Agents going absent
     */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE) && refrigGroup.equals(RefrigUtils.getGroupName(agent))) {
                subsys.getMessagingAccess().removeStatusMessageListener(this);
                //refrigColdState = ColdState.UNKNOWN;
            }
        }
    }


    /**
     *  Handles refrigeration status messages.
     *
     *  @param  msg  The status message
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        refrigColdState = msg.getState().getState(ColdState.class);
    }


    /**
     *  Gets the state of the thermal control system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The thermal control state
     */
    @Command(type=CommandType.QUERY, description="Get the thermal control state", level=0)
    public ThermalState getSystemState()
    {
        return thermalState;
    }    


    /**
     *  Sets a trim heater operation state.
     *
     *  @param  htr    The heater number
     *  @param  state  The heater operation state to set: OFF, POWER or TEMP.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater operation state")
    public void setTrimHeaterState(int htr, TrimHeaterOpState state) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                TrimHeaterOpState oldState = thermalState.getTrimHeaterOpState(htr);
                if (state != oldState & thermalState.getTrimHeaterState(htr) != TrimHeaterState.DISABLD) {
                    thermalState.setTrimHeaterOpState(htr, state);
                    heater.tempCtrl.setOpState(state);
                    heater.inited = true;
                }
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets a trim heater power value.
     *
     *  @param  htr    The heater number
     *  @param  power  The load power set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater power set point")
    public void setTrimHeaterPower(int htr, double power) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                thermalState.setTrimHeaterPower(htr, power);
                heater.tempCtrl.setPower(power);
                heater.inited = true;
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets a plate temperature value.
     *
     *  @param  htr   The heater number
     *  @param  temp  The plate temperature set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a plate temperature set point")
    public void setPlateTemperature(int htr, double temp) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                thermalState.setPlateTemperature(htr, temp);
                heater.tempCtrl.setTemp(temp);
                heater.inited = true;
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Turns on/off the bulk power supply.
     *
     *  @param  on  Whether to turn the power on
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off the bulk power supply")
    public void bulkPowerOn(boolean on)
    {
        heaterPs.setBulkPowerOn(on);
        publishState();
    }


    /**
     *  Publishes the state of the thermal module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        thermalState.setTickMillis(monitorControl.getPublishPeriod());
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(ThermalState.KEY, thermalState));
    }    


    /**
     *  Checks the temperature limits.
     */
    private void checkTempLimits()
    {
        boolean coldHigh = false, minusColdLow = false, centerColdLow = false, plusColdLow = false, cryoLow = false;
        for (int j = 0; j < coldTemps.size(); j++) {
            Channel channel = coldTemps.get(j);
            coldHigh |= coldHighCondition.get(j).update(channel.getValue() > coldTempLimit);
            boolean isLow = coldLowCondition.get(j).update(channel.getValue() < coldTempLowLimit);
            minusColdLow |= isLow && minusColdTemps.contains(channel);
            centerColdLow |= isLow && centerColdTemps.contains(channel);
            plusColdLow |= isLow && plusColdTemps.contains(channel);
        }
        for (int j = 0; j < cryoTemps.size(); j++) {
            cryoLow |= cryoLowCondition.get(j).update(cryoTemps.get(j).getValue() < cryoTempLowLimit);
        }
        if (coldHigh != this.coldHigh) {
            if (coldHigh) {
                raiseAlert(ThermalAlert.COLD_TEMP_HIGH, AlertState.ALARM, "At least one cold-plate RTD is above " + coldTempLimit + " C",
                           null, null);
            }
            else {
                lowerAlert(ThermalAlert.COLD_TEMP_HIGH, "No cold-plate RTD is above " + coldTempLimit + " C", null, null);
            }
            this.coldHigh = coldHigh;
        }
        if (minusColdLow != this.minusColdLow) {
            if (minusColdLow) {
                raiseAlert(ThermalAlert.COLD_MINUS_TEMP_LOW, AlertState.ALARM,
                           "At least one minus-end cold-plate RTD is below " + coldTempLowLimit + " C",
                           RefrigAction.Action.STOP_COLD_COMP, minusColdComp);
            }
            else {
                lowerAlert(ThermalAlert.COLD_MINUS_TEMP_LOW, "No minus-end cold-plate RTD is below " + coldTempLowLimit + " C",
                           RefrigAction.Action.STOP_COLD_COMP, minusColdComp);
            }
            this.minusColdLow = minusColdLow;
        }
        if (centerColdLow != this.centerColdLow) {
            if (centerColdLow) {
                raiseAlert(ThermalAlert.COLD_CENTER_TEMP_LOW, AlertState.ALARM,
                           "At least one center cold-plate RTD is below " + coldTempLowLimit + " C",
                           RefrigAction.Action.STOP_ALL_COLD_COMPS, "");
            }
            else {
                lowerAlert(ThermalAlert.COLD_CENTER_TEMP_LOW, "No center cold-plate RTD is below " + coldTempLowLimit + " C",
                           RefrigAction.Action.STOP_ALL_COLD_COMPS, "");
            }
            this.centerColdLow = centerColdLow;
        }
        if (plusColdLow != this.plusColdLow) {
            if (plusColdLow) {
                raiseAlert(ThermalAlert.COLD_PLUS_TEMP_LOW, AlertState.ALARM,
                           "At least one plus-end cold-plate RTD is below " + coldTempLowLimit + " C",
                           RefrigAction.Action.STOP_COLD_COMP, plusColdComp);
            }
            else {
                lowerAlert(ThermalAlert.COLD_PLUS_TEMP_LOW, "No plus-end cold-plate RTD is below " + coldTempLowLimit + " C",
                           RefrigAction.Action.STOP_COLD_COMP, plusColdComp);
            }
            this.plusColdLow = plusColdLow;
        }
        if (cryoLow != this.cryoLow) {
            if (cryoLow) {
                raiseAlert(ThermalAlert.CRYO_TEMP_LOW, AlertState.ALARM, "At least one cryo-plate RTD is below " + cryoTempLowLimit + " C",
                           RefrigAction.Action.STOP_ALL_CRYO_COMPS, "");
            }
            else {
                lowerAlert(ThermalAlert.CRYO_TEMP_LOW, "No cryo-plate RTD is below " + cryoTempLowLimit + " C",
                           RefrigAction.Action.STOP_ALL_CRYO_COMPS, "");
            }
            this.cryoLow = cryoLow;
        }
        boolean coldCompsOff = refrigColdState == ColdState.BOTH_OFF || refrigColdState == ColdState.UNKNOWN;
        if (coldCompsOff) {
            raiseAlert(ThermalAlert.COLD_COMP_OFF, AlertState.ALARM, "Both cold compressors are turned off", null, null);
        }
        else {
            lowerAlert(ThermalAlert.COLD_COMP_OFF, "At least one cold compressor is back on", null, null);
        }
    }


    /**
     *  Updates the thermal state periodically.
     */
    private void updateThermalState()
    {
        checkTempLimits();
        if (!inited) return;  // If postStart stage hasn't occurred yet
        synchronized(thermalState) {
            boolean changed = monitorControl.hasPeriodChanged();
            
            for (int htr : trimHeaterSet) {
                TrimHeater heater = trimHeaters[htr];
                TrimHeaterState oldState = thermalState.getTrimHeaterState(htr);
                TrimHeaterState state = coldHigh && htr != ThermalState.TRIM_HEATER_CRYO ? TrimHeaterState.DISABLD :
                                        !heater.powerDevc.isOnline() ? TrimHeaterState.OFFLINE :
                                        heater.tempCtrl.isOverTemp() ? TrimHeaterState.OVRTEMP :
                                        !heater.powerDevc.isEnabled(heater.powerChan) ? TrimHeaterState.OFF :
                                        heater.powerDevc.hasVoltError(heater.powerChan) ? TrimHeaterState.VOLTERR :
                                        heater.powerDevc.hasNoLoad(heater.powerChan) ? TrimHeaterState.NOLOAD :
                                        thermalState.getTrimHeaterOpState(htr) == TrimHeaterOpState.TEMP ? TrimHeaterState.TEMP
                                                                                                         : TrimHeaterState.POWER;
                if (state != oldState) {
                    thermalState.setTrimHeaterState(htr, state);
                    if (oldState == TrimHeaterState.OFFLINE) {
                        if (!heater.inited) {
                            if (heater.powerDevc.isEnabled(heater.powerChan)) {
                                double power = heater.powerDevc.getPower(heater.powerChan);
                                thermalState.setTrimHeaterPower(htr, power);
                                heater.tempCtrl.setPower(power);
                                thermalState.setTrimHeaterOpState(htr, TrimHeaterOpState.POWER);
                                heater.tempCtrl.setOpState(TrimHeaterOpState.POWER);
                            }
                            else {
                                thermalState.setTrimHeaterOpState(htr, TrimHeaterOpState.OFF);
                                heater.tempCtrl.setOpState(TrimHeaterOpState.OFF);
                            }
                            heater.inited = true;
                        }
                        else {
                            heater.tempCtrl.setPower(thermalState.getTrimHeaterPower(htr));
                            heater.tempCtrl.setTemp(thermalState.getPlateTemperature(htr));
                            heater.tempCtrl.setOpState(thermalState.getTrimHeaterOpState(htr));
                        }
                    }
                    if (state == TrimHeaterState.DISABLD) {
                        thermalState.setTrimHeaterOpState(htr, TrimHeaterOpState.OFF);
                        heater.tempCtrl.setOpState(TrimHeaterOpState.OFF);
                    }
                    changed = true;
                }
            }
            Boolean bulkOn = heaterPs.isBulkPowerOn();
            SwitchState bulkPowerState = bulkOn == null ? SwitchState.OFFLINE : bulkOn ? SwitchState.ON : SwitchState.OFF;
            if (bulkPowerState != thermalState.getBulkPowerState()) {
                thermalState.setBulkPowerState(bulkPowerState);
                changed = true;
            }
            if (changed) {
                publishState();
            }
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The thermal alert to raise
     *  @param  state  The alert state (WARNING or ALARM)
     *  @param  cond   The alert condition
     *  @param  action  The action to perform, or null if none
     *  @param  comp  The compressor name for the action, or "" if for all
     */
    private void raiseAlert(ThermalAlert alert, AlertState state, String cond, RefrigAction.Action action, String comp)
    {
        Boolean wasActive = activeAlertMap.put(alert.getId(), true);
        if (wasActive != Boolean.TRUE) {
            Alert al = alert.newAlert();
            if (action != null) {
                RefrigAction.addData(al, action, comp);
            }
            alertService.raiseAlert(al, state, cond);
        }
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The thermal alert to lower
     *  @param  cond   The alert condition
     *  @param  action  The action to perform, or null if none
     *  @param  comp  The compressor name for the action, or "" if for all
     */
    private void lowerAlert(ThermalAlert alert, String cond, RefrigAction.Action action, String comp)
    {
        Boolean wasActive = activeAlertMap.put(alert.getId(), false);
        if (wasActive == Boolean.TRUE) {
            Alert al = alert.newAlert();
            if (action != null) {
                RefrigAction.addData(al, action, comp);
            }
            alertService.raiseAlert(al, AlertState.NOMINAL, cond);
        }
    }


    /**
     *  Checks a trim heater number for validity and returns its object.
     *
     *  @param  htr  The heater number
     *  @return  The TrimHeater object
     *  @throws  RefrigException if the number is invalid or object is null
     */
    private TrimHeater getTrimHeater(int htr) throws RefrigException
    {
        if (!trimHeaterSet.contains(htr)) {
            throw new RefrigException("Invalid trim heater number: " + htr);
        }
        return trimHeaters[htr];
    }

}
