package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusSubsystemData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.common.data.MonitorTask;
import org.lsst.ccs.subsystem.refrig.constants.HexAlert;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.data.HexState;
import org.lsst.ccs.subsystem.refrig.data.RefrigAction;
import org.lsst.ccs.subsystem.refrig.data.RefrigUtils;
import org.lsst.ccs.subsystem.refrig.data.UpdatePeriod;

/**
 *  Implements the refrigeration heat exchanger monitoring subsystem.
 *
 *  @author Owen Saxton
 */
public class HexMain extends Subsystem implements HasLifecycle, AgentPresenceListener, StatusMessageListener {

    /**
     *  Data fields.
     */
    private static final int TEMP_REPEAT_COUNT = 3;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();

    @ConfigurationParameter(isFinal = true)
    private volatile double rtnToEvapOffset = 0.3;
    @ConfigurationParameter(isFinal = true)
    private volatile double lowColdTempLimit;
    @ConfigurationParameter(isFinal = true)
    private volatile double lowCryoTempLimit;

    private List<String> coldTempChans;
    private List<String> cryoTempChans;

    private static final Logger LOG = Logger.getLogger(HexMain.class.getName());
    private final HexState hexState = new HexState();
    private String refrigGroup;
    private MonitorTaskControl monitorControl;
    private Map<String, MonitorTask> monitorTaskMap;
    private final List<Channel> coldTempChannels = new ArrayList<>(), cryoTempChannels = new ArrayList<>();
    private final List<StickyCondition> coldConditions = new ArrayList<>(), cryoConditions = new ArrayList<>();
    private final Map<String, Boolean> activeAlarms = new HashMap<>();


    /**
     *  Constructor.
     */
    public HexMain() {
        super("hex", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build() {

        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, RefrigConstants.MONITOR_CONTROL);

        // Create and schedule an AgentPeriodicTask to publish the hex state when updated
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("hex-state", () -> publishUpdatedHexState()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);

        // Create and schedule an AgentPeriodicTask to check temperature limits
        pt = new AgentPeriodicTask("temp-check", () -> checkTemperatures()).withPeriod(Duration.ofMillis(1000));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     * Init phase.
     * 
     * We register here all the Alerts raised by this Subsystem.
     */
    @Override
    public void init() {
        for (String chanName : coldTempChans) {
            alertService.registerAlert(HexAlert.EVAP_TEMP_LOW.newAlert(chanName.split("/")[0]));
        }
        for (String chanName : cryoTempChans) {
            alertService.registerAlert(HexAlert.EVAP_TEMP_LOW.newAlert(chanName.split("/")[0]));
        }
    }


    /**
     *  Initializes the subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a heat exchanger subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.HEX_TYPE, HexMain.class.getCanonicalName());

        // Add an agent present listener
        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        refrigGroup = RefrigUtils.getGroupName(subsys.getAgentInfo());

        // Initialize monitor task data
        monitorTaskMap = monitorControl.getMonitorTaskMap();

        // Check defined temperature channels
        for (String chanName : coldTempChans) {
            Channel chan = allChannels.get(chanName);
            if (chan == null) {
                throw new RuntimeException("Channel " + chanName + " doesn't exist");
            }
            coldTempChannels.add(chan);
            coldConditions.add(new StickyCondition(TEMP_REPEAT_COUNT));
            activeAlarms.put(chanName, false);
        }
        for (String chanName : cryoTempChans) {
            Channel chan = allChannels.get(chanName);
            if (chan == null) {
                throw new RuntimeException("Channel " + chanName + " doesn't exist");
            }
            cryoTempChannels.add(chan);
            cryoConditions.add(new StickyCondition(TEMP_REPEAT_COUNT));
            activeAlarms.put(chanName, false);
        }
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        // Announce startup
        LOG.info("Heat exchanger subsystem started");
    }


    /**
     *  Listens for the arrival of the companion refrigeration subsystem.
     *
     *  @param  agents  Array of agents present
     */
    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
                String agentName = agent.getName();
                if (RefrigUtils.getGroupName(agent).equals(refrigGroup)) {
                    getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName)
                                                                                   && msg instanceof StatusSubsystemData);
                    break;
                }
            }
        }
    }


    /**
     *  Listens for the departure of the companion refrigeration subsystem.
     *
     *  @param  agents  Agents going absent
     */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE) && RefrigUtils.getGroupName(agent).equals(refrigGroup)) {
                getMessagingAccess().removeStatusMessageListener(this);
            }
        }
    }


    /**
     *  Handles refrigeration status messages.
     *
     *  Sets the tick period from the refrig system value.
     *
     *  @param  msg  The status message
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        StatusSubsystemData sd = (StatusSubsystemData)msg;
        if (sd.getDataKey().equals(UpdatePeriod.KEY)) {
            int tickMillis = ((UpdatePeriod)((KeyValueData)sd.getSubsystemData()).getValue()).getTickMillis();
            if (tickMillis != monitorControl.getPublishPeriod()) {
                monitorControl.setPublishPeriod(tickMillis);
            }
        }
    }


    /**
     *  Gets the state of the heat exchanger module.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The refrigeration state
     */
    @Command(type=CommandType.QUERY, description="Get the heat exchanger state", level=0)
    public HexState getSystemState()
    {
        synchronized (hexState) {
            hexState.setTickMillis(monitorControl.getPublishPeriod());
            hexState.clearMonitorTasks();
            for (MonitorTask task : monitorTaskMap.values()) {
                hexState.addMonitorTask(task);
            }
            return hexState;
        }
    }    


    /**
     *  Publishes the updated heat exchanger state.
     *
     *  This is called regularly as a timer task.
     */
    private void publishUpdatedHexState()
    {
        synchronized (hexState) {
            boolean changed = monitorControl.hasPeriodChanged();
            hexState.clearMonitorTasks();
            for (MonitorTask task : monitorTaskMap.values()) {
                if (task.hasChanged()) {
                    hexState.addMonitorTask(task);
                    changed = true;
                }
            }
            if (changed) {
                hexState.setTickMillis(monitorControl.getPublishPeriod());
                publishSubsystemDataOnStatusBus(new KeyValueData(HexState.KEY, hexState));
            }
        }
    }


    /**
     *  Checks evaporator temperature limits.
     * 
     *  This is called regularly as a timer task.
     */
    private void checkTemperatures()
    {
        for (int j = 0; j < coldTempChannels.size(); j++) {
            Channel chan = coldTempChannels.get(j);
            double value = chan.getValue();
            if (coldConditions.get(j).update(value < lowColdTempLimit)) {
                raiseTempAlarm(chan.getName(), true, value);
            }
            else {
                lowerTempAlarm(chan.getName(), true, value);
            }
        }
        for (int j = 0; j < cryoTempChannels.size(); j++) {
            Channel chan = cryoTempChannels.get(j);
            double value = chan.getValue();
            if (cryoConditions.get(j).update(value < lowCryoTempLimit)) {
                raiseTempAlarm(chan.getName(), false, value);
            }
            else {
                lowerTempAlarm(chan.getName(), false, value);
            }
        }
    }


    /**
     *  Raises the too-low evaporator exit temperature alarm.
     * 
     *  @param  chanName  The temperature channel name
     *  @param  isCold  Whether cold system, otherwise cryo
     *  @param  value  The temperature value
     */
    private void raiseTempAlarm(String chanName, boolean isCold, double value)
    {
        if (!activeAlarms.get(chanName)) {
            activeAlarms.put(chanName, true);
            String compName = chanName.split("/")[0];
            Alert alert = HexAlert.EVAP_TEMP_LOW.newAlert(compName);
            RefrigAction.addData(alert, isCold ? RefrigAction.Action.STOP_COLD_COMP : RefrigAction.Action.STOP_CRYO_COMP, compName);
            double limit = isCold ? lowColdTempLimit : lowCryoTempLimit;
            String sValue = String.format("%.1f", value);
            alertService.raiseAlert(alert, AlertState.ALARM, "Temperature is too low: value (" + sValue + ") < " + limit);
        }
    }


    /**
     *  Lowers the too-low evaporator exit temperature alarm.
     * 
     *  @param  chanName  The temperature channel name
     *  @param  isCold  Whether cold system, otherwise cryo
     *  @param  value  The temperature value
     */
    private void lowerTempAlarm(String chanName, boolean isCold, double value)
    {
        if (activeAlarms.get(chanName)) {
            activeAlarms.put(chanName, false);
            String compName = chanName.split("/")[0];
            Alert alert = HexAlert.EVAP_TEMP_LOW.newAlert(compName);
            RefrigAction.addData(alert, isCold ? RefrigAction.Action.STOP_COLD_COMP : RefrigAction.Action.STOP_CRYO_COMP, compName);
            String sValue = String.format("%.1f", value);
            alertService.raiseAlert(alert, AlertState.NOMINAL, "Temperature is normal: value = " + sValue);
        }
        
    }


    /**
     *  Gets the offset between the return and evaporator temperatures.
     * 
     *  @return  The offset
     */
    double getRtnToEvapOffset() {
        return rtnToEvapOffset;
    }
    
}
