package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.AgentPropertyPredicate;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.AgentStatusAggregatorService;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 * A virtual device used to calculate cold system evaporator pressure drop
 * 
 * @author The LSST CCS Team
 */
public class EvaporatorPressDevice extends Device {

    public static final int
        CHAN_COLD1 = 0,
        CHAN_COLD2 = 1,
        NUM_CHANS = 2;
    private static final int x = 0;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStatusAggregatorService asas;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private final Map<String, Channel> allChannels = new HashMap<>();

    private List<String> returnPressChans;
    private List<String> coldRtdChans;
    private String thermalSubsysName;

    private static final Logger LOG = Logger.getLogger(EvaporatorPressDevice.class.getName());
    private final List<Channel> returnPrsChannels = new ArrayList<>();

    /**
     *  Check device parameters
     */
    @Override
    protected void initDevice()
    {
        if (returnPressChans == null || returnPressChans.size() != NUM_CHANS) {
            ErrorUtils.reportConfigError(LOG, name, "returnPressChans", "must contain " + NUM_CHANS + " elements");
        }
        for (String chanName : returnPressChans) {
            Channel chan = allChannels.get(chanName);
            if (chan == null) {
                ErrorUtils.reportConfigError(LOG, name, "returnPressChans", "contains non-existent channel: " + chanName);
            }
            returnPrsChannels.add(chan);
        }
        if (thermalSubsysName == null) {
            ErrorUtils.reportConfigError(LOG, name, "thermalSubsysName", "has not been specified");
        }
        if (coldRtdChans == null || coldRtdChans.size() != NUM_CHANS) {
            ErrorUtils.reportConfigError(LOG, name, "coldRtdChans", "must contain " + NUM_CHANS + " elements");
        }
        AgentPropertyPredicate app = new AgentPropertyPredicate("[agentName:" + thermalSubsysName + "]");
        asas.setAggregatePattern(app, coldRtdChans.get(0) + "|" + coldRtdChans.get(1));
        coldRtdChans.set(0, thermalSubsysName + "/" + coldRtdChans.get(0));
        coldRtdChans.set(1, thermalSubsysName + "/" + coldRtdChans.get(1));
    }


    /**
     *  Initializes the device.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
    }


    /**
     *  Closes the device.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
        }
        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        Double temp = (Double)asas.getLast(coldRtdChans.get(hwChan));
        temp = temp == null ? Double.NaN : temp;
        return PressureTemperatureUtils.evaluatePressure(temp) - returnPrsChannels.get(hwChan).getValue();
    }
    
}
