package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.CompAlert;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompLatches;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.CompressorState;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.constants.SwCondState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.data.CompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Controls a refrigeration compressor.
 *
 *  @author Owen Saxton
 */
public class Compressor implements HasLifecycle {

    public static interface SwitchDevice {

        public void setSwitchOn(int chan, boolean on) throws RefrigException;

        public Boolean isSwitchOn(int chan);

    }

    class AlertEnableSwitch implements SwitchDevice {

        @Override
        public void setSwitchOn(int chan, boolean on)
        {
            alertsEnabled = powerEnabled ? true: on;
        }

        @Override
        public Boolean isSwitchOn(int chan)
        {
            return alertsEnabled;
        }

    }

    public static class SwCondData {

        protected Channel channel = null;    // Monitor channel with the value: null if not used
        protected Channel channel2 = null;   // Monitor channel with the second value, if applicable
        private Boolean immedCond = null;    // Immediate condition; null if not used
        private boolean newImmedCond = false;  // New value for immediate condition
        private double value = 0.0;        // The value that exceeded the limit
        private long endTime = 0;          // The computed end time
        private SwCondState swState = SwCondState.CLEAR; // The software condition state
        private SwCondState provSwState = SwCondState.CLEAR; // The provisional software condition state
        private int provCount = 0;         // Number of successive times provSwState has same value

    }

    /**
     *  Constants.
     */
    protected static final String COMP_LIMITS = "CompLimits";
    private static final int MIN_SW_STATE_SITS = 5;

    private static final Map<Integer, CompAlert> ccsAlertMap = new HashMap<>();
    static {
        ccsAlertMap.put(CompSwConds.SWC_DISC_PRESS, CompAlert.DISC_PRESS_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_DISC_TEMP, CompAlert.DISC_TEMP_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_CMPR_POWER, CompAlert.COMP_POWER_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_LIQUID_TEMP, CompAlert.LIQD_TEMP_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_OIL_LEVEL, CompAlert.OIL_LEVEL_LOW);
        ccsAlertMap.put(CompSwConds.SWC_SUCT_TEMP, CompAlert.SUCT_TEMP_LOW);
        ccsAlertMap.put(CompSwConds.SWC_PHASE_SEP_TEMP, CompAlert.PHASE_TEMP_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_PLATE_TEMP, CompAlert.CRYO_TEMP_LOW);
        ccsAlertMap.put(CompSwConds.SWC_PRESS_DIFF, CompAlert.PRESS_DIFF_HIGH);
        ccsAlertMap.put(CompSwConds.SWC_DISC_TEMP_LOW, CompAlert.DISC_TEMP_LOW);
    }
    private static final Map<Integer, CompAlert> plcAlertMap = new HashMap<>();
    static {
        plcAlertMap.put(CompLatches.LATCH_DISCHARGE_PRESS, CompAlert.DISC_PRESS_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_DISCHARGE_TEMP, CompAlert.DISC_TEMP_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_POWER, CompAlert.COMP_POWER_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_LIQUID_TEMP, CompAlert.LIQD_TEMP_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_OIL_LEVEL, CompAlert.OIL_LEVEL_LOW_PLC);
        plcAlertMap.put(CompLatches.LATCH_SUCTION_TEMP, CompAlert.SUCT_TEMP_LOW_PLC);
        plcAlertMap.put(CompLatches.LATCH_AFTER_COOLER, CompAlert.AFTER_TEMP_HIGH_PLC);
        plcAlertMap.put(CompLatches.LATCH_SENSORS_VALID, CompAlert.SENS_READ_BAD_PLC);
        plcAlertMap.put(CompLatches.LATCH_SMOKE_DETC, CompAlert.SMOKE_DETC_PLC);
        plcAlertMap.put(CompLatches.LATCH_EXT_PERMIT, CompAlert.EXT_PERMIT_PLC);
    }

    /**
     *  Data fields.
     */
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    protected CompPlutoDevice plutoDevc;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    protected CompMaq20Device maq20Devc;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    protected final Map<String, Channel> channelMap = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private FanControl fanControl;

    protected String discPressChan, discTempChan, suctTempChan, cmprPowerChan;  // From Groovy file

    private static final Logger LOG = Logger.getLogger(Compressor.class.getName());
    protected final CompState state;
    protected final SwitchDevice[] switchDevices;
    protected final int[] switchChannels;
    protected Channel discPress, discTemp;
    protected boolean gotCommand = false;  // Forces status update after command processed
    protected boolean stateChanged = false;
    private final LatchState[] savedLatchStates = new LatchState[CompLatches.NUM_LATCHES];
    protected CompLimits.LimitData[] limitData;
    protected final SwCondData[] condData = new SwCondData[CompSwConds.NUM_SW_CONDITIONS];
    private final SwCondState[] savedSwStates = new SwCondState[CompSwConds.NUM_SW_CONDITIONS];
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private final AlertEnableSwitch alertEnableSwitch = new AlertEnableSwitch();
    private int activeConds = 0;
    private boolean powerEnabled = false, alertsEnabled = false;


    /**
     *  Constructor.
     *
     *  @param  state  The compressor state object
     */
    public Compressor(CompState state)
    {
        this.state = state;
        switchDevices = new SwitchDevice[CompSwitches.NUM_SWITCHES];
        switchChannels = new int[CompSwitches.NUM_SWITCHES];
        for (int j = 0; j < CompSwConds.NUM_SW_CONDITIONS; j++) {
            condData[j] = new SwCondData();
        }
    }


    @Override
    public void init() {
        //Register the Alerts raised by this class
        alertService.registerAlert(CompAlert.PLC_DEAD.newAlert(name));
        for (int cond : state.getValidLatches()) {
            CompAlert alert = plcAlertMap.get(cond);
            alertService.registerAlert(alert.newAlert(name));            
        }
        
        for (int cond : state.getValidSwConditions()) {
            CompAlert alert = ccsAlertMap.get(cond);
            alertService.registerAlert(alert.newAlert(name));                        
        }
    }
    
    /**
     *  Initializes the compressor control.
     */
    @Override
    public void postInit()
    {
        if (plutoDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "plutoDevc", "not specified");
        }
        if (maq20Devc == null) {
            ErrorUtils.reportConfigError(LOG, name, "maq20Devc", "not specified");
        }

        discTemp = (discTempChan != null) ? channelMap.get(discTempChan) : null;
        if (discTemp == null) {
            ErrorUtils.reportConfigError(LOG, name, "discTempChan", "not specified or not defined");
        }
        condData[CompSwConds.SWC_DISC_TEMP].channel = discTemp;
        Channel suctTemp = (suctTempChan != null) ? channelMap.get(suctTempChan) : null;
        if (suctTemp == null) {
            ErrorUtils.reportConfigError(LOG, name, "suctTempChan", "not specified or not defined");
        }
        condData[CompSwConds.SWC_SUCT_TEMP].channel = suctTemp;
        discPress = (discPressChan != null) ? channelMap.get(discPressChan) : null;
        if (discPress == null) {
            ErrorUtils.reportConfigError(LOG, name, "discPressChan", "not specified or not defined");
        }
        condData[CompSwConds.SWC_DISC_PRESS].channel = discPress;
        Channel cmprPower = (cmprPowerChan != null) ? channelMap.get(cmprPowerChan) : null;
        if (cmprPower == null) {
            ErrorUtils.reportConfigError(LOG, name, "cmprPowerChan", "not specified or not defined");
        }
        condData[CompSwConds.SWC_CMPR_POWER].channel = cmprPower;
        condData[CompSwConds.SWC_PLATE_TEMP].immedCond = false;

        switchDevices[CompSwitches.SW_ENABLE] = plutoDevc;
        switchChannels[CompSwitches.SW_ENABLE] = CompPlutoDevice.SW_ENABLE;
        switchDevices[CompSwitches.SW_LIGHTS] = plutoDevc;
        switchChannels[CompSwitches.SW_LIGHTS] = CompPlutoDevice.SW_LIGHTS;
        switchDevices[CompSwitches.SW_ENABLE_ALERTS] = alertEnableSwitch;

        state.setName(name);
        plutoDevc.setType(state.getType());
        maq20Devc.setType(state.getType());
    }


    /**
     *  Starts the compressor control.
     */
    @Override
    public void postStart()
    {
        if (fanControl != null) {
            LOG.log(Level.INFO, "Starting fan speed controller for compressor {0}", name);
            fanControl.startLoop();
        }
    }


    /**
     *  Sets the compressor index.
     *
     *  @param  index  The index value
     */
    public void setIndex(int index)
    {
        state.setIndex(index);
    }


    /**
     *  Gets the compressor index.
     *
     *  @return  The index value
     */
    public int getIndex()
    {
        return state.getIndex();
    }


    /**
     *  Command to get the valid switch names.
     *
     *  @return  The index value
     */
    @Command(type=CommandType.QUERY, description="Get the valid switch names", level=0)
    public List<String> getSwitchNames()
    {
        List<String> names = new ArrayList<>();
        for (int swId : state.getValidSwitches()) {
            names.add(CompSwitches.ID_MAP.get(swId));
        }
        return names;
    }


    /**
     *  Command to turn a (named) switch on or off..
     *
     *  @param  swName  The switch name
     *  @param  on      Whether to turn on
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a compressor's switch state")
    public synchronized void setSwitchOn(@Argument(description="Switch name") String swName,
                                         @Argument(description="Whether to turn on") boolean on) throws RefrigException
    {
        Integer swId = CompSwitches.NAME_MAP.get(swName);
        if (swId == null || !state.getValidSwitches().contains(swId)) {
            throw new RefrigException("Invalid switch name: " + swName);
        }
        gotCommand = true;
        if (swId != CompSwitches.SW_ENABLE || !on || activeConds == 0) {
            setSwitch(swId, on);
            if (swId == CompSwitches.SW_ENABLE) {
                powerEnabled = on;
                alertEnableSwitch.setSwitchOn(0, on);
            }
        }
    }


    /**
     *  Command to reset the latches.
     */
    @Command(type=CommandType.ACTION, description="Reset a compressor's latches")
    public synchronized void resetLatches()
    {
        gotCommand = true;
        plutoDevc.resetLatches();
    }


    /**
     *  Updates the compressor system.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @return  Whether any state value changed
     */
    public synchronized boolean updateSystem()
    {
        checkLimits();
        return updateState();
    }


    /**
     *  Gets the compressor state.
     *
     *  @return  The compressor state
     */
    public CompState getState()
    {
        return state;
    }    
            

    /**
     *  Sets whether the plate temperature is too low.
     * 
     *  @param  on  Whether low condition is set
     */
    public void setPlateTempLow(boolean on)
    {
        condData[CompSwConds.SWC_PLATE_TEMP].newImmedCond = on;
    }


    /**
     *  Checks compressor shut-off limits.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     */
    private synchronized void checkLimits()
    {
        for (int cond : state.getValidSwConditions()) {
            SwCondData cd = condData[cond];
            if (cd.immedCond != null && cd.newImmedCond != cd.immedCond) {
                cd.immedCond = cd.newImmedCond;
                if (cd.immedCond) {
                    cd.swState = SwCondState.ACTIVE;
                    setCondition(cond);
                }
                else {
                    cd.swState = SwCondState.CLEAR;
                    clearCondition(cond);
                }
                continue;
            }
            if (cd.channel == null) continue;
            CompLimits.LimitData ld = limitData[cond];
            double value = cd.channel.getValue();
            if (cd.channel2 != null) {
                value = Math.abs(value - cd.channel2.getValue());
            }
            if (Double.isNaN(value)) continue;
            long endTime = cd.endTime;
            SwCondState swState = cd.swState;
            if (!ld.isLower && value > ld.immedLimit || ld.isLower && value < ld.immedLimit) {
                endTime = -1;
                swState = SwCondState.ACTIVE;
            }
            else if (!ld.isLower && value > ld.delayLimit || ld.isLower && value < ld.delayLimit) {
                if (endTime == 0) {
                    endTime = System.currentTimeMillis() + ld.delayTime;
                    swState = SwCondState.DLYPEND;
                }
                if (endTime >= 0) {
                    if (ld.delayTime < 0 && state.getConditionState(CompConditions.COND_CMP_ON_8HRS) == ConditionState.YES
                          || ld.delayTime >= 0 && System.currentTimeMillis() > endTime) {
                        endTime = -1;
                        swState = SwCondState.DLYACTV;
                    }
                }
            }
            else {
                endTime = 0;
                swState = SwCondState.CLEAR;
            }
            if (swState != cd.provSwState) {
                cd.provCount = 0;
                cd.provSwState = swState;
            }
            if (++cd.provCount >= MIN_SW_STATE_SITS && cd.provSwState != cd.swState) {
                cd.value = value;
                cd.swState = cd.provSwState;
                cd.endTime = endTime;
                if (endTime < 0) {
                    setCondition(cond);
                }
                else {
                    clearCondition(cond);
                }
            }
        }
    }


    /**
     *  Updates the compressor state.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @return  Whether any value changed
     */
    protected synchronized boolean updateState()
    {
        boolean changed = gotCommand;
        gotCommand = false;
        Boolean active = plutoDevc.isPLCActive();
        CompressorState compState = active == null ? CompressorState.OFFLINE : !active ? CompressorState.PLC_DEAD : null;
        for (int sw : state.getValidSwitches()) {
            Boolean on = switchDevices[sw].isSwitchOn(switchChannels[sw]);
            SwitchState swState = on == null ? SwitchState.OFFLINE : on ? SwitchState.ON : SwitchState.OFF;
            if (swState != state.getSwitchState(sw)) {
                state.setSwitchState(sw, swState);
                changed = true;
            }
        }
        boolean cmprPowered = false;
        for (int cond : state.getValidConditions()) {
            ConditionState condState = plutoDevc.getConditionState(cond);
            if (cond == CompConditions.COND_CMP_POWERED && condState != ConditionState.NO) {
                cmprPowered = true;
            }
            if (condState != state.getConditionState(cond)) {
                state.setConditionState(cond, condState);
                changed = true;
            }
        }
        alertEnableSwitch.setSwitchOn(0, alertsEnabled);
        for (int cond : state.getValidLatches()) {
            LatchState latchState = plutoDevc.getLatchState(cond);
            LatchState oldLatchState = state.getLatchState(cond);
            if (latchState != oldLatchState) {
                state.setLatchState(cond, latchState);
                changed = true;
            }
            LatchState savedLatchState = savedLatchStates[cond];
            if (alertsEnabled) {
                if (savedLatchState != null && latchState != savedLatchState
                      || savedLatchState == null && latchState != oldLatchState) {
                    raisePlcAlert(cond, latchState, savedLatchState == null ? oldLatchState : savedLatchState);
                    savedLatchStates[cond] = null;
                }
            }
            else {
                if (savedLatchState == null && latchState != oldLatchState) {
                    savedLatchStates[cond] = oldLatchState;
                }
            }
        }
        for (int cond : state.getValidSwConditions()) {
            SwCondData cd = condData[cond];
            CompLimits.LimitData ld = limitData[cond];
            SwCondState swState = ld.noShutoff && powerEnabled ? SwCondState.CLEAR : cd.swState;
            if (compState == null && (swState == SwCondState.ACTIVE || swState == SwCondState.DLYACTV)) {
                compState = CompressorState.SW_DSAB;
            }
            SwCondState oldSwState = state.getSwConditionState(cond);
            if (swState != oldSwState) {
                state.setSwConditionState(cond, swState);
                changed = true;
            }
            SwCondState savedSwState = savedSwStates[cond];
            if (alertsEnabled) {
                if (savedSwState != null && swState != savedSwState
                      || savedSwState == null && swState != oldSwState) {
                    raiseCcsAlert(cond, cd.value, ld, swState);
                    savedSwStates[cond] = null;
                }
            }
            else {
                if (savedSwState == null && swState != oldSwState) {
                    savedSwStates[cond] = oldSwState;
                }
            }
        }
        if (compState == null) {
            compState = state.getConditionState(CompConditions.COND_CMP_WAITING) == ConditionState.YES ? CompressorState.WAITING :
                        state.getConditionState(CompConditions.COND_CMP_POWERED) == ConditionState.YES ? CompressorState.RUNNING :
                        state.getConditionState(CompConditions.COND_LATCHES_CLEAR) == ConditionState.NO ? CompressorState.HW_DSAB :
                        CompressorState.STOPPED;
        }
        CompressorState oldCompState = state.getCompressorState();
        stateChanged = compState != oldCompState;
        if (stateChanged) {
            if (compState == CompressorState.PLC_DEAD) {
                raiseAlert(CompAlert.PLC_DEAD, AlertState.ALARM, name + " PLC has died: error code = " + plutoDevc.getErrorCode());
            }
            else if (oldCompState == CompressorState.PLC_DEAD) {
                lowerAlert(CompAlert.PLC_DEAD, name + " PLC is alive");
            }
            state.setCompressorState(compState);
            changed = true;
        }
        if (powerEnabled && !cmprPowered) {
            disable();
        }
        
        return changed;
    }


    /**
     *  Raise PLC latch state alert.
     *
     *  @param  cond           The PLC latch condition
     *  @param  latchState     The new PLC latch state
     *  @param  oldLatchState  The current PLC latch state
     */
    private void raisePlcAlert(int cond, LatchState latchState, LatchState oldLatchState)
    {
        CompAlert alert = plcAlertMap.get(cond);
        if (latchState == LatchState.ACTIVE) {
            raiseAlert(alert, AlertState.ALARM, name + " PLC error condition set");
        }
        else if (latchState == LatchState.WARNING) {
            raiseAlert(alert, AlertState.WARNING, name + " PLC warning condition set");
        }
        else {
            if (latchState == LatchState.LATCHED && oldLatchState == LatchState.CLEAR) {
                raiseAlert(alert, AlertState.ALARM, name + " PLC error condition set");
            }
            lowerAlert(alert, name + " PLC error condition cleared");
        }
    }


    /**
     *  Raise CCS error state alert.
     *
     *  @param  cond     The CCS error condition
     *  @param  value    The value that was checked
     *  @oaram  ld       The associated limit data
     *  @param  swState  The new CCS software error state
     */
    private void raiseCcsAlert(int cond, double value, CompLimits.LimitData ld, SwCondState swState)
    {
        CompAlert alert = ccsAlertMap.get(cond);
        boolean isImmed = condData[cond].immedCond != null;
        String sValue = String.format("%.1f", value);
        switch (swState) {
        case ACTIVE:
            raiseAlert(alert, AlertState.ALARM, name + " CCS immediate error condition set"
                                                  + (isImmed ? "" : ": value (" + sValue + (ld.isLower ? ") < " : ") > ") + ld.immedLimit));
            break;
        case DLYACTV:
            String timeText = ld.delayTime >= 0 ? " for " + ld.delayTime / 1000 + " sec" : " and compressor on for 8 hours";
            raiseAlert(alert, AlertState.ALARM, name + " CCS delayed error condition set: value (" + sValue
                                                  + (ld.isLower ? ") < " : ") > ") + ld.delayLimit + timeText);
            break;
        case DLYPEND:
            raiseAlert(alert, AlertState.WARNING, name + " CCS delayed error condition pending: value (" + sValue
                                                    + (ld.isLower ? ") < " : ") > ") + ld.delayLimit);
            break;
        default:
            lowerAlert(alert, name + " CCS error condition cleared" + (isImmed ? "" : ": value = " + sValue));
        }
    }


    /**
     *  Turns a switch on or off..
     *
     *  @param  sw  The switch ID
     *  @param  on  Whether to turn on
     *  @throws  RefrigException
     */
    protected void setSwitch(int sw, boolean on) throws RefrigException
    {
        switchDevices[sw].setSwitchOn(switchChannels[sw], on);
    }


    /**
     *  Sets a condition that disables the compressor
     *
     *  @param  cond  The condition number
     */
    private void setCondition(int cond)
    {
        LOG.log(Level.FINE, "{0} compressor encountered \"{1}\" condition", new Object[]{name, CompSwConds.DESCS[cond]});
        if (!limitData[cond].noShutoff) {
            disable();
        }
        activeConds |= 1 << cond;
    }


    /**
     *  Clears a condition that disabled the compressor
     *
     *  @param  cond  The condition number
     */
    private void clearCondition(int cond)
    {
        activeConds &= ~(1 << cond);
    }


    /**
     *  Disables the compressor
     */
    private void disable()
    {
        try {
            setSwitch(CompSwitches.SW_ENABLE, false);
            powerEnabled = false;
        }
        catch (RefrigException e) {
            LOG.log(Level.SEVERE, "Error shutting off {0} compressor: {1}", new Object[]{name, e});
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The refrigeration alert to raise
     *  @param  state  The alert state (WARNING or ALARM)
     *  @param  cond   The alert condition
     */
    private void raiseAlert(CompAlert alert, AlertState state, String cond)
    {
        activeAlertMap.put(alert.getId(name), true);
        alertService.raiseAlert(alert.newAlert(name), state, cond);
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The refrigeration alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(CompAlert alert, String cond)
    {
        if (activeAlertMap.put(alert.getId(name), false) == Boolean.TRUE) {
            alertService.raiseAlert(alert.newAlert(name), AlertState.NOMINAL, cond);
        }
    }

    /**
     *  Compares two numbers for equality, accounting for possible NaNs.
     *
     *  @param  value1  First value
     *  @param  value2  Second value
     *  @return  Whether the values are equal
     */
    protected boolean areEqual(double value1, double value2)
    {
        return value1 == value2 || Double.isNaN(value1) && Double.isNaN(value2);
    }

}
