package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.auxelex.HeaterPS;
import org.lsst.ccs.drivers.auxelex.RebBulkPS;
import org.lsst.ccs.drivers.commons.DriverConstants;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;

/**
 *  Handles a SLAC heater power supply.
 *
 *  @author Owen Saxton
 */
public class HeaterPsDevice extends PowerDevice {

    /**
     *  Class for running the power-setting threads
     */
    class PowerThread extends Thread {

        private final int chan;
        private final BlockingQueue dataQueue;
        private final List<BlockingQueue> returnQueues = new ArrayList<>();

        PowerThread(int chan, BlockingQueue dataQueue) {
            this.chan = chan;
            this.dataQueue = dataQueue;
        }

        public void addReturnQueue(BlockingQueue returnQueue) {
            returnQueues.add(returnQueue);
        }

        @Override
        public void run() {
            while (true) {
                try {
                    setChannelPower(chan, (Double)dataQueue.take());
                }
                catch (InterruptedException e) {}
                for (BlockingQueue queue : returnQueues) {
                    queue.offer(chan);
                }
            }
        }

    }

    /**
     *  Constants.
     */
    public static final int
        TYPE_VOLTAGE     = 0,
        TYPE_CURRENT     = 1,
        TYPE_POWER       = 2,
        TYPE_TOTAL_POWER = 3,
        TYPE_TEMP        = 4,
        TYPE_MAIN_VOLTS  = 5,
        TYPE_MAIN_CURR   = 6,
        TYPE_MAIN_POWER  = 7,
        TYPE_MAIN_TEMP   = 8,
        TYPE_MAIN_STATUS = 9;

    private static final int
        N_HW_CHANS = HeaterPS.NUM_HEATERS,
        N_SET_CHANS = HeaterPS.NUM_SECT_HEATERS,
        CRYO_CHANNEL_FIRST = HeaterPS.CHAN_CRYO_FIRST,
        COLD_CHANNEL_FIRST = HeaterPS.CHAN_COLD_FIRST;
    private static final String
        NODE   = "node";

    private static final Set<Integer>[] channelSets = new Set[ThermalState.NUM_TRIM_HEATERS];
    static {
        for (int chanSet = 0; chanSet < ThermalState.NUM_TRIM_HEATERS; chanSet++) {
            channelSets[chanSet] = new HashSet<>();
        }
        for (int chan = CRYO_CHANNEL_FIRST; chan < CRYO_CHANNEL_FIRST + N_SET_CHANS; chan++) {
            channelSets[ThermalState.TRIM_HEATER_CRYO].add(chan);
        }
        for (int chan = COLD_CHANNEL_FIRST; chan < COLD_CHANNEL_FIRST + N_SET_CHANS; chan++) {
            channelSets[ThermalState.TRIM_HEATER_COLD].add(chan);
        }
        channelSets[ThermalState.TRIM_HEATER_COLD_MYE].add(COLD_CHANNEL_FIRST);
        channelSets[ThermalState.TRIM_HEATER_COLD_PYE].add(COLD_CHANNEL_FIRST + N_SET_CHANS - 1);
        for (int chan = COLD_CHANNEL_FIRST + 1; chan < COLD_CHANNEL_FIRST + 5; chan++) {
            channelSets[ThermalState.TRIM_HEATER_COLD_C].add(chan);
        }
        channelSets[ThermalState.TRIM_HEATER_COLD_MYC].add(COLD_CHANNEL_FIRST + 1);
        channelSets[ThermalState.TRIM_HEATER_COLD_MYC].add(COLD_CHANNEL_FIRST + 2);
        channelSets[ThermalState.TRIM_HEATER_COLD_PYC].add(COLD_CHANNEL_FIRST + 3);
        channelSets[ThermalState.TRIM_HEATER_COLD_PYC].add(COLD_CHANNEL_FIRST + 4);
    }

    /**
     *  Private lookup maps.
     */
    private static final Map<String, Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("VOLTAGE", TYPE_VOLTAGE);
        typeMap.put("CURRENT", TYPE_CURRENT);
        typeMap.put("POWER", TYPE_POWER);
        typeMap.put("TOTALPOWER", TYPE_TOTAL_POWER);
        typeMap.put("TEMP", TYPE_TEMP);
        typeMap.put("MAINVOLTS", TYPE_MAIN_VOLTS);
        typeMap.put("MAINCURR", TYPE_MAIN_CURR);
        typeMap.put("MAINPOWER", TYPE_MAIN_POWER);
        typeMap.put("MAINTEMP", TYPE_MAIN_TEMP);
        typeMap.put("MAINSTAT", TYPE_MAIN_STATUS);
    }

    /**
     *  Configuration parameters.
     */
    @ConfigurationParameter(name=NODE, category="Refrig", isFinal=true)
    private volatile Integer node;      // Heater PS node number
    @ConfigurationParameter(category="Refrig", maxLength=N_SET_CHANS)
    private double[] coldPowerWeights = new double[N_SET_CHANS];
    @ConfigurationParameter(category="Refrig", maxLength=N_SET_CHANS)
    private double[] cryoPowerWeights = new double[N_SET_CHANS];

    /**
     *  Data fields.
     */
    private static final Logger LOG = Logger.getLogger(HeaterPsDevice.class.getName());
    private final HeaterPS htr = new HeaterPS();  // Associated heater PS object
    private final double[] powerValues = new double[ThermalState.NUM_TRIM_HEATERS];
    private final double[] powerWeights = new double[N_HW_CHANS];
    private final double[] totalWeight = new double[ThermalState.NUM_TRIM_HEATERS];
    private final BlockingQueue<Double>[] powerQueues = new BlockingQueue[N_HW_CHANS];
    private final PowerThread[] powerThreads = new PowerThread[N_HW_CHANS];
    private final BlockingQueue<Integer>[] returnQueues = new BlockingQueue[ThermalState.NUM_TRIM_HEATERS];
    private final double[] currents = new double[N_HW_CHANS];  // Read currents 
    private final double[] voltages = new double[N_HW_CHANS];  // Read voltages
    private final boolean[] outputs = new boolean[N_HW_CHANS]; // Read output states
    private boolean initError = false;


    /**
     *  Constructor.
     */
    public HeaterPsDevice()
    {
        super("Heater PS", null, 0, 0, N_HW_CHANS - 1);
        connType = DriverConstants.ConnType.NET;
        devcId = "";
        settlingTime = 1000;   // Wait time (ms) needed to get good current reading after setting voltage
        setColdPowerWeights(new double[]{1.0, 1.0, 1.0, 1.0, 1.0, 1.0});
        setCryoPowerWeights(new double[]{1.0, 1.0, 1.0, 1.0, 1.0, 1.0});
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (node == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "node", "is missing");
        }

        for (int chan = 0; chan < N_HW_CHANS; chan++) {
            powerQueues[chan] = new ArrayBlockingQueue<>(1);
            powerThreads[chan] = new PowerThread(chan, powerQueues[chan]);
            powerThreads[chan].setDaemon(true);
            powerThreads[chan].start();
        }
        for (int chanSet = 0; chanSet < ThermalState.NUM_TRIM_HEATERS; chanSet++) {
            returnQueues[chanSet] = new ArrayBlockingQueue<>(channelSets[chanSet].size());
            for (int chan : channelSets[chanSet]) {
                powerThreads[chan].addReturnQueue(returnQueues[chanSet]);
            }
        }

        fullName = getPath() + " (Heater power supply: " + node + ")";
    }


    /**
     *  Sets cold trim heat power weight factors.
     *
     *  @param  weights  6-element array of weights for cold trim heaters
     */
    @ConfigurationParameterChanger
    public final void setColdPowerWeights(double[] weights)
    {
        checkWeights("cold", weights);
        System.arraycopy(weights, 0, coldPowerWeights, 0, N_SET_CHANS);
        System.arraycopy(weights, 0, powerWeights, COLD_CHANNEL_FIRST, N_SET_CHANS);
        setTotalWeight(ThermalState.TRIM_HEATER_COLD);
        setTotalWeight(ThermalState.TRIM_HEATER_COLD_MYE);
        setTotalWeight(ThermalState.TRIM_HEATER_COLD_PYE);
        setTotalWeight(ThermalState.TRIM_HEATER_COLD_C);
        setTotalWeight(ThermalState.TRIM_HEATER_COLD_MYC);
        setTotalWeight(ThermalState.TRIM_HEATER_COLD_PYC);
    }


    /**
     *  Sets cryo trim heat power weight factors.
     *
     *  @param  weights  6-element array of weights for cryo trim heaters
     */
    @ConfigurationParameterChanger
    public final void setCryoPowerWeights(double[] weights)
    {
        checkWeights("cryo", weights);
        System.arraycopy(weights, 0, cryoPowerWeights, 0, N_SET_CHANS);
        System.arraycopy(weights, 0, powerWeights, CRYO_CHANNEL_FIRST, N_SET_CHANS);
        setTotalWeight(ThermalState.TRIM_HEATER_CRYO);
    }


    /**
     *  Checks power weights for validity
     *
     *  @param  section  The name of the trim heater section
     *  @param  weights  Array of weights
     */
    private void checkWeights(String section, double[] weights)
    {
        if (weights.length != N_SET_CHANS) {
            ErrorUtils.reportConfigError(LOG, getName(), section + "PowerWeights",
                                         "must have exactly " + N_SET_CHANS + " elements");
        }
        for (double value : weights) {
            if (value < 0.0) {
                ErrorUtils.reportConfigError(LOG, getName(), section + "PowerWeights", "element must not be negative");
            }
        }
        
    }


    /**
     *  Sets a total power weight
     *
     *  @param  chanSet  The channel set number
     */
    private void setTotalWeight(int chanSet)
    {
        totalWeight[chanSet] = 0.0;
        for (int chan : channelSets[chanSet]) {
            totalWeight[chanSet] += powerWeights[chan];
        }
        if (totalWeight[chanSet] == 0.0) {
            totalWeight[chanSet] = 1.0;
        }
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            htr.open(node, HeaterPS.PORT_1);
            htr.setMainPowerOn(true);
            htr.setSwitchPeriod(500);
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException e) {
            if (!initError) {  // Avoid reporting consecutive errors
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                htr.close();
            }
            catch (DriverException ce) {
                // Will happen if open was unsuccessful
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            htr.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
        for (int j = 0; j < N_HW_CHANS; j++) {
            currents[j] = 0;
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string (not used)
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        Integer iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }
        int numChan = (iType == TYPE_VOLTAGE || iType == TYPE_CURRENT || iType == TYPE_POWER) ? N_HW_CHANS :
                      (iType == TYPE_TOTAL_POWER) ? ThermalState.NUM_TRIM_HEATERS : 1;
        if (hwChan < 0 || hwChan >= numChan) {
            ErrorUtils.reportChannelError(LOG, name, "HW channel", hwChan);
        }
        return new int[]{iType, 0};
    }


    /**
     *  Reads all heater channels as a group.
     */
    @Override
    protected void readChannelGroup()
    {
        if (!isOnline()) return;
        try {
            for (int chan = 0; chan < N_HW_CHANS; chan++) {
                currents[chan] = htr.readCurrent(chan);
                voltages[chan] = htr.getVoltage(chan);
                outputs[chan] = htr.getOutput(chan);
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading currents & voltages from {0}: {1}", new Object[]{fullName, e});
            setOnline(false);
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (isOnline()) {
            try {
                switch (type) {
                case TYPE_VOLTAGE:
                    value = outputs[hwChan] ? voltages[hwChan] : 0.0;
                    break;
                case TYPE_CURRENT:
                    value = currents[hwChan];
                    break;
                case TYPE_POWER:
                    value = outputs[hwChan] ? voltages[hwChan] * currents[hwChan] : 0.0;
                    break;
                case TYPE_TOTAL_POWER:
                    value = 0.0;
                    for (int chan : channelSets[hwChan]) {
                        value += outputs[chan] ? voltages[chan] * currents[chan] : 0.0;
                    }
                    break;
                case TYPE_TEMP:
                    value = htr.readBoardTemperature();
                    break;
                case TYPE_MAIN_VOLTS:
                    value = htr.readMainVoltage();
                    break;
                case TYPE_MAIN_CURR:
                    value = htr.readMainCurrent();
                    break;
                case TYPE_MAIN_POWER:
                    value = htr.readMainVoltage() * htr.readMainCurrent();
                    break;
                case TYPE_MAIN_TEMP:
                    value = htr.readMainTemperature();
                    break;
                case TYPE_MAIN_STATUS:
                    value = htr.readMainStatus();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading value from {0}: {1}", new Object[]{fullName, e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Enables/disables the output of a set of channels.
     *
     *  @param  chanSet  The channel set (cryo or cold)
     *  @param  enable  Whether to enable
     */
    @Override
    public void enableOutput(int chanSet, boolean enable)
    {
        for (int chan : channelSets[chanSet]) {
            super.enableOutput(chan, enable);
        }
    }


    /**
     *  Gets the enabled state of a set of channels.
     *
     *  @param  chanSet  The channel set (cryo or cold)
     *  @return  Whether enabled
     */
    @Override
    public boolean isEnabled(int chanSet)
    {
        boolean enabled = false;
        for (int chan : channelSets[chanSet]) {
            enabled |= super.isEnabled(chan);
        }
        return enabled;
    }


    /**
     *  Sets the power for a set of channels.
     *
     *  @param  chanSet  The channel set (cryo or cold)
     *  @param  value  The power to set, or NaN to use current value
     */
    @Override
    public void setPower(int chanSet, double value)
    {
        powerValues[chanSet] = value;
        int count = 0;
        for (int chan : channelSets[chanSet]) {
            try {
                powerQueues[chan].put(value * powerWeights[chan] / totalWeight[chanSet]);
                count++;
            }
            catch (InterruptedException e) {}
        }
        while (count-- > 0) {
            try {
                returnQueues[chanSet].take();
            }
            catch (InterruptedException e) {}
        }
    }


    /**
     *  Sets the power for a channel.
     *
     *  @param  chan   The channel
     *  @param  value  The power to set, or NaN to use current value
     */
    private void setChannelPower(int chan, double value)
    {
        super.setPower(chan, value);
    }


    /**
     *  Maintains the power for all channels at their set values.
     */
    @Override
    protected void maintainPower()
    {
        for (int chanSet = 0; chanSet < ThermalState.NUM_TRIM_HEATERS; chanSet++) {
            double value = powerValues[chanSet];
            if (value != 0.0) {
                setPower(chanSet, value);
            }
        }
    }


    /**
     *  Gets the power of a set of channels.
     *
     *  @param  chanSet  The channel set (cryo or cold)
     *  @return  Whether enabled
     */
    @Override
    public double getPower(int chanSet)
    {
        double power = 0.0;
        for (int chan : channelSets[chanSet]) {
            power += super.getPower(chan);
        }
        return power;
    }


    /**
     *  Gets whether a voltage setting error occurred.
     *
     *  @param  chanSet  The channel set (cryo or cold)
     *  @return  The error state
     */
    @Override
    public boolean hasVoltError(int chanSet)
    {
        boolean state = false;
        for (int chan : channelSets[chanSet]) {
            state |= super.hasVoltError(chan);
        }
        return state;
    }


    /**
     *  Gets whether a "no load" error occurred.
     *
     *  @param  chanSet  The channel set (cryo or cold)
     *  @return  The error state
     */
    @Override
    public boolean hasNoLoad(int chanSet)
    {
        boolean state = false;
        for (int chan : channelSets[chanSet]) {
            state |= super.hasNoLoad(chan);
        }
        return state;
    }


    /**
     *  Sets the output state.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The output state to set, true or false
     *  @throws  DriverException
     */
    @Override
    public void setOutput(int chan, boolean value) throws DriverException
    {
        htr.setOutput(chan, value);
    }


    /**
     *  Gets the output state.
     *
     *  @param  chan  The hardware channel
     *  @return  The output state
     *  @throws  DriverException
     */
    @Override
    public boolean getOutput(int chan) throws DriverException
    {
        return htr.getOutput(chan);
    }


    /**
     *  Sets the voltage.
     *
     *  @param  chan   The hardware channel
     *  @param  value  The voltage to set
     *  @throws  DriverException
     */
    @Override
    public void setVoltage(int chan, double value) throws DriverException
    {
        htr.setVoltage(chan, value);
    }


    /**
     *  Reads the voltage.
     *
     *  @param  chan  The hardware channel
     *  @return  The read voltage
     *  @throws  DriverException
     */
    @Override
    public double readVoltage(int chan) throws DriverException
    {
        return htr.getVoltage(chan);
    }


    /**
     *  Sets the current.
     *
     *  This is a no-op.
     * 
     *  @param  chan   The hardware channel (not used)
     *  @param  value  The current to set
     */
    @Override
    public void setCurrent(int chan, double value)
    {
    }


    /**
     *  Reads the current.
     *
     *  @param  chan  The hardware channel
     *  @return  The read current
     *  @throws  DriverException
     */
    @Override
    public double readCurrent(int chan) throws DriverException
    {
        return htr.readCurrent(chan);
    }


    /**
     *  Turns on/off the bulk power supply.
     *
     *  @param  on  Whether to turn on
     */
    public void setBulkPowerOn(boolean on)
    {
        try {
            htr.setMainPowerOn(on);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting bulk PS state for {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Gets the on state of the bulk power supply.
     *
     *  @return  Whether on, or null if offline or error
     */
    public Boolean isBulkPowerOn()
    {
        if (isOnline()) {
            try {
                return (htr.getMainIoStatus() & RebBulkPS.IOSTAT_REM_ON) != 0;
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading bulk PS state from {0}: {1}", new Object[]{fullName, e});
            }
        }
        return null;
    }

}
