package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.ColdCompValves;
import org.lsst.ccs.subsystem.refrig.constants.CompSwConds;
import org.lsst.ccs.subsystem.refrig.constants.CompSwitches;
import org.lsst.ccs.subsystem.refrig.constants.OilSepHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.constants.VfdState;
import org.lsst.ccs.subsystem.refrig.data.ColdCompState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Controls a refrigeration cold compressor.
 *
 *  @author Owen Saxton
 */
public class ColdCompressor extends Compressor {

    private static final int[] maqChannels = new int[ColdCompValves.NUM_VALVES];
    static {
        maqChannels[ColdCompValves.VALVE_HGB] = CompMaq20Device.CHAN_HGB_VALVE;
        maqChannels[ColdCompValves.VALVE_COOLANT] = CompMaq20Device.CHAN_COOLANT_VALVE;
        maqChannels[ColdCompValves.VALVE_EEPR] = CompMaq20Device.CHAN_EEPR_VALVE;
    }
    
    @LookupName
    private String name;
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private A1000Device vfdDevc;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private AgentStateService stateService;

    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile int discPressDelayTime;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double liquidTempImmedLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile double liquidTempDelayLimit;
    @ConfigurationParameter(category=COMP_LIMITS, isFinal=true)
    private volatile int liquidTempDelayTime;

    private String liquidTempChan;  // From Groovy file

    private static final Logger LOG = Logger.getLogger(ColdCompressor.class.getName());
    private final ColdCompState coldState;


    /**
     *  Constructor.
     */
    public ColdCompressor() {
        super(new ColdCompState());
        coldState = (ColdCompState)getState();
    }
            

    /**
     *  Build phase
     */
    @Override
    public void build() {
        // Register heater states
        stateService.registerState(OilSepHeaterState.class, "Oil separator heater state", this);
        stateService.updateAgentComponentState(this, OilSepHeaterState.OFFLINE);
    }


     /**
     *  Initializes the cold compressor control.
     */
    @Override
    public void postInit()
    {
        super.postInit();
        if (vfdDevc == null) {
            ErrorUtils.reportConfigError(LOG, name, "vfdDevc", "not specified");
        }
        switchDevices[CompSwitches.SW_HEATER] = maq20Devc;
        switchChannels[CompSwitches.SW_HEATER] = CompMaq20Device.SW_OIL_SEP_HEATER;
        Channel liquidTemp = (liquidTempChan != null) ? channelMap.get(liquidTempChan) : null;
        if (liquidTemp == null) {
            ErrorUtils.reportConfigError(LOG, name, "liquidTempChan", "not specified or not defined");
        }
        limitData[CompSwConds.SWC_LIQUID_TEMP].channel = liquidTemp;
    }


    /**
     *  Sets the liquid temperature immediate limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempImmedLimit(double value)
    {
        liquidTempImmedLimit = value;
        limitData[CompSwConds.SWC_LIQUID_TEMP].immedLimit = value;
    }


    /**
     *  Sets the liquid temperature delayed limit.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayLimit(double value)
    {
        liquidTempDelayLimit = value;
        limitData[CompSwConds.SWC_LIQUID_TEMP].delayLimit = value;
    }


    /**
     *  Sets the liquid temperature delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setLiquidTempDelayTime(int value)
    {
        liquidTempDelayTime = value;
        limitData[CompSwConds.SWC_LIQUID_TEMP].delayTime = 1000 * value;
    }


    /**
     *  Sets the discharge pressure delay.
     *
     *  @param  value  The value to set
     */
    @ConfigurationParameterChanger
    public void setDiscPressDelayTime(int value)
    {
        discPressDelayTime = value;
        limitData[CompSwConds.SWC_DISC_PRESS].delayTime = 1000 * value;
    }


    /**
     *  Command to get the valid valve names.
     *
     *  @return  The index value
     */
    @Command(type=Command.CommandType.QUERY, description="Get the valid valve names", level=0)
    public List<String> getValveNames()
    {
        List<String> names = new ArrayList<>();
        for (String vName : ColdCompValves.NAME_MAP.keySet()) {
            names.add(vName);
        }
        return names;
    }


    /**
     *  Command to set a valve position.
     *
     *  @param  valve  The valve name
     *  @param  posn   The valve position  (0 - 1)
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Set a cold compressor's valve position")
    public synchronized void setValvePosition(@Argument(description="Valve name") String valve,
                                              @Argument(description="Valve position") double posn) throws RefrigException
    {
        gotCommand = true;
        Integer valveId = ColdCompValves.NAME_MAP.get(valve);
        if (valveId == null) {
            throw new RefrigException("Invalid cold valve name: " + valve);
        }
        maq20Devc.setValvePosition(maqChannels[valveId], posn);
    }


    /**
     *  Command to set the VFD frequency.
     *
     *  @param  freq  The frequency
     *  @throws  RefrigException
     */
     @Command(type=Command.CommandType.ACTION, description="Set a cold compressor's VFD frequency")
    public synchronized void setVfdFrequency(@Argument(description="Frequency") double freq) throws RefrigException
    {
        gotCommand = true;
        vfdDevc.setFrequency(freq);
    }


    /**
     *  Command to reset a VFD fault condition.
     *
     *  @throws  RefrigException
     */
    @Command(type=Command.CommandType.ACTION, description="Reset a cold compressor's VFD fault")
    public synchronized void resetVfdFault() throws RefrigException
    {
        gotCommand = true;
        vfdDevc.resetFault();
    }


    /**
     *  Updates the compressor state.
     *
     *  This should be called at regular (short) intervals to maintain the correct state
     *
     *  @return  Whether any value changed
     */
    @Override
    protected synchronized boolean updateState()
    {
        boolean changed = super.updateState();
        if (changed) {
            SwitchState swState = coldState.getSwitchState(CompSwitches.SW_HEATER);
            OilSepHeaterState osState = swState == SwitchState.ON ? OilSepHeaterState.ON :
                                        swState == SwitchState.OFF ? OilSepHeaterState.OFF : OilSepHeaterState.OFFLINE;
            stateService.updateAgentComponentState(this, osState);
        }
        for (int valveId = 0; valveId < ColdCompValves.NUM_VALVES; valveId++) {
            double posn = maq20Devc.getValvePosition(maqChannels[valveId]);
            if (!areEqual(posn, coldState.getValvePosition(valveId))) {
                coldState.setValvePosition(valveId, posn);
                changed = true;
            }
        }
        double freq = vfdDevc.getFrequency();
        if (!areEqual(freq, coldState.getVfdFrequency())) {
            coldState.setVfdFrequency(freq);
            changed = true;
        }
        VfdState vfdState = vfdDevc.getState();
        if (vfdState != coldState.getVfdState()) {
            coldState.setVfdState(vfdState);
            changed = true;
        }
        return changed;
    }

}
