package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.CompConditions;
import org.lsst.ccs.subsystem.refrig.constants.CompLatches;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.constants.ConditionState;
import org.lsst.ccs.subsystem.refrig.constants.LatchState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Simulates the compressor Pluto PLC.
 *
 *  @author Owen Saxton
 */
public class SimCompPlutoDevice extends CompPlutoDevice {

    /**
     *  Constants
     */
    private static final Map<String, Integer> latchMap = new LinkedHashMap<>();
    static {
        latchMap.put("AfterTmp", CompLatches.LATCH_AFTER_COOLER);
        latchMap.put("DischPrs", CompLatches.LATCH_DISCHARGE_PRESS);
        latchMap.put("DischTmp", CompLatches.LATCH_DISCHARGE_TEMP);
        latchMap.put("ExtPermit", CompLatches.LATCH_EXT_PERMIT);
        latchMap.put("LiquidTmp", CompLatches.LATCH_LIQUID_TEMP);
        latchMap.put("OilLevel", CompLatches.LATCH_OIL_LEVEL);
        latchMap.put("Power", CompLatches.LATCH_POWER);
        latchMap.put("SensValid", CompLatches.LATCH_SENSORS_VALID);
        latchMap.put("SmokeDetc", CompLatches.LATCH_SMOKE_DETC);
        latchMap.put("SuctnTmp", CompLatches.LATCH_SUCTION_TEMP);
    }
    private static final Map<String, Integer> conditionMap = new LinkedHashMap<>();
    static {
        //conditionMap.put("CmpEnabled", CompConditions.COND_CMP_ENABLED);
        conditionMap.put("CmpOn8Hrs", CompConditions.COND_CMP_ON_8HRS);
        //conditionMap.put("CmpPowered", CompConditions.COND_CMP_POWERED);
        //conditionMap.put("CmpWaiting", CompConditions.COND_CMP_WAITING);
        conditionMap.put("CurrentValid", CompConditions.COND_CURRENT_VALID);
        conditionMap.put("CurrSensErr", CompConditions.COND_CURR_SENSOR_ERR);
        conditionMap.put("DiscPrsValid", CompConditions.COND_DISC_PRESS_VALID);
        conditionMap.put("DiscTmpValid", CompConditions.COND_DISC_TEMP_VALID);
        conditionMap.put("KeySwitchOn", CompConditions.COND_KEYSWITCH_ON);
        //conditionMap.put("LatchesClear", CompConditions.COND_LATCHES_CLEAR);
        conditionMap.put("LiqdTmpValid", CompConditions.COND_LIQD_TEMP_VALID);
        conditionMap.put("OilLvlValid", CompConditions.COND_OIL_LEVEL_VALID);
        //conditionMap.put("PowerLedOn", CompConditions.COND_POWER_LED);
        conditionMap.put("SuctPrsValid", CompConditions.COND_SUCT_PRESS_VALID);
        conditionMap.put("SuctTmpValid", CompConditions.COND_SUCT_TEMP_VALID);
        conditionMap.put("VoltageValid", CompConditions.COND_VOLTAGE_VALID);
    }

    /**
     *  Data fields.
     */
    private static final Logger LOG = Logger.getLogger(SimCompPlutoDevice.class.getName());
    private final boolean[] coldSwitches = new boolean[NUM_COLD_SWITCHES];
    private final boolean[] cryoSwitches = new boolean[NUM_CRYO_SWITCHES];
    private final LatchState[] latches = new LatchState[CompLatches.NUM_LATCHES];
    private final ConditionState[] conditions = new ConditionState[CompConditions.NUM_CONDITIONS];
    private boolean[] switches;
    private long powerTime = 0;
    private final Map<String, Integer> channelMap = new LinkedHashMap<>();
    private final List<Double> channelValues = new ArrayList<>();
    private boolean plcActive = true;


    /**
     *  Constructor.
     */
    public SimCompPlutoDevice()
    {
        Arrays.fill(latches, LatchState.CLEAR);
        Arrays.fill(conditions, ConditionState.YES);
        conditions[CompConditions.COND_CMP_ENABLED] = ConditionState.NO;
        conditions[CompConditions.COND_CMP_WAITING] = ConditionState.NO;
        conditions[CompConditions.COND_CMP_POWERED] = ConditionState.NO;
        conditions[CompConditions.COND_POWER_LED] = ConditionState.NO;
    }


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
        LOG.log(Level.INFO, "Connected to {0} (simulated Pluto gateway)", getPath());
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if there is a parameter error
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan >= 100) {
            ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
        }
        int index = channelValues.size();
        channelMap.put(name, index);
        channelValues.add((double)index);
        return new int[]{index, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return channelValues.get(type);
    }


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    @Override
    public void setType(int type)
    {
        switches = type == CompTypes.TYPE_COLD ? coldSwitches : cryoSwitches;
    }


    /**
     *  Sets a switch on or off.
     *
     *  For the vacuum Pluto, this is implemented as a pair of push buttons,
     *  one for on, one for off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        switches[sw] = on;
        if (sw == SW_ENABLE) {
            if (on) {
                powerTime = System.currentTimeMillis() + 30000;
                conditions[CompConditions.COND_CMP_ENABLED] = ConditionState.YES;
                if (areLatchesClear()) {
                    conditions[CompConditions.COND_CMP_WAITING] = ConditionState.YES;
                    conditions[CompConditions.COND_CMP_POWERED] = ConditionState.YES;
                }
            }
            else {
                powerTime = 0;
                conditions[CompConditions.COND_CMP_ENABLED] = ConditionState.NO;
                conditions[CompConditions.COND_CMP_WAITING] = ConditionState.NO;
                conditions[CompConditions.COND_CMP_POWERED] = ConditionState.NO;
            }
        }
        else if (sw == SW_LIGHTS) {
            conditions[CompConditions.COND_POWER_LED] = on ? ConditionState.YES : ConditionState.NO;
        }
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        if (powerTime != 0 && System.currentTimeMillis() >= powerTime) {
            conditions[CompConditions.COND_CMP_WAITING] = ConditionState.NO;
            powerTime = 0;
        }
        return switches[sw];
    }


    /**
     *  Gets the state of a latched condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    @Override
    public LatchState getLatchState(int cond)
    {
        return latches[cond];
    }


    /**
     *  Gets the state of a condition.
     * 
     *  @param  cond  The condition number
     *  @return  The condition state
     */
    @Override
    public ConditionState getConditionState(int cond)
    {
        return conditions[cond];
    }


    /**
     *  Resets all latches.
     */
    @Override
    public void resetLatches()
    {
        for (int cond = 0; cond < latches.length; cond++) {
            LatchState state = latches[cond];
            latches[cond] = state == LatchState.LATCHED ? LatchState.CLEAR : state;
        }
        if (areLatchesClear()) {
            conditions[CompConditions.COND_LATCHES_CLEAR] = ConditionState.YES;
            if (switches[SW_ENABLE]) {
                conditions[CompConditions.COND_CMP_POWERED] = ConditionState.YES;
            }
        }
    }


    /**
     *  Checks whether PLC is active
     *
     *  @return  Whether active
     */
    @Override
    public Boolean isPLCActive()
    {
        return plcActive;
    }


    /**
     *  Gets the PLC error code.
     *
     *  @return  The error code
     */
    @Override
    public int getErrorCode()
    {
        return 42;
    }


    private boolean areLatchesClear()
    {
        boolean clear = true;
        for (LatchState state : latches) {
            if (state != LatchState.CLEAR) {
                clear = false;
                break;
            }
        }
        return clear;
    }


    @Command(type=Command.CommandType.QUERY, description="Get the list of latched condition names")
    public String getLatchNames()
    {
        return latchMap.keySet().toString();
    }

    @Command(type=Command.CommandType.ACTION, description="Make a latched condition active")
    public void setLatchActive(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[getLatch(cond)] = LatchState.ACTIVE;
        conditions[CompConditions.COND_CMP_POWERED] = ConditionState.NO;
        conditions[CompConditions.COND_CMP_WAITING] = ConditionState.NO;
        conditions[CompConditions.COND_LATCHES_CLEAR] = ConditionState.NO;
    }

    @Command(type=Command.CommandType.ACTION, description="Make a latched condition latched")
    public void setLatchLatched(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[getLatch(cond)] = LatchState.LATCHED;
        conditions[CompConditions.COND_CMP_POWERED] = ConditionState.NO;
        conditions[CompConditions.COND_CMP_WAITING] = ConditionState.NO;
        conditions[CompConditions.COND_LATCHES_CLEAR] = ConditionState.NO;
    }

    @Command(type=Command.CommandType.ACTION, description="Put a latched condition into warning state")
    public void setLatchWarning(@Argument(description="Latched condition name") String cond) throws RefrigException {
        latches[getLatch(cond)] = LatchState.WARNING;
    }

    private int getLatch(String name) throws RefrigException {
        Integer cond = latchMap.get(name);
        if (cond == null) {
            throw new RefrigException("Unknown latched condition name: " + name);
        }
        return cond;
    }

    @Command(type=Command.CommandType.QUERY, description="Get the list of condition names")
    public String getConditionNames()
    {
        return conditionMap.keySet().toString();
    }

    @Command(type=Command.CommandType.ACTION, description="Set the state of a condition")
    public void setCondition(@Argument(description="Running condition name") String cond,
                             @Argument(description="Condition on state") boolean set) throws RefrigException {
        Integer condNum = conditionMap.get(cond);
        if (condNum == null) {
            throw new RefrigException("Unknown condition name: " + cond);
        }
        if (condNum != CompConditions.COND_CMP_POWERED) {
            conditions[condNum] = set ? ConditionState.YES : ConditionState.NO;
        }
    }

    @Command(type=Command.CommandType.ACTION, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chan,
                                @Argument(description="Channel value") double value) throws RefrigException
    {
        Integer index = channelMap.get(chan);
        if (index == null) {
            throw new RefrigException("Invalid channel name");
        }
        channelValues.set(index, value);
    }

    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

    @Command(type=Command.CommandType.ACTION, description="Set whether PLC is active")
    public void setPLCActive(@Argument(description="PLC active state") boolean set) {
        plcActive = set;
    }

}
