package org.lsst.ccs.subsystem.refrig;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.CompTypes;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Handles the simulated compressor MAQ20 device.
 *
 *  @author Owen Saxton
 */
public class SimCompMaq20Device extends CompMaq20Device {

    private static final Logger LOG = Logger.getLogger(SimCompMaq20Device.class.getName());
    private final boolean[] coldSwitches = new boolean[NUM_COLD_SWITCHES];
    private final boolean[] cryoSwitches = new boolean[NUM_CRYO_SWITCHES];
    private boolean[] switches;
    private final Map<String, Integer> channelMap = new HashMap<>();
    private final List<Double> channelValues = new ArrayList<>();
    private final Map<Integer, Double> valveMap = new HashMap<>();


    /**
     *  Performs basic initialization.
     */
    @Override
    protected void initDevice()
    {
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
        LOG.log(Level.INFO, "Connected to {0} (simulated MAQ20)", getPath());
    }


    /**
     *  Closes device connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if there is a parameter error
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan >= 100) {
            ErrorUtils.reportChannelError(LOG, name, "hwChan", hwChan);
        }
        int index = channelValues.size();
        channelMap.put(name, index);
        channelValues.add((double)index);
        return new int[]{index, 0};
    }


    /**
     *  Reads all referenced channels.
     */
    @Override
    protected void readChannelGroup()
    {
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The read value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        return channelValues.get(type) + 0.2 * (Math.random() - 0.5);
    }


    /**
     *  Sets the compressor type.
     *
     *  @param  type  The type (cold or cryo)
     */
    @Override
    public void setType(int type)
    {
        switches = type == CompTypes.TYPE_COLD ? coldSwitches : cryoSwitches;
    }


    /**
     *  Sets a switch on or off.
     *
     *  @param  sw  The switch number.
     *  @param  on  The on state to set: true or false
     */
    @Override
    public void setSwitchOn(int sw, boolean on)
    {
        switches[sw - 1] = on;
    }


    /**
     *  Gets the on state of a switch.
     *
     *  @param  sw  The switch number.
     *  @return  Whether the switch is on
     */
    @Override
    public Boolean isSwitchOn(int sw)
    {
        return switches[sw - 1];
    }


    /**
     *  Sets a valve position.
     *
     *  @param  valve  The valve channel number
     *  @param  posn   Requested position (0 - 1)
     */
    @Override
    public void setValvePosition(int valve, double posn)
    {
        valveMap.put(valve, posn);
    }


    /**
     *  Gets a valve position.
     *
     *  @param  valve  The valve channel number
     *  @return  Valve position (0 - 1), or NaN if error
     */
    @Override
    public double getValvePosition(int valve)
    {
        Double value = valveMap.get(valve);
        return value == null ? 0.0 : value;
    }


    @Command(type=Command.CommandType.ACTION, description="Set a channel value")
    public void setChannelValue(@Argument(description="Channel name") String chan,
                                @Argument(description="Channel value") double value) throws RefrigException
    {
        Integer index = channelMap.get(chan);
        if (index == null) {
            throw new RefrigException("Invalid channel name");
        }
        channelValues.set(index, value);
    }

    @Command(type=Command.CommandType.QUERY, description="Get the list of channel names")
    public String getChannelNames()
    {
        return channelMap.keySet().toString();
    }

}
