package org.lsst.ccs.subsystem.refrig;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mcc.MccUsb;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles a Measurement Computing DAQ box.
 *
 *  @author Owen Saxton
 */
public class MCCDevice extends Device {

    /**
     *  Constants.
     */
    final static int
        N_DEV_CHANS = 8;
    final static float[]
        RTD_COEFFS = {100.0F, 0.003908F, -5.8019E-7F, -4.2735E-12F};

    /**
     *  Private lookup maps.
     */
    private final static Map<String,Integer> didMap = new HashMap<>();
    static {
        didMap.put("TC_AI", MccUsb.USB_TC_AI_DID);
        didMap.put("TC",    MccUsb.USB_TC_DID);
        didMap.put("TEMP",  MccUsb.USB_TEMP_DID);
    }

    private final static Map<String,Integer> typeMap = new HashMap<>();
    static {
        typeMap.put("TEMP",   Channel.TYPE_TEMP);
        typeMap.put("VOLTS",  Channel.TYPE_VOLTS);
        typeMap.put("SWITCH", Channel.TYPE_SWITCH);
    }

    private final static Map<String,Integer> tSbTypMap = new HashMap<>();
    static {
        tSbTypMap.put("TC",  MccUsb.STP_THERMOCOUPLE);
        tSbTypMap.put("RTD", MccUsb.STP_RTD);
    }

    private final static Map<String,Integer> tcSbTypMap = new HashMap<>();
    static {
        tcSbTypMap.put("J", MccUsb.TC_TYPE_J);
        tcSbTypMap.put("K", MccUsb.TC_TYPE_K);
        tcSbTypMap.put("T", MccUsb.TC_TYPE_T);
        tcSbTypMap.put("E", MccUsb.TC_TYPE_E);
        tcSbTypMap.put("R", MccUsb.TC_TYPE_R);
        tcSbTypMap.put("S", MccUsb.TC_TYPE_S);
        tcSbTypMap.put("B", MccUsb.TC_TYPE_B);
        tcSbTypMap.put("N", MccUsb.TC_TYPE_N);
    }

    private final static Map<String,Integer> rtSbTypMap = new HashMap<>();
    static {
        rtSbTypMap.put("2WIRE1", MccUsb.TCT_2WIRE_1SENSOR);
        rtSbTypMap.put("2WIRE2", MccUsb.TCT_2WIRE_2SENSOR);
        rtSbTypMap.put("3WIRE",  MccUsb.TCT_3WIRE);
        rtSbTypMap.put("4WIRE",  MccUsb.TCT_4WIRE);
    }

    private final static Map<String,Integer> vSbTypMap = new HashMap<>();
    static {
        vSbTypMap.put("10V",   MccUsb.RANGE_10V);
        vSbTypMap.put("5V",    MccUsb.RANGE_5V);
        vSbTypMap.put("2.5V",  MccUsb.RANGE_2_5V);
        vSbTypMap.put("1.25V", MccUsb.RANGE_1_25V);
    }


    /**
     *  Data fields.
     */
    private String  devName;  // MCC device name
    private String  serial;   // USB serial number

    private static final Logger LOG = Logger.getLogger(MCCDevice.class.getName());
    private int     did;      // MCC device ID
    private final MccUsb  mcc = new MccUsb();  // Associated USB object
    private int     tcMask;   // Mask of available thermocouple channels
    private int     rtMask;   // Mask of available RT channels
    private int     vMask;    // Mask of available voltage channels
    private boolean initError = false;


    /**
     *  Constructor.
     *
     *  @param  dName     The MCC device name
     *  @param  serialNo  The USB serial number
     */
    public MCCDevice(String dName, String serialNo)
    {
        devName = dName;
        serial = serialNo;
    }


    public MCCDevice() {
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        if (devName == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "devName", "is missing");
        }
        Integer iDid = didMap.get(devName);
        if (iDid == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "devName", "is invalid: " + devName);
        }
        did = iDid;
        if (did == MccUsb.USB_TC_AI_DID) {
            tcMask = 0x0f;
            vMask = 0xf0;
        }
        else if (did == MccUsb.USB_TC_DID) {
            tcMask = 0xff;
        }
        else if (did == MccUsb.USB_TEMP_DID) {
            tcMask = 0xff;
            rtMask = 0xff;
        }
        fullName = "MCC " + devName + " module"
                     + (serial == null ? "" : " (" + serial + ")");
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            mcc.open(did, serial, true);
            mcc.blink();
            for (int line = 0, mask = lineMask; line < N_DEV_CHANS;
                 line++, mask >>= 1) {
                mcc.dioConfigBit(line, 1 - (mask & 1));
            }
            if (did == MccUsb.USB_TC_AI_DID) {
                for (int j = 0; j < N_DEV_CHANS; j++) {
                    mcc.configAlarm(j, 0, 0, 0f, 0f);
                }
            }
            else if (did == MccUsb.USB_TEMP_DID) {
                for (int j = 0; j < N_DEV_CHANS; j += 2) {
                    mcc.setSensorType(j, MccUsb.STP_DISABLED);
                }
            }
            initSensors();
            setOutputLines();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
            setOnline(true);
        }
        catch (DriverException | ChannelInitializationException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                mcc.close();
            }
            catch (DriverException ce) {
                // Can happen if open failed
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            mcc.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype)
        throws Exception
    {
        Integer iSubtype = 0, iType = typeMap.get(type.toUpperCase());
        if (iType == null) {
            ErrorUtils.reportChannelError(LOG, name, "type", type);
        }

        String sbtype = subtype.toUpperCase();
        if (iType == Channel.TYPE_TEMP) {
            String[] sbtypes = sbtype.split(":", -1);
            if (sbtypes.length < 2) {
                ErrorUtils.reportChannelError(LOG, name, "subtype", subtype);
            }
            iSubtype = tSbTypMap.get(sbtypes[0]);
            if (iSubtype == null) {
                ErrorUtils.reportChannelError(LOG, name, "subtype", subtype);
            }
            Integer iStp = (iSubtype == MccUsb.STP_RTD)
                             ? rtSbTypMap.get(sbtypes[1])
                             : tcSbTypMap.get(sbtypes[1]);
            if (iStp == null) {
                ErrorUtils.reportChannelError(LOG, name, "subtype", subtype);
            }
            iSubtype |= (iStp << 8);
        }
        else if (iType == Channel.TYPE_VOLTS) {
            iSubtype = vSbTypMap.get(sbtype);
            if (iSubtype == null) {
                ErrorUtils.reportChannelError(LOG, name, "subtype", subtype);
            }
        }

        boolean chanOk = true;
        if (iType == Channel.TYPE_TEMP) {
            int mask = (iSubtype & 0xff) == MccUsb.STP_RTD ? rtMask : tcMask;
            if ((1 << hwChan & mask) == 0) {
                chanOk = false;
            }
        }
        else if (iType == Channel.TYPE_VOLTS) {
            if ((1 << hwChan & vMask) == 0) {
                chanOk = false;
            }
        }
        else {
            if (hwChan < 0 || hwChan >= N_DEV_CHANS) {
                chanOk = false;
            }
        }
        if (!chanOk) {
            ErrorUtils.reportChannelError(LOG, name, "hw channel number", hwChan);
        }

        return new int[]{iType, iSubtype};
    }


    /**
     *  Initializes a channel.
     *
     *  @param  name     The channel name
     *  @param  id       The channel ID
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @param  subtype  The channel subtype returned by checkChannel.
     */
    @Override
    protected void initChannel(String name, int id, int hwChan, int type, int subtype)
    {
        try {
            if (type == Channel.TYPE_TEMP) {
                int sType = subtype & 0xff, tType = subtype >> 8;
                mcc.setSensorType(hwChan, sType);
                if (sType == MccUsb.STP_RTD) {
                    mcc.setThermConnType(hwChan, tType);
                    mcc.setExcitation(hwChan, MccUsb.CEX_210UA);
                    mcc.setGain(hwChan, MccUsb.GAIN_4X);
                    mcc.setGain(hwChan + 1, MccUsb.GAIN_4X);
                    for (int j = 0; j < RTD_COEFFS.length; j++) {
                        mcc.setCoefficient(hwChan, j, RTD_COEFFS[j]);
                    }
                }
                else {
                    mcc.setTCType(hwChan, tType);
                }
            }
            else if (type == Channel.TYPE_VOLTS) {
                mcc.setGain(hwChan, subtype);
                mcc.setVoltageConnType(hwChan, MccUsb.VCT_DIFFERENTIAL);
            }
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error configuring {0}: {1}", new Object[]{fullName, e});
            throw new ChannelInitializationException(e);            
        }
    }

    private class ChannelInitializationException extends RuntimeException {
        
        public ChannelInitializationException(DriverException ex) {
            super(ex);
        }
    }

    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  Channel value
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        if (!isOnline()) {
        }
        else if (type == Channel.TYPE_SWITCH) {
            try {
                value = mcc.dioInBit(hwChan);
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading DIO line: {0}", e);
                setOnline(false);
            }
        }
        else {
            try {
                value = mcc.adcIn(hwChan, 0);
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading ADC: {0}", e);
                setOnline(false);
            }
        }

        return value;
    }


    /**
     *  Checks an output line number.
     *
     *  @param  name  The name of the output line.
     *  @param  line  The hardware line number of the output line.
     *  @throws  Exception  If line invalid
     */
    @Override
    protected void checkHwLine(String name, int line) throws Exception
    {
        if (line < 0 || line >= N_DEV_CHANS) {
            ErrorUtils.reportChannelError(LOG, name, "line", line);
        }
        addLine(line);
    }


    /**
     *  Sets an output line on or off.
     *
     *  @param  line  The output line number.
     *  @param  on    The on state to set: true or false
     */
    @Override
    protected void setHwLine(int line, boolean on)
    {
        try {
            mcc.dioOutBit(line, on ? 1 : 0);
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error setting DIO line: {0}", e);
            setOnline(false);
        }
    }


    /**
     *  Gets the set state of an output line.
     *
     *  @param  line  The output line number.
     *  @return  Whether the line is on
     */
    @Override
    protected Boolean isHwLineSet(int line)
    {
        try {
            return mcc.dioInBit(line) != 0;
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error reading DIO line: {0}", e);
            setOnline(false);
            return false;
        }
    }

}
