package org.lsst.ccs.subsystem.refrig;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.monitor.Channel;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.common.MonitorTaskControl;
import org.lsst.ccs.subsystem.refrig.constants.ColdState;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterState;
import org.lsst.ccs.subsystem.refrig.constants.RefrigAgentProperties;
import org.lsst.ccs.subsystem.refrig.constants.RefrigConstants;
import org.lsst.ccs.subsystem.refrig.constants.SwitchState;
import org.lsst.ccs.subsystem.refrig.constants.ThermalAlert;
import org.lsst.ccs.subsystem.refrig.constants.TrimHeaterOpState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;
import org.lsst.ccs.subsystem.refrig.data.ThermalState;
import org.lsst.ccs.subsystem.refrig.data.RefrigUtils;

/**
 *  Implements the refrigeration thermal control subsystem.
 *
 *  @author Owen Saxton
 */
public class ThermalMain extends Subsystem implements HasLifecycle, AgentPresenceListener, StatusMessageListener {

    /**
     *  Inner class for holding trim heater parameters.
     */
    static class TrimHeater {

        TrimHeaterControl tempCtrl;
        PowerDevice powerDevc;
        int powerChan;
        boolean inited = false;

        TrimHeater(TrimHeaterControl tempCtrl, PowerDevice powerDevc, int powerChan)
        {
            this.tempCtrl = tempCtrl;
            this.powerDevc = powerDevc;
            this.powerChan = powerChan;
        }

    }

    /**
     *  Constants.
     */
    private static final String THERMAL_LIMITS = "ThermalLimits";

    private static final int
        UPDATE_STATE_INTVL = 1000;

    private static final List<Set<Integer>> validTrimHeaterSets = new ArrayList<>();
    static {
        Set<Integer> trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
        trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYE);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_C);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYE);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
        trimSet = new LinkedHashSet<>();
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYE);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_PYC);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYC);
        trimSet.add(ThermalState.TRIM_HEATER_COLD_MYE);
        trimSet.add(ThermalState.TRIM_HEATER_CRYO);
        validTrimHeaterSets.add(trimSet);
    }

    /**
     *  Data fields.
     */
    @LookupName
    private String name;

    @LookupField(strategy=LookupField.Strategy.TOP)
    private Subsystem subsys;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService propertiesService;
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, TrimHeaterControl> tempControlMap = new HashMap<>();
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private HeaterPsDevice heaterPs;
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private Map<String, Channel> allChannels = new HashMap<>();

    @ConfigurationParameter(category=THERMAL_LIMITS, isFinal=true)
    private volatile double coldTempLimit = 40.0;

    private String[] coldTempNames;       // Names of the cold plate temperature channels

    private static final Logger LOG = Logger.getLogger(ThermalMain.class.getName());
    private Set<Integer> trimHeaterSet;
    private final TrimHeater[] trimHeaters = new TrimHeater[ThermalState.NUM_TRIM_HEATERS];
    private final List<Channel> coldTemps = new ArrayList<>();
    private final ThermalState thermalState = new ThermalState();
    private final Map<String, Boolean> activeAlertMap = new HashMap<>();
    private int highTempCount = 0;
    private ColdState refrigColdState = ColdState.UNKNOWN;
    private String refrigGroup = null;
    private MonitorTaskControl monitorControl;
    private boolean inited = false;


    /**
     *  Constructor
     */
        public ThermalMain() {
        super("thermal", AgentInfo.AgentType.WORKER);
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");
    }


    /**
     *  Build phase
     */
    @Override
    public void build()
    {
        // Create the monitor task control object and node
        monitorControl = MonitorTaskControl.createNode(this, RefrigConstants.MONITOR_CONTROL);

        //Create and schedule an AgentPeriodicTask to update the refrigeration state
        AgentPeriodicTask pt;
        pt = new AgentPeriodicTask("thermal-state", () -> updateThermalState()).withPeriod(Duration.ofMillis(UPDATE_STATE_INTVL));
        periodicTaskService.scheduleAgentPeriodicTask(pt);
    }


    /**
     *  Initializes the thermal subsystem.
     */
    @Override
    public void postInit()
    {
        // Set a property to define that this Agent is a thermal subsystem.
        propertiesService.setAgentProperty(RefrigAgentProperties.THERMAL_TYPE, ThermalMain.class.getCanonicalName());

        // Add an agent present listener
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);

        // General initialization
        refrigGroup = RefrigUtils.getGroupName(subsys.getAgentInfo());

        // Initialize trim heater control
        Set<Integer> heaters = new HashSet<>();
        for (TrimHeaterControl ctrl : tempControlMap.values()) {
            int htr = ctrl.getPowerChannel();
            heaters.add(htr);
            trimHeaters[htr] = new TrimHeater(ctrl, ctrl.getPowerDevice(), htr);
        }
        for (Set heaterSet : validTrimHeaterSets) {
            if (heaterSet.equals(heaters)) {
                trimHeaterSet = heaterSet;
                break;
            }
        }
        if (trimHeaterSet == null) {
            ErrorUtils.reportConfigError(LOG, name, "trim heater loop set", "is not a valid one");
        }
        thermalState.setTrimHeaters(trimHeaterSet.toArray(thermalState.getTrimHeaters()));

        // Initialize cold temperature channel list
        if (coldTempNames == null) {
            LOG.severe("No cold-plate temperature monitoring channels specified");
        }
        else {
            for (String coldName : coldTempNames) {
                Channel chan = allChannels.get(coldName);
                if (chan == null) {
                    ErrorUtils.reportConfigError(LOG, name, "coldTempNames", "contains non-channel item");
                }
                coldTemps.add(chan);
            }
        }
        
        //Register the Alerts raised by this subsystem
        alertService.registerAlert(ThermalAlert.COLD_TEMP_HIGH.newAlert());
        alertService.registerAlert(ThermalAlert.COLD_COMP_OFF.newAlert());
    }


    /**
     *  Starts the subsystem.
     */
    @Override
    public void postStart()
    {
        publishState();
        inited = true;
        LOG.info("Thermal control system started");
    }


    /**
     *  Listens for the arrival of the companion refrigeration subsystem.
     *
     *  @param  agents  Array of agents present
     */
    @Override
    public void connected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE)) {
                String agentName = agent.getName();
                if (refrigGroup.equals(RefrigUtils.getGroupName(agent))) {
                    subsys.getMessagingAccess().addStatusMessageListener(this, (msg) -> msg.getOriginAgentInfo().getName().equals(agentName));
                    break;
                }
            }
        }
    }


    /**
     *  Listens for the departure of the companion refrigeration subsystem.
     *
     *  @param  agents  Agents going absent
     */
    @Override
    public void disconnected(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (agent.hasAgentProperty(RefrigAgentProperties.COMPRESSOR_TYPE) && refrigGroup.equals(RefrigUtils.getGroupName(agent))) {
                subsys.getMessagingAccess().removeStatusMessageListener(this);
                //refrigColdState = ColdState.UNKNOWN;
            }
        }
    }


    /**
     *  Handles refrigeration status messages.
     *
     *  @param  msg  The status message
     */
    @Override
    public void onStatusMessage(StatusMessage msg) {
        refrigColdState = msg.getState().getState(ColdState.class);
    }


    /**
     *  Gets the state of the thermal control system.
     *
     *  This is intended to be called by GUIs during initialization
     *
     *  @return  The thermal control state
     */
    @Command(type=CommandType.QUERY, description="Get the thermal control state", level=0)
    public ThermalState getSystemState()
    {
        return thermalState;
    }    


    /**
     *  Sets a trim heater operation state.
     *
     *  @param  htr    The heater number
     *  @param  state  The heater operation state to set: OFF, POWER or TEMP.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater operation state")
    public void setTrimHeaterState(int htr, TrimHeaterOpState state) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                TrimHeaterOpState oldState = thermalState.getTrimHeaterOpState(htr);
                if (state != oldState & thermalState.getTrimHeaterState(htr) != TrimHeaterState.DISABLD) {
                    thermalState.setTrimHeaterOpState(htr, state);
                    heater.tempCtrl.setOpState(state);
                    heater.inited = true;
                }
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets a trim heater power value.
     *
     *  @param  htr    The heater number
     *  @param  power  The load power set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a trim heater power set point")
    public void setTrimHeaterPower(int htr, double power) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                thermalState.setTrimHeaterPower(htr, power);
                heater.tempCtrl.setPower(power);
                heater.inited = true;
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Sets a plate temperature value.
     *
     *  @param  htr   The heater number
     *  @param  temp  The plate temperature set point.
     *  @throws  RefrigException
     */
    @Command(type=CommandType.ACTION, description="Set a plate temperature set point")
    public void setPlateTemperature(int htr, double temp) throws RefrigException
    {
        synchronized(thermalState) {
            try {
                TrimHeater heater = getTrimHeater(htr);
                thermalState.setPlateTemperature(htr, temp);
                heater.tempCtrl.setTemp(temp);
                heater.inited = true;
            }
            finally {
                publishState();
            }
        }
    }


    /**
     *  Turns on/off the bulk power supply.
     *
     *  @param  on  Whether to turn the power on
     */
    @Command(type=Command.CommandType.ACTION, description="Turn on/off the bulk power supply")
    public void bulkPowerOn(boolean on)
    {
        heaterPs.setBulkPowerOn(on);
        publishState();
    }


    /**
     *  Publishes the state of the thermal module.
     *
     *  This is intended to be called whenever any element of the state is
     *  changed.
     */
    private void publishState()
    {
        thermalState.setTickMillis(monitorControl.getPublishPeriod());
        subsys.publishSubsystemDataOnStatusBus(new KeyValueData(ThermalState.KEY, thermalState));
    }    


    /**
     *  Updates the thermal state periodically.
     */
    private void updateThermalState()
    {
        if (!inited) return;  // If postStart stage hasn't occurred yet
        boolean coldHigh = false;
        for (Channel chan : coldTemps) {
            coldHigh |= (chan.getValue() > coldTempLimit);
        }
        if (coldHigh) {
            coldHigh &= (++highTempCount >= 3);
        }
        else {
            highTempCount = 0;
        }
        synchronized(thermalState) {
            boolean changed = monitorControl.hasPeriodChanged();
            
            for (int htr : trimHeaterSet) {
                TrimHeater heater = trimHeaters[htr];
                TrimHeaterState oldState = thermalState.getTrimHeaterState(htr);
                TrimHeaterState state = coldHigh && htr != ThermalState.TRIM_HEATER_CRYO ? TrimHeaterState.DISABLD :
                                        !heater.powerDevc.isOnline() ? TrimHeaterState.OFFLINE :
                                        heater.tempCtrl.isOverTemp() ? TrimHeaterState.OVRTEMP :
                                        !heater.powerDevc.isEnabled(heater.powerChan) ? TrimHeaterState.OFF :
                                        heater.powerDevc.hasVoltError(heater.powerChan) ? TrimHeaterState.VOLTERR :
                                        heater.powerDevc.hasNoLoad(heater.powerChan) ? TrimHeaterState.NOLOAD :
                                        thermalState.getTrimHeaterOpState(htr) == TrimHeaterOpState.TEMP ? TrimHeaterState.TEMP
                                                                                                         : TrimHeaterState.POWER;
                if (state != oldState) {
                    thermalState.setTrimHeaterState(htr, state);
                    if (oldState == TrimHeaterState.OFFLINE) {
                        if (!heater.inited) {
                            if (heater.powerDevc.isEnabled(heater.powerChan)) {
                                double power = heater.powerDevc.getPower(heater.powerChan);
                                thermalState.setTrimHeaterPower(htr, power);
                                heater.tempCtrl.setPower(power);
                                thermalState.setTrimHeaterOpState(htr, TrimHeaterOpState.POWER);
                                heater.tempCtrl.setOpState(TrimHeaterOpState.POWER);
                            }
                            else {
                                thermalState.setTrimHeaterOpState(htr, TrimHeaterOpState.OFF);
                                heater.tempCtrl.setOpState(TrimHeaterOpState.OFF);
                            }
                            heater.inited = true;
                        }
                        else {
                            heater.tempCtrl.setPower(thermalState.getTrimHeaterPower(htr));
                            heater.tempCtrl.setTemp(thermalState.getPlateTemperature(htr));
                            heater.tempCtrl.setOpState(thermalState.getTrimHeaterOpState(htr));
                        }
                    }
                    if (state == TrimHeaterState.DISABLD) {
                        thermalState.setTrimHeaterOpState(htr, TrimHeaterOpState.OFF);
                        heater.tempCtrl.setOpState(TrimHeaterOpState.OFF);
                    }
                    changed = true;
                }
            }
            Boolean bulkOn = heaterPs.isBulkPowerOn();
            SwitchState bulkPowerState = bulkOn == null ? SwitchState.OFFLINE : bulkOn ? SwitchState.ON : SwitchState.OFF;
            if (bulkPowerState != thermalState.getBulkPowerState()) {
                thermalState.setBulkPowerState(bulkPowerState);
                changed = true;
            }
            if (changed) {
                publishState();
                if (coldHigh) {
                    raiseAlert(ThermalAlert.COLD_TEMP_HIGH, AlertState.ALARM, "At least one cold-plate RTD is above " + coldTempLimit + " C");
                }
                else {
                    lowerAlert(ThermalAlert.COLD_TEMP_HIGH, "No cold-plate RTD is above " + coldTempLimit + " C");
                }
                boolean coldCompsOff = refrigColdState == ColdState.BOTH_OFF || refrigColdState == ColdState.UNKNOWN;
                if (coldCompsOff) {
                    raiseAlert(ThermalAlert.COLD_COMP_OFF, AlertState.ALARM, "Both cold compressors are turned off");
                }
                else {
                    lowerAlert(ThermalAlert.COLD_COMP_OFF, "At least one cold compressor is back on");
                }
            }
        }
    }


    /**
     *  Raises an alert.
     *
     *  @param  alert  The thermal alert to raise
     *  @param  state  The alert state (WARNING or ALARM)
     *  @param  cond   The alert condition
     */
    private void raiseAlert(ThermalAlert alert, AlertState state, String cond)
    {
        Boolean wasActive = activeAlertMap.put(alert.getId(), true);
        if (wasActive != Boolean.TRUE) {
            alertService.raiseAlert(alert.newAlert(), state, cond);
        }
    }


    /**
     *  Lowers an alert.
     *
     *  @param  alert  The thermal alert to lower
     *  @param  cond   The alert condition
     */
    private void lowerAlert(ThermalAlert alert, String cond)
    {
        Boolean wasActive = activeAlertMap.put(alert.getId(), false);
        if (wasActive == Boolean.TRUE) {
            alertService.raiseAlert(alert.newAlert(), AlertState.NOMINAL, cond);
        }
    }


    /**
     *  Checks a trim heater number for validity and returns its object.
     *
     *  @param  htr  The heater number
     *  @return  The TrimHeater object
     *  @throws  RefrigException if the number is invalid or object is null
     */
    private TrimHeater getTrimHeater(int htr) throws RefrigException
    {
        if (!trimHeaterSet.contains(htr)) {
            throw new RefrigException("Invalid trim heater number: " + htr);
        }
        return trimHeaters[htr];
    }

}
