package org.lsst.ccs.subsystem.refrig;

import java.util.HashSet;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.command.annotations.Command.CommandType;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.yaskawa.A1000;
import org.lsst.ccs.framework.ClearAlertHandler.ClearAlertCode;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.common.ErrorUtils;
import org.lsst.ccs.subsystem.refrig.constants.VfdState;
import org.lsst.ccs.subsystem.refrig.data.RefrigException;

/**
 *  Handles a Yaskawa A1000 series variable frequency motor drive.
 *
 *  @author Owen Saxton
 */
public class A1000Device extends Device {

    /**
     *  Constants
     */
    public static final int
        CHAN_FREQUENCY = 0,
        CHAN_CURRENT = 1,
        CHAN_VOLTAGE = 2,
        CHAN_TEMPERATURE = 3,
        NUM_CHANS = 4;
    private static final double
        INPUT_VOLTAGE = 380.0,
        MAX_FREQ_PCT = 110.0;

    /**
     *  Data fields.
     */
    @ConfigurationParameter(category = "Device", isFinal = true)
    private volatile String  devcName;      // Device name of the serial line
    @ConfigurationParameter(category = "Device", isFinal = true)
    private volatile int     baudRate = 0;  // The serial line baud rate (0 = driver default)
    @ConfigurationParameter(category = "Device", isFinal = true)
    private volatile int     address = 0;   // RS-485 address of VFD controller (0 = driver default)

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    private static final Logger LOG = Logger.getLogger(A1000Device.class.getName());
    private final A1000 a1000 = new A1000();  // Associated A1000 object
    private String compName;  // Associated conpressor name
    private boolean initError = false;
    private int faultCode = 0;
    private Set<Integer> alarmCodes = new HashSet<>();


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        super.configure(mon);
        if (devcName == null) {
            ErrorUtils.reportConfigError(LOG, getPath(), "devcName", "is missing");
        }
        fullName = getPath() + " (A1000 VFD at " + devcName + ":" + address + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            a1000.open(devcName, address, baudRate);
            try {
                if ((a1000.getDriveStatus() & A1000.STATUS_RUNNING) == 0) {  // If not running
                    initParams();
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error initializing {0} parameters: {1}", new Object[]{fullName, e});
            }
            setOnline(true);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        }
        catch (DriverException e) {
            if (!initError) {  // Avoid reporting consecutive errors
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                a1000.close();
            }
            catch (DriverException ce) {
                // Will happen if open was unsuccessful
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            a1000.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Sets the compressor name.
     *
     *  @param  name  The compressor name
     */
    public void setCompressorName(String name)
    {
        compName = name;
    }


    /**
     *  Sets the frequency.
     * 
     *  @param  freq  The value to set
     *  @throws RefrigException 
     */
    public void setFrequency(double freq) throws RefrigException
    {
        try {
            a1000.setFrequency(freq);
        }
        catch (DriverException e) {
            setOnline(false);
            throw new RefrigException("Error setting frequency for " + getPath() + ": " + e);
        }
    }


    /**
     *  Gets the frequency.
     * 
     *  @return  The set frequency, or NaN if device is offline
     */
    public double getFrequency()
    {
        double value = Double.NaN;
        if (isOnline()) {
            try {
                value = a1000.getFrequencyApp();
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error getting frequency for {0}: {1}", new Object[]{getPath(), e});
                setOnline(false);
            }
        }
        return value;
    }


    /**
     *  Resets the current fault
     * 
     *  @throws RefrigException 
     */
    public void resetFault() throws RefrigException
    {
        try {
            a1000.resetFault();
        }
        catch (DriverException e) {
            setOnline(false);
            throw new RefrigException("Error resetting fault for " + getPath() + ": " + e);
        }
    }


    /**
     *  Checks a channel's parameters for validity.
     *
     *  @param  name     The channel name
     *  @param  hwChan   The hardware channel number
     *  @param  type     The channel type string
     *  @param  subtype  The channel subtype string
     *  @return  A two-element array containing the encoded type [0] and subtype [1] values.
     *  @throws  Exception if any errors found in the parameters.
     */
    @Override
    protected int[] checkChannel(String name, int hwChan, String type, String subtype) throws Exception
    {
        if (hwChan < 0 || hwChan >= NUM_CHANS) {
            ErrorUtils.reportChannelError(LOG, getPath() + "/" + name, "hw channel number", hwChan);
        }

        return new int[]{0, 0};
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);
        String quantity = null;
        if (isOnline()) {
            try {
                switch (hwChan) {
                case CHAN_FREQUENCY:
                    quantity = "frequency";
                    value = a1000.readFrequency();
                    break;
                case CHAN_CURRENT:
                    quantity = "current";
                    value = a1000.readCurrent();
                    break;
                case CHAN_VOLTAGE:
                    quantity = "voltage";
                    value = a1000.readVoltage();
                    break;
                case CHAN_TEMPERATURE:
                    quantity = "temperature";
                    value = a1000.readTemperature();
                    break;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0} {1}: {2}", new Object[]{getPath(), quantity, e});
                setOnline(false);
            }
        }

        return value;
    }


    /**
     *  Gets VFD state and checks for alarm and fault conditions.
     *
     *  @return The VFD state
     */
    public VfdState getState()
    {
        VfdState state = VfdState.OFFLINE;
        if (isOnline()) {
            try {
                int driveStatus = a1000.getDriveStatus();
                state = (driveStatus & A1000.STATUS_REVERSE) != 0 ? VfdState.REVERSE :
                        (driveStatus & A1000.STATUS_ZERO_SPEED) != 0 ? VfdState.ZEROFREQ :
                        (driveStatus & A1000.STATUS_RUNNING) != 0 ? VfdState.RUNNING : VfdState.STOPPED;
                int code;
                if ((driveStatus & A1000.STATUS_FAULT) != 0) {
                    code = a1000.getCurrFaultCode();
                    state = VfdState.FAULT;
                }
                else {
                    code = 0;
                }
                if (code != faultCode) {
                    if (faultCode != 0) {
                        lowerFault(faultCode);
                    }
                    if (code != 0) {
                        raiseFault(code);
                    }
                    faultCode = code;
                }
                Set<Integer> codes;
                if ((driveStatus & A1000.STATUS_ALARM) != 0) {
                    codes = a1000.getAlarmCodes();
                    state = VfdState.ALARM;
                }
                else {
                    codes = new HashSet<>();
                }
                if (!codes.equals(alarmCodes)) {
                    for (int alarmCode : alarmCodes) {
                        if (!codes.contains(alarmCode)) {
                            lowerAlarm(alarmCode);
                        }
                    }
                    for (int alarmCode : codes) {
                        if (!alarmCodes.contains(alarmCode)) {
                            raiseAlarm(alarmCode);
                        }
                    }
                    alarmCodes = codes;
                }
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading {0}: {1}", new Object[]{getPath(), e});
                setOnline(false);
            }
        }
        return state;
    }


    /**
     *  Raises an alarm for a fault condition
     * 
     *  @param code 
     */
    private void raiseFault(int code) {
        Alert alert = new Alert(compName + "VfdFault" + code, compName + "VFD fault condition");
        alertService.raiseAlert(alert, AlertState.ALARM, "Fault raised: " + a1000.getFaultDesc(code));
    }


    /**
     *  Lowers a fault condition alarm
     * 
     *  @param code 
     */
    private void lowerFault(int code) {
        Alert alert = new Alert(compName + "VfdFault" + code, compName + "VFD fault condition");
        alertService.raiseAlert(alert, AlertState.NOMINAL, "Fault cleared: " + a1000.getFaultDesc(code));
    }


    /**
     *  Raises a warning for an alarm condition
     * 
     *  @param code 
     */
    private void raiseAlarm(int code) {
        Alert alert = new Alert(compName + "VfdAlarm" + code, compName + "VFD alarm condition");
        alertService.raiseAlert(alert, AlertState.WARNING, "Alarm raised: " + a1000.getAlarmDesc(code));
    }


    /**
     *  Lowers an alarm condition warning
     * 
     *  @param code 
     */
    private void lowerAlarm(int code) {
        Alert alert = new Alert(compName + "VfdAlarm" + code, compName + "VFD alarm condition");
        alertService.raiseAlert(alert, AlertState.NOMINAL, "Alarm cleared: " + a1000.getAlarmDesc(code));
    }


    /**
     *  Enables an alert to be cleared.
     * 
     *  @param  alert  The alert
     *  @return  Action code, or null if not a VFD alert
     */
    public ClearAlertCode canClearAlert(Alert alert)
    {
        if (!alert.getAlertId().contains("Vfd")) return null;
        Boolean active = null;
        String id = alert.getAlertId();
        int index = id.indexOf("Fault");
        if (index >= 0) {
            active = Integer.decode(id.substring(index + 5)) == faultCode;
        }
        else {
            index = id.indexOf("Alarm");
            if (index >= 0) {
                active = alarmCodes.contains(Integer.decode(id.substring(index + 5)));
            }
        }
        return active == null ? ClearAlertCode.UNKNOWN_ALERT : active ? ClearAlertCode.DONT_CLEAR_ALERT : ClearAlertCode.CLEAR_ALERT;
    }


    @Command(type=CommandType.ACTION, description="Initialize parameters")
    public void initParams() throws DriverException
    {
        a1000.setAccessLevel(A1000.AccessLevel.FULL);
        a1000.setRunCmndFwdRev(true);
        a1000.setRunProg(A1000.RunProg.RUN_ONLY);
        a1000.setStopMethod(A1000.StopMethod.COAST);
        a1000.setControlMode(A1000.ControlMode.VF);
        a1000.setErrorAction(A1000.ErrorAction.ALARM_ONLY);
        a1000.setFaultDetected(false);
        a1000.setInputVoltage(INPUT_VOLTAGE);
        a1000.setVfSelection(A1000.VfSelection.HZ50);
        a1000.setVoltageUnits(A1000.VoltageUnits.TENTHS);
        a1000.setMaxFrequency(MAX_FREQ_PCT);
        a1000.setEnterRequired(false);
        a1000.setReversePermitted(false);
        a1000.setRunCmndSrc(A1000.RunCmndSrc.MODBUS);
        a1000.setFreqRefSrc(A1000.FreqRefSrc.MODBUS);
        a1000.enterParameters();
        a1000.stop();        // Might be needed if switching from local mode
        a1000.runForward();  // Always switch VFD on
        a1000.runForward();  // Needs to be done twice if switching from local mode
    }

    /* Commands to aid in debugging */
    
    private final int MAX_READ = 16;

    @Command(type=CommandType.QUERY, level=0, description="Read a register")
    public String readRegister(@Argument(description="Register to read") int number) throws DriverException
    {
        return String.format("%04x: %04x", number, a1000.readRegister((short)number));
    }

    @Command(type=CommandType.QUERY, level=0, description="Read a set of registers")
    public String readRegisters(@Argument(description="First register to read") int number,
                                @Argument(description="Number of registers to read") int count)
    {
        StringBuilder text = new StringBuilder();
        int lineSize = 8;
        int nRead = 0;
        try {
            while (nRead < count) {
                short[] reply = a1000.readRegisters((short)number, (short)Math.min(count - nRead, MAX_READ));
                for (int j = 0; j < reply.length; j++, nRead++, number++) {
                    if ((nRead % lineSize) == 0) {
                        if (nRead != 0) {
                            text.append("\n");
                        }
                        text.append(String.format("%04x:", number));
                    }
                    text.append(String.format(" %04x", reply[j]));
                }
            }
        }
        catch (DriverException e) {
            if (text.length() > 0) {
                text.append("\n");
            }
            text.append("***** Got read error: ").append(e.getMessage());
        }
        return text.toString();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show model name")
    public String showModelName() throws DriverException
    {
        return a1000.getModelName();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show node address")
    public int showNodeAddress() throws DriverException
    {
        return a1000.getNodeAddress();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show baud rate")
    public String showBaudRate() throws DriverException
    {
        return a1000.getBaudRate().toString().substring(1);
    }

    @Command(type=CommandType.QUERY, level=0, description="Show parity")
    public A1000.Parity showParity() throws DriverException
    {
        return a1000.getParity();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show access level")
    public A1000.AccessLevel showAccessLevel() throws DriverException
    {
        return a1000.getAccessLevel();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show control mode")
    public A1000.ControlMode showControlMode() throws DriverException
    {
        return a1000.getControlMode();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show whether enter required")
    public boolean isEnterRequired() throws DriverException
    {
        return a1000.isEnterRequired();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show communications error action")
    public A1000.ErrorAction showErrorAction() throws DriverException
    {
        return a1000.getErrorAction();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show whether fault detected")
    public boolean isFaultDetected() throws DriverException
    {
        return a1000.isFaultDetected();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show frequency reference source")
    public A1000.FreqRefSrc showFreqRefSrc() throws DriverException
    {
        return a1000.getFreqRefSrc();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show input voltage")
    public double showInputVoltage() throws DriverException
    {
        return a1000.getInputVoltage();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show whether reverse permitted")
    public boolean isReversePermitted() throws DriverException
    {
        return a1000.isReversePermitted();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show whether run command style is forward/reverse")
    public boolean isRunCmndFwdRev() throws DriverException
    {
        return a1000.isRunCmndFwdRev();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show run command source")
    public A1000.RunCmndSrc showRunCmndSrc() throws DriverException
    {
        return a1000.getRunCmndSrc();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show run/program interaction")
    public A1000.RunProg showRunProg() throws DriverException
    {
        return a1000.getRunProg();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the stop method")
    public A1000.StopMethod showStopMethod() throws DriverException
    {
        return a1000.getStopMethod();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show V/f profile selection")
    public A1000.VfSelection showVfSelection() throws DriverException
    {
        return a1000.getVfSelection();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show voltge units")
    public A1000.VoltageUnits showVoltageUnits() throws DriverException
    {
        return a1000.getVoltageUnits();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show maximum frequency percentage")
    public double showMaxFrequency() throws DriverException
    {
        return a1000.getMaxFrequency();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show minimum frequency percentage")
    public double showMinFrequency() throws DriverException
    {
        return a1000.getMinFrequency();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the reference frequency")
    public double showFrequency() throws DriverException
    {
        return a1000.getFrequency();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the applied reference frequency")
    public double showFrequencyApp() throws DriverException
    {
        return a1000.getFrequencyApp();
    }

    @Command(type=CommandType.QUERY, level=0, description="Read the frequency")
    public double readFrequency() throws DriverException
    {
        return a1000.readFrequency();
    }

    @Command(type=CommandType.QUERY, level=0, description="Read the current")
    public double readCurrent() throws DriverException
    {
        return a1000.readCurrent();
    }

    @Command(type=CommandType.QUERY, level=0, description="Read the voltage")
    public double readVoltage() throws DriverException
    {
        return a1000.readVoltage();
    }

    @Command(type=CommandType.QUERY, level=0, description="Read the temperature")
    public double readTemperature() throws DriverException
    {
        return a1000.readTemperature();
    }

    @Command(type=CommandType.QUERY, level=0, description="Read the bus voltage")
    public double readBusVoltage() throws DriverException
    {
        return a1000.readBusVoltage();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the drive status")
    public String showDriveStatus() throws DriverException
    {
        return String.format("0x%02x", a1000.getDriveStatus());
    }

    @Command(type=CommandType.QUERY, description = "Show the alarm code")
    public int showAlarmCode() throws DriverException
    {
        return a1000.getAlarmCode();
    }

    @Command(type=CommandType.QUERY, description = "Show the alarm codes")
    public Set<Integer> showAlarmCodes() throws DriverException
    {
        return a1000.getAlarmCodes();
    }

    @Command(type=CommandType.QUERY, description = "Show the alarm description")
    public String showAlarmDesc() throws DriverException
    {
        return a1000.getAlarmDesc();
    }

    @Command(type=CommandType.QUERY, description = "Show an alarm description")
    public String showAlarmDesc(@Argument(description="Alarm code") int code) throws DriverException
    {
        return a1000.getAlarmDesc(code);
    }

    @Command(type=CommandType.QUERY, description = "Show the current fault code")
    public int showCurrFaultCode() throws DriverException
    {
        return a1000.getCurrFaultCode();
    }

    @Command(type=CommandType.QUERY, description = "Show the previous fault code")
    public int showPrevFaultCode() throws DriverException
    {
        return a1000.getPrevFaultCode();
    }

    @Command(type=CommandType.QUERY, description = "Show the current fault description")
    public String showCurrFaultDesc() throws DriverException
    {
        return a1000.getCurrFaultDesc();
    }

    @Command(type=CommandType.QUERY, description = "Show the previous fault description")
    public String showPrevFaultDesc() throws DriverException
    {
        return a1000.getPrevFaultDesc();
    }

    @Command(type=CommandType.QUERY, description = "Show a fault description")
    public String showFaultDesc(@Argument(description="Fault code") int code) throws DriverException
    {
        return a1000.getFaultDesc(code);
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the cumulative operation time")
    public int showOperationTime() throws DriverException
    {
        return a1000.getOperationTime();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show whether cumulative operation time is only time active")
    public boolean isOpTimeActive() throws DriverException
    {
        return a1000.isOpTimeActive();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the cumulative fan operation time")
    public int showFanOperationTime() throws DriverException
    {
        return a1000.getFanOperationTime();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the run command count")
    public int showRunCount() throws DriverException
    {
        return a1000.getRunCount();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the cumulative output kilowatt hours")
    public double showOutputKwh() throws DriverException
    {
        return a1000.getOutputKwh();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the fan lifetime fraction used")
    public double showFanLifetimeUsed() throws DriverException
    {
        return a1000.getFanLifetimeUsed();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the capacitor lifetime fraction used")
    public double showCapLifetimeUsed() throws DriverException
    {
        return a1000.getCapLifetimeUsed();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the soft charge bypass relay lifetime fraction used")
    public double showScbrLifetimeUsed() throws DriverException
    {
        return a1000.getScbrLifetimeUsed();
    }

    @Command(type=CommandType.QUERY, level=0, description="Show the IGBT lifetime fraction used")
    public double showIgbtLifetimeUsed() throws DriverException
    {
        return a1000.getIgbtLifetimeUsed();
    }

    @Command(type=CommandType.QUERY, level=0, description="Flash the display LEDs")
    public void flashLeds() throws DriverException
    {
        a1000.flashLeds();
    }

    @Command(type=CommandType.ACTION, level=50, description="Write a set of registers")
    public void writeRegisters(@Argument(description="First register to write") int number,
                               @Argument(description="Values to write") int... values) throws DriverException
    {
        short count = (short)values.length;
        if (count == 0) {
            throw new DriverException("No values supplied");
        }
        else {
            short[] sValues = new short[count];
            for (int j = 0; j < count; j++) {
                sValues[j] = (short)values[j];
            }
            a1000.writeRegisters((short)number, sValues);
        }
    }

    @Command(type=CommandType.ACTION, level=50, description="Enter parameters")
    public void enterParameters() throws DriverException
    {
        a1000.enterParameters();
    }

    @Command(type=CommandType.ACTION, level=50, description="Save parameters")
    public void saveParameters() throws DriverException
    {
        a1000.saveParameters();
    }

    @Command(type=CommandType.ACTION, level=50, description="Set node address")
    public void setNodeAddress(@Argument(description="Node address") int address) throws DriverException
    {
        a1000.setNodeAddress(address);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set baud rate")
    public void setBaudRate(@Argument(description="Baud rate enum") A1000.BaudRate baudRate) throws DriverException
    {
        a1000.setBaudRate(baudRate);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set parity")
    public void setParity(@Argument(description="Parity enum") A1000.Parity parity) throws DriverException
    {
        a1000.setParity(parity);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set access level")
    public void setAccessLevel(@Argument(description="Access level enum") A1000.AccessLevel level) throws DriverException
    {
        a1000.setAccessLevel(level);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set control mode")
    public void setControlMode(@Argument(description="Control mode enum") A1000.ControlMode mode) throws DriverException
    {
        a1000.setControlMode(mode);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set whether enter required")
    public void setEnterRequired(@Argument(description="Whether enter required") boolean required) throws DriverException
    {
        a1000.setEnterRequired(required);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set communications error action")
    public void setErrorAction(@Argument(description="Error action enum") A1000.ErrorAction action) throws DriverException
    {
        a1000.setErrorAction(action);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set whether fault detected")
    public void setFaultDetected(@Argument(description="Whether fault detected") boolean detect) throws DriverException
    {
        a1000.setFaultDetected(detect);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set frequency reference source")
    public void setFreqRefSrc(@Argument(description="Freq refc source enum") A1000.FreqRefSrc source) throws DriverException
    {
        a1000.setFreqRefSrc(source);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set the input voltage")
    public void setInputVoltage(@Argument(description="The voltage") double voltage) throws DriverException
    {
        a1000.setInputVoltage(voltage);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set whether reverse permitted")
    public void setReversePermitted(@Argument(description="Whether reverse permitted") boolean permit) throws DriverException
    {
        a1000.setReversePermitted(permit);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set whether run command style is forward/reverse")
    public void setRunCmndFwdRev(@Argument(description="Whether forward/reverse") boolean fwdRev) throws DriverException
    {
        a1000.setRunCmndFwdRev(fwdRev);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set run command source")
    public void setRunCmndSrc(@Argument(description="Run cmnd source enum") A1000.RunCmndSrc source) throws DriverException
    {
        a1000.setRunCmndSrc(source);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set run/program interaction")
    public void setRunProg(@Argument(description="Run/program enum") A1000.RunProg runProg) throws DriverException
    {
        a1000.setRunProg(runProg);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set the stop method")
    public void setStopMethod(@Argument(description="Stop method enum") A1000.StopMethod method) throws DriverException
    {
        a1000.setStopMethod(method);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set V/f profile selection")
    public void setVfSelection(@Argument(description="V/f selection enum") A1000.VfSelection select) throws DriverException
    {
        a1000.setVfSelection(select);
    }

    //@Command(type=CommandType.ACTION, level=50, description="Set the reference frequency")
    //public void setFrequency(@Argument(description="The frequency") double freq) throws DriverException
    //{
    //    a1000.setFrequency(freq);
    //}

    @Command(type=CommandType.ACTION, level=50, description="Set voltage units")
    public void setVoltageUnits(@Argument(description="Voltage units enum") A1000.VoltageUnits units) throws DriverException
    {
        a1000.setVoltageUnits(units);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set maximum frequency percentage")
    public void setMaxFrequency(@Argument(description="The frequency percentage") double freqPct) throws DriverException
    {
        a1000.setMaxFrequency(freqPct);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set minimum frequency percentage")
    public void setMinFrequency(@Argument(description="The frequency percentage") double freqPct) throws DriverException
    {
        a1000.setMinFrequency(freqPct);
    }

    @Command(type=CommandType.ACTION, level=50, description="Run the motor forwards")
    public void runForward() throws DriverException
    {
        a1000.runForward();
    }

    @Command(type=CommandType.ACTION, level=50, description="Run the motor in reverse")
    public void runReverse() throws DriverException
    {
        a1000.runReverse();
    }

    @Command(type=CommandType.ACTION, level=50, description="Stop the motor")
    public void stop() throws DriverException
    {
        a1000.stop();
    }

    @Command(type=CommandType.ACTION, level=50, description="Set the cumulative operation time")
    public void setOperationTime(@Argument(description="The time (10 hrs)") int time) throws DriverException
    {
        a1000.setOperationTime(time);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set whether cumulative operation time is only time active")
    public void setOpTimeActive(@Argument(description="Whether to use time active") boolean isActvTime) throws DriverException
    {
        a1000.setOpTimeActive(isActvTime);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set the cumulative fan operation time")
    public void setFanOperationTime(@Argument(description="The time") int time) throws DriverException
    {
        a1000.setFanOperationTime(time);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set the capacitor lifetime fraction used")
    public void setCapLifetimeUsed(@Argument(description="The fraction used") double fract) throws DriverException
    {
        a1000.setCapLifetimeUsed(fract);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set the soft charge bypass relay lifetime fraction used")
    public void setScbrLifetimeUsed(@Argument(description="The fraction used") double fract) throws DriverException
    {
        a1000.setScbrLifetimeUsed(fract);
    }

    @Command(type=CommandType.ACTION, level=50, description="Set the IGBT lifetime fraction used")
    public void setIgbtLifetimeUsed(@Argument(description="The fraction used") double fract) throws DriverException
    {
        a1000.setIgbtLifetimeUsed(fract);
    }

}
