package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.subsystem.refrig.constants.VfdState;

/**
 *  Handles a Simulated Yaskawa A1000 series variable frequency motor drive.
 *
 *  @author Owen Saxton
 */
public class SimA1000Device extends A1000Device {

    /**
     *  Constants
     */
    private static final Logger LOG = Logger.getLogger(SimA1000Device.class.getName());
    private double frequency;


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        fullName = getPath() + " (simulated A1000)";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        setOnline(true);
        LOG.log(Level.INFO, "Connected to {0}", fullName);
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
    }


    /**
     *  Sets the frequency.
     * 
     *  @param  freq  The value to set
     */
    @Override
    public void setFrequency(double freq)
    {
        frequency = freq;
    }


    /**
     *  Gets the frequency.
     * 
     *  @return  The set frequency, or NaN if device is offline
     */
    @Override
    public double getFrequency()
    {
        return frequency;
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = Double.NaN;
        if (online) {
            switch (hwChan) {
            case CHAN_FREQUENCY:
                value = frequency;
                break;
            case CHAN_CURRENT:
                value = 1.0;
                break;
            case CHAN_VOLTAGE:
                value = 380.0;
                break;
            case CHAN_TEMPERATURE:
                value = 29.0;
                break;
            }
        }
        return value;
    }


    /**
     *  Gets VFD state (and checks for alarm and fault conditions).
     *
     *  @return The VFD state
     */
    @Override
    public VfdState getState()
    {
        return VfdState.RUNNING;
    }

}
