package org.lsst.ccs.subsystem.refrig;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.drivers.commons.DriverException;
import org.lsst.ccs.drivers.mks.GP390;
import org.lsst.ccs.monitor.Device;
import org.lsst.ccs.subsystem.common.ErrorUtils;

/**
 *  Handles an MKS GP390 vacuum gauge.
 *
 *  @author Owen Saxton
 */
public class GP390Device extends Device {

    /**
     *  Data fields.
     */
    private final static int BAUD_RATE = 19200;

    private String  serial;   // Serial number of FTDI adapter
    private int     address;  // RS-485 address of gauge

    private static final Logger LOG = Logger.getLogger(GP390Device.class.getName());
    private final GP390 gp = new GP390();  // Associated GP390 object
    private boolean initError = false;


    /**
     *  Constructor.
     *
     *  @param  serialNo  The serial number of the FTDI adapter
     *  @param  addr      The RS-485 node address
     */
    public GP390Device(String serialNo, int addr)
    {
        serial = serialNo;
        address = addr;
    }


    public GP390Device() {
    }


    /**
     *  Performs configuration.
     */
    @Override
    protected void initDevice()
    {
        super.configure(mon);
        if (serial == null) {
            ErrorUtils.reportConfigError(LOG, getName(), "serial", "is missing");
        }
        fullName = "GP390 vacuum gauge (" + serial + ":" + address + ")";
    }


    /**
     *  Performs full initialization.
     */
    @Override
    protected void initialize()
    {
        try {
            gp.open(GP390.ConnType.FTDI, serial, BAUD_RATE, address);
            gp.setLock(false);
            gp.setPressureUnit(GP390.UNIT_TORR);
            gp.setLock(true);
            setOnline(true);
            initSensors();
            LOG.log(Level.INFO, "Connected to {0}", fullName);
            initError = false;
        }
        catch (DriverException e) {
            if (!initError) {
                LOG.log(Level.SEVERE, "Error connecting to {0}: {1}", new Object[]{fullName, e});
                initError = true;
            }
            try {
                gp.close();
            }
            catch (DriverException ce) {
                // Can happen if open failed
            }
        }
    }


    /**
     *  Closes the connection.
     */
    @Override
    protected void close()
    {
        try {
            gp.close();
        }
        catch (DriverException e) {
            LOG.log(Level.SEVERE, "Error disconnecting from {0}: {1}", new Object[]{fullName, e});
        }
    }


    /**
     *  Reads a channel.
     *
     *  @param  hwChan   The hardware channel number.
     *  @param  type     The encoded channel type returned by checkChannel.
     *  @return  The value read from the channel
     */
    @Override
    protected double readChannel(int hwChan, int type)
    {
        double value = super.readChannel(hwChan, type);

        if (online) {
            try {
                value = gp.readPressure();
            }
            catch (DriverException e) {
                LOG.log(Level.SEVERE, "Error reading pressure: {0}", e);
                setOnline(false);
            }
        }

        return value;
    }

}
